<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: class2xsdVisitor.class.php 93 2006-04-16 15:23:57Z sakamoto $
 */

/**
 * クラス→XSDデータへの変換用ビジタークラス
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_class2xsdVisitor implements AutoSOAP_ReflectionVisitor {
    private $_root;

    private $_xml;
    private $_schema;
    private $_array_fctry;

    private $_ArrayMap;
    private $_classMap;
    private $_RestrictionMap;
    private $_restrictionCount;

    static private $_xsd_var_types = array('anyType',
        'string',
        'boolean',
        'float',
        'double',
        'decimal',
        'integer',
        'nonPositiveInteger',
        'negativeInteger',
        'long',
        'int',
        'short',
        'byte',
        'nonNegativeInteger',
        'unsignedLong',
        'unsignedInt',
        'unsignedShort',
        'unsignedByte',
        'positiveInteger',
        'date',
        'time',
        'anyURI',
        'base64Binary',
        'hexBinary',
        'dateTime',
        'duration',
        'ENTITIES',
        'ENTITY',
        'gDay',
        'gMonth',
        'gMonthDay',
        'gYear',
        'gYearMonth',
        'ID',
        'IDREF',
        'IDREFS',
        'language',
        'Name',
        'NCName',
        'NMTOKEN',
        'NMTOKENS',
        'normalizedString',
        'NOTATION',
        'QName',
        'token',
        );
    /**
     * 初期化
     */
    public function init($root, $array_fctry = null)
    {
        $this -> _root = $root;
        $this -> _Target_uri = AutoSOAP_XMLogic_XMLFactory :: searchAttrigute($this -> _root, 'targetNamespace') -> getName();
        $this -> _schema = AutoSOAP_XMLogic_SchemaFactory :: getInstance();
        $this -> _array_fctry = (is_null($array_fctry)) ? $this -> _schema : $array_fctry ;

        /**
         * その他の初期化
         */
        $this -> _classMap = array();
        $this -> _ArrayMap = array();
        $this -> _restrictionCount = 0;
    } 

    /**
     * プロパティを処理する
     */
    public function visitProperty(AutoSOAP_ReflectionAccept $property)
    { 
        // アクセス制御がPublicで静的なものでなければ処理する。
        if (!$property -> isPublic() || $property -> isStatic()) {
            return ;
        } 

        /**
         * アノテーション
         */
        $annotation = AutoSOAP_Anno :: parseReflector($property); 
        // プロパティが静的かチェック
        if ($annotation -> execute(new AutoSOAP_Anno_StaticCmd())) {
            return ;
        } 
        // アクセス制御を取得
        $anno_access = $annotation -> execute(new AutoSOAP_Anno_AccessCmd());
        if ('public' !== $anno_access && !is_null($anno_access)) {
            return ;
        } 

        $elemType = AutoSOAP_ElementType :: createByProperty($annotation);
        $options = array();

        return $this -> getDataTypeElement($property -> getName(), $elemType, $options);
    } 

    /**
     * クラスを処理する
     */
    public function visitClass(AutoSOAP_ReflectionAccept $class)
    {
        $elem_name = "Class-" . $class -> getName(); 
        // まだ、クラスマップに追加さていないクラスなら追加する
        if (!isset($this -> _classMap[$elem_name])) {
            // クラスマップに追加
            $this -> _classMap[$elem_name] = $class -> getName(); 
            // インターフェースは登録しない
            if ($class -> isInterface()) {
                // アノテーションの取得
                $annotation = AutoSOAP_Anno :: parseReflector($class);
                $elemTypes = AutoSOAP_ElementType :: createByClass($annotation);
                foreach($elemTypes as $newtype => $elemType) {
                    $this -> getElementInfo('', $elemType, array('new_type' => $newtype));
                } 

                return $elem_name;
            } 

            /**
             * インターフェースから新しい型をインクルードする
             */
            $interfaces = $class -> getInterfaces();
            $interfaces -> accept($this); 
            // プロパティ追加
            $property = $class -> getProperties() -> accept($this);
            $result = $this -> _schema -> createComplexType($elem_name);
            if (!is_null($property)) {
                $result -> add($property);
            } 

            $this -> _root -> add($result);
        } 

        return $elem_name;
    } 

    /**
     * 子を処理する
     */
    public function visitChildren(AutoSOAP_ReflectionChildren $children)
    {
        $itr = $children -> getIterator();
        if (!$itr -> valid()) {
            return ;
        } 
        // クラスはインターフェースとみなして処理する
        if ('ReflectionClass' == $itr -> current() -> getReflectionType()) {
            while ($itr -> valid()) {
                $itr -> current() -> accept($this);
                $itr -> next();
            } 
            return ;
        } 

        $seq = $this -> _schema -> createSequence();
        while ($itr -> valid()) {
            $tmp = $itr -> current() -> accept($this);
            if (!is_null($tmp))
                $seq -> add($tmp);
            $itr -> next();
        } 
        return $seq;
    } 

    /**
     * 型指定の為のエレメントを返す。
     */
    public function getDataTypeElement($name, AutoSOAP_ElementType $elemType, $options = null)
    {
        $info = $this -> getElementInfo($name, $elemType, $options);
        return $this -> _schema -> createElement($info['name'], $info['datatype'], $info['namespace'], $options);
    } 

    /**
     * 型を生成する為の情報を取得する
     */
    public function getElementInfo($name, AutoSOAP_ElementType $elemType, $options)
    {
        $datatype = (isset($options['ref_class'])) ? $options['ref_class'] : $elemType -> datatype;
        $restriction = $elemType -> restriction;

        $namespace = $this -> _Target_uri;
        $flg_array = false;
        $array_options = array('array_min' => -1, 'array_max' => -1);
        $ref_class = null;

        if (is_null($datatype)) {
            $datatype = 'anyType';
        } 

        if (is_string($datatype)) {
            // 配列への対応
            $matches = array();
            if (preg_match("/([^\[\]]+)\[(?:(-*[0-9]+)(?:-([0-9]+))?)?\]/i", $datatype, $matches)) {
                $datatype = $matches[1];
                if (isset($matches[3])) {
                    $array_options['array_min'] = (int)$matches[2] ;
                    $array_options['array_max'] = (int)$matches[3] ;
                } else if (isset($matches[2])) {
                    $array_options['array_min'] = -1 ;
                    $array_options['array_max'] = (int)$matches[2] ;
                } 
                $flg_array = true;
            } 
            if (preg_match("/mixed/i", $datatype)) {
                $datatype = "anyType";
            } 
            // 頭に「#」が付いていた場合、制限をかけた型をみなす。
            if (preg_match("/^#/i", $datatype)) {
                $tmp_type = substr($datatype, 1);
                if (!isset($this -> _RestrictionMap[$tmp_type])) {
                    trigger_error("Restriction Type '" . $datatype . "' is not registed Type", E_USER_WARNING);
                } 
                $datatype = $this -> _RestrictionMap[$tmp_type];
                $ref_class = '#';
            } else {
                // 既存の型かチェックし、なければ、クラス名として扱う。
                $flg_simple_type = false;
                foreach (self :: $_xsd_var_types as $tmp_type) {
                    if (0 == strcasecmp($datatype, $tmp_type)) {
                        $datatype = $tmp_type;
                        $flg_simple_type = true;
                        break;
                    } 
                } 
                if (false === $flg_simple_type) {
                    $ref_class = new AutoSOAP_ReflectionAcceptImpl(new ReflectionClass($datatype));
                } 
            } 
        } else {
            $ref_class = $datatype;
        } 

        if ($ref_class instanceof AutoSOAP_ReflectionAcceptImpl) {
            $datatype = $ref_class -> accept($this);
        } else if (0 == count($restriction)) {
            if ('#' !== $ref_class) {
                $namespace = null;
            } 
        } 
        // 制限はanyTypeもしくはクラス指定ではない場合のみ適応
        else if ('anyType' !== $datatype) {
            $restriction_name = (isset($options['new_type'])) ? $options['new_type'] : null ;
            $r_datatype = ('#' === $ref_class) ? array($datatype, $this -> _Target_uri) : array($datatype, null) ;
            $datatype = $this -> createRestrictionElement($r_datatype, $restriction, $restriction_name);
        } 
        // 配列化
        if (true === $flg_array) {
            $datatype = $this -> getArrayElementName($datatype, $namespace, $array_options);
            $namespace = $this -> _Target_uri;
        } 

        return array('name' => $name, 'datatype' => $datatype, 'namespace' => $namespace);
    } 

    /**
     * 配列型を生成し、そのエレメント名を返す。
     */
    private function getArrayElementName($datatype, $namespace, $options)
    {
        $min_count = (0 < $options['array_min']) ? (int)$options['array_min'] : 0;
        $max_count = (0 < $options['array_max']) ? (int)$options['array_max'] : 0;

        $name = 'ArrayOf' . $datatype . '_' . $min_count . '-' . $max_count;

        if (!isset($this -> _ArrayMap[$name])) {
            $this -> _root -> add($this -> _array_fctry -> createArray($name, $datatype, $namespace, $min_count, $max_count));
            $this -> _ArrayMap[$name] = true;
        } 
        return $name;
    } 

    /**
     * 制限された型を生成し、そのエレメント名を返す。
     */
    private function createRestrictionElement($datatype, $restriction, $restriction_name = null)
    {
        if (is_null($restriction_name)) {
            $restriction_name = 'Restriction-' . ($this -> _restrictionCount++);
        } else {
            if (isset($this -> _RestrictionMap[$restriction_name])) {
                trigger_error("Restriction Type '" . $restriction_name . "' is already registed", E_USER_ERROR);
                return null;
            } 
            $this -> _RestrictionMap[$restriction_name] = 'R-' . $restriction_name;
            $restriction_name = $this -> _RestrictionMap[$restriction_name] ;
        } 

        $simple = $this -> _schema -> createSimpleType($restriction_name);
        $base = $this -> _schema -> createRestrictionBase($datatype[0], $datatype[1]);
        $simple -> add($base);
        foreach($restriction as $num => $rest) {
            list($rest_type, $rest_val) = each($rest);
            $base -> add($this -> _schema -> createRestriction($rest_type, $rest_val));
        } 

        $this -> _root -> add($simple);

        return $restriction_name;
    } 

    /**
     * 結果を返す
     */
    public function getResult()
    {
    } 
    /**
     * クラスマップを返す
     */
    public function getClassmap()
    {
        return $this -> _classMap;
    } 
    /**
     * 配列マップを返す
     */
    public function getArraymap()
    {
        return $this -> _ArrayMap;
    } 
    /**
     * 引数を処理する
     */
    public function visitParameter(AutoSOAP_ReflectionAccept $param)
    {
    } 
    /**
     * メソッドを処理する
     */
    public function visitMethod(AutoSOAP_ReflectionAccept $method)
    {
    } 
    /**
     * オブジェクトを処理する
     */
    public function visitObject(AutoSOAP_ReflectionAccept $object)
    {
    } 
    /**
     * エクステンションを処理する
     */
    public function visitExtension(AutoSOAP_ReflectionAccept $extension)
    {
    } 
    /**
     * 関数を処理する
     */
    public function visitFunction(AutoSOAP_ReflectionAccept $func)
    {
    } 
} 

?>