/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.lex.Keyword
import plus.lex.Node
import plus.lex.Symbols
import plus.util.NumHelper

/**
 * Code Generator - Implement.
 *
 * @author kunio himei.
 */
@CompileStatic
abstract class GenImpl extends GenInvoke {

    /** 連続代入対応 - a = b = value -> a = value; b = value
     */
    private static String chainAssign(String name, Object value) {
        Closure isAtomc = { String x ->
            x.startsWith('new AtomicNumber')
        }
        String x = value
        if (x.matches(/^\w+(\[.+])?$/)) { // 変数名[index]
            String chain = chainMap.get(x) // name = x = atom
            if ((null != chain) && isAtomc(chain))
                x = chain // ショートカット name = (x =) atom
            else chainMap.clear()
        }
        if (isAtomc(x)) chainMap.clear() // Atom なら連鎖マップをクリア.
        chainMap.put(name, x)
        return x
    }

    //* 代入文
    Object assignStmt(Node.Ass e) {
        List buf = []
        Closure append = {
            if (exists(it)) buf.add(it)
        }
        isVarValDef = // var,val定義開始
                (Keyword.SymVAL == e.id || Keyword.SymVAR == e.id)
        finalValDef = // final変数定義を初期化
                (Keyword.SymVAL == e.id) ? 'final ' : ''

        if (exists(e.left.index) && !Symbols.findType(e.left.name))
            mkGlobalArray(e.left.name) // 配列が未定義なら新規作成
        Object ee = e.right
        switch (ee) {
            case Node.Ass: // 右辺は代入文
                Node.Ass x = ee as Node.Ass
                Object value = eval(x.left)
                append(assignStmt(new Node.Ass(
                        e.id, x.name, x.left, x.right, x.nType, e.sType)))
                append(assignImpl(e, value))
                break
            case Node.YyVariable: // 右辺は変数
                Node.YyVariable x = ee as Node.YyVariable
                Object value = eval(x)
                append(assignImpl(e, value))
                break
            default:
                append(assignImpl(e, eval(ee)))
        }
        isVarValDef = false // var,val定義終了
        finalValDef = '' // final変数定義を初期化
        return mkString(buf.toArray(), '', '; ', '')
    }

    //* 代入文(下請け)
    String assignImpl(Node.Ass e, Object value) {
        List buf = []
        String[] vNam = varName(e.left) // vnam, index
        String name = mkVerNameX(vNam) // name[index]
        String dd = getAttribute(vNam[0], e.sType) // 属性取得
        Closure append = {
            if (exists(it)) buf.add(it)
        }
        Closure reassign = { // 再代入
            String ddx = (isVarValDef) ? dd : '' // 'Var/Val'の場合のみ属性を設定.
            "${finalValDef}${ddx}${name} ${e.name} ${it}"
        }
        Closure assign = { Object x -> // 代入
            String opx = getOperator(e.name) // = += -= *= /= %=
            String ddx = ('=' == opx) ? dd : '' // '='の場合のみ属性を設定.
            if (Symbols.isPredef(vNam[0])) // 組み込み変数
                return "${name}(${x})"
            return "${ddx}${name} ${opx} ${x}"
        }

        Object x = (value instanceof Number) ? // 数値の正規化
                NumHelper.normalise((Number) value) : value
        if (exists(vNam[1])) { // 配列
            append(assign(x))

        } else if ('=' == e.name) { // 代入文 (新規変数作成または再代入)
            if (Keyword.SymVAL == e.id || Keyword.SymVAR == e.id) {
                if (Keyword.SymVAR == e.id && x instanceof Number) {
                    // REMIND varでプリミティブ(小文字)のとき.
                    boolean isPrimitive = exists(e.sType) &&
                            (e.sType == e.sType.toLowerCase())
                    if (!isPrimitive)
                        x = "new AtomicNumber(${x})"
                }
                x = chainAssign(name, x) // 連続代入対応 ☆
                append("${finalValDef}${dd}${name} = ${x}")
                merkType(vNam[0], e.nType, isGlobalDef)

            } else if (Symbols.isPredef(vNam[0])) { // 組み込み変数
                append(reassign(x))

            } else if (Symbols.isLocal(vNam[0])) { // 関数内ローカル変数
                String claz = newClass(e.nType)
                if (exists(claz))
                    append(reassign(x))
                else append(reassign(parseNumber(e)))

                if (!Symbols.isLocal(vNam[0]))
                    Symbols.markLocal(vNam[0], e.nType) // 変数を登録

            } else if (Symbols.isGlobal(name)) { // グローバル定義済み
                append(reassign(parseNumber(x)))

            } else { // グローバル未定義
                String claz = newClass(e.nType)
                genGlobal(name, "${finalValDef}${dd}${name} = ${claz}")
                append(reassign(parseNumber(x)))

                if (!Symbols.isGlobal(vNam[0]))
                    Symbols.markGlobal(vNam[0], e.nType) // 変数を登録
            }
        } else { // 既存変数の更新
            append(assign(x))
        }
        return mkString(buf.toArray(), '', '; ', '')
    }

    //* 代入文 - called from Node.Getline.
    static String assignGline(String name, String index, String op, Object x) {
        name = getAlias(name) // POSTIT エイリアスの設定☆
        String opx = getOperator(op) // = += -= *= /= %=
        String dd = getAttribute(name, '') // Type無し.
        if (exists(index))
            return "${dd}${name}[${index}] ${opx} ${x}"
        if (Symbols.isPredef(name)) // 組み込み変数
            return "${name}(${x})"
        return "${dd}${name} ${opx} ${x}"
    }

    //* インクリメント, デクリメント、ビット否定
    String incdecNegate(String op, Node.YyVariable e) {
        String[] vNam = varName(e) // name, index
        String item = mkVerNameX(vNam)
        checkAndGlobalAlloc(vNam[0]) // 未定義ならグロ－バル変数定義
        if ('++' == op || '--' == op || '~' == op) op + item
        else if ('+-' == op) item + '++'
        else item + '--'
    }

    //* 真値判定(Boolean|0|1) 論理否定(!)
    String boolIf(String id, Object e) {
        String x = eval(e)
        String b = x.startsWith('_in') ? // 論理式ならオプティマイズ
                paren(x) : '_if' + paren(x)
        return (Node.NOT_OP == id) ? '!' + b : b
    }

    //* break, continue 開始. (do while エミュレータで使用) NOTE 雛形として残す.
    static void beginBreak() {
        yyIndent += 1
        gen("try {")
    }

    //* break, continue 終了. NOTE 雛形として残す.
    static void endBreak(String hasNext) {
        gen("} catch (plus.exception.BreakException e) { ${hasNext} = false")
        gen("} catch (plus.exception.ContinueException e) {") // continue
        gen("}")
        yyIndent -= 1
    }
}