/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus;

import plus.concurrent.Argument;
import plus.eval.Eval;
import plus.lex.Parser;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * AWK~plus Interpreter.
 *
 * @author kunio himei.
 */
public final class Plus {

    //* library path.
    private String libPath = "./";
    //* Import file name.
    private static final String IMPORT = "lib/import.awk";
    //* Per-define Script file name.
    private static final String PREDEF = "lib/predefJ.awk";
    //* User-define Script file name.
    private static final String USER = "lib/user.awk";
    /**
     * Specify where to place source files.
     */
    private String sourcePath = "";

    /**
     * ファイルが存在するかどうかを返す.
     */
    private static boolean hasFile(File f) {
        return f.isFile() && f.canRead();
    }

    /**
     * Selects an interval of elements.
     */
    private static String[] slice(String[] a, int start) {
        int length = a.length - start;
        if (0 < length) {
            String[] arr = new String[length];
            System.arraycopy(a, start, arr, 0, length);
            return arr;
        }
        return new String[0];
    }

    /**
     * Main.
     */
    public static void main(String[] args) throws Throwable {
        if (0 == args.length) {
            System.out.print(Argument.USAGE);
            System.out.println("AWK~plus Ver.2.1");
        } else {
            int exitCode = new Plus().execute(args);
            if (0 != exitCode) {
                System.exit(exitCode);
            }
        }
    }

    /*
     * ファイルの存在位置を返す.
     */
    private static final String AWK_MAGIC_NUMBER = "#!/awk\n";
    private static final Pattern IS_FULL_AWK = Pattern.compile( // \b: Word boundaries
            "^" + AWK_MAGIC_NUMBER + "|\\b(?:BEGIN|END)\\b");

    private String getPath(String name) throws IllegalArgumentException {
        if (hasFile(new File(name)))
            return name;
        File source = new File(sourcePath, name);
        if (hasFile(source))
            return source.getPath();
        if (IS_FULL_AWK.matcher(name).find())
            return name;
        throw new IllegalArgumentException(name);
    }

    //* Execute.
    private int execute(String[] args) throws Throwable {
        List<String> list = new ArrayList<>();
        int argp = 0;
        String src;
        if ("-sp".equals(args[argp])) { // source path
            argp++;
            sourcePath = args[argp++];
        }
        if ("-lib".equals(args[argp])) { // library path
            argp++;
            libPath = args[argp++] + '/';
        }
        if ("-e".equals(args[argp])) { // Eval
            argp++;
            src = AWK_MAGIC_NUMBER + args[argp++];
        } else {
            src = getPath(args[argp++]);
        }
        list.add(libPath + IMPORT);
        list.add(libPath + PREDEF);
        File user = new File(libPath, USER);
        if (user.isFile()) list.add(user.getPath());
        list.add(src);
        String[] scripts = list.toArray(new String[0]);
        Parser.SyntaxTree x = new Parser().parse(scripts);
        return new Eval(x).execute(slice(args, argp));
    }
}