/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

/**
 * Parser - Action.
 *
 * @author kunio himei.
 */
abstract class Action extends Expression {

    //* 空文 (print).
    private static final Node.Print EMPTY_PRINTSTMT = new Node.Print(
            Keyword.toName(Keyword.SymPRINT), Function.EMPTY_ARRAY, "", "");
    //* ^[;}\n].
    private static final LexRegx rxDOSTMT = new LexRegx("^[;}\n]");
    //* ^[;}\n].
    private static final LexRegx rxOPTSTATEMENT = new LexRegx("^[;}\n]");
    //* break, continue の出現数.
    private int cBREAK;
    //* Pattern to Pattern Matcher.
    private int cP2PID;

    /**
     * アクション.
     */
    @Override
    void action() {
        eat("{");
        nl();
        while ((null != super.tok) && !"}".equals(super.tok)) {
            optStatement();
        }
        eat("}");
        nl();
    }

    /**
     * 文 do optStatement while (logicalop).
     */
    private void doStmt() {
        Object[] backup = unloadBuffer();
        int breakCount = cBREAK;
        eat(Keyword.SymDO);
        optSEMI();
        if (rxDOSTMT.find(super.tok)) {
            nl(); // 空文検出
        } else if (Keyword.SymWHILE != super.tok) {
            if ("{".equals(super.tok)) {
                statement();
            } else {
                while (Keyword.SymWHILE != super.tok) {
                    statement();
                }
            }
        }
        eat(Keyword.SymWHILE);
        Object e = logicalop();
        Object[] stmt = super.codeBuf.toArray();
        restoreBuffer(backup);
        gen(new Node.While((hasBreak(breakCount)) ?
                Keyword.SyyBDO : Keyword.SymDO, e, stmt));
    }

    /**
     * for の式 for(n=1; 2; 3).
     */
    private Object forArgments(int n) {
        if (1 == n) {
            super.enableReAssign++; // 再代入を許可する
        }
        Object e = Function.EMPTY_ARRAY;
        if (!";".equals(super.tok) && !")".equals(super.tok)) {
            e = ((2 == n) ? genCompare(expression()) : expression());
        }
        if (";".equals(super.tok)) {
            advance();
            if ("\n".equals(super.tok)) {
                advance();
            }
        }
        if (1 == n) {
            super.enableReAssign--;
        }
        return Type.unWrapList(e);
    }

    /**
     * 文 for (expression; logicalop; expression) optStatement | for (expression
     * in expression) optStatement.
     */
    private void forStmt() {
        Object[] backup = unloadBuffer();
        int breakCount = cBREAK;
        eat(Keyword.SymFOR);
        eat("(");
        Object e1 = forArgments(1);
        if (")".equals(super.tok)) { // for(in)
            eat(")");
            Node.YyVariable left;
            Object right;
            if (e1 instanceof Node.Ass x) { // relativeop からの戻り値を分解する
                // for(`var e1 in .)
                left = new Node.NAME(x.id, x.left.name);
                right = x.right;
            } else if (e1 instanceof Node.Comp x) { // for(e1 in .)
                Node.Call a = (Node.Call) x.left;
                Object o = Type.unWrapList(a.args);
                if (o instanceof Node.YyVariable z) {
                    assignType("ForIn", z.name, x.right, 0); // NAME, Arr を代入済みにする
                    left = z;
                } else {
                    left = new Node.NAME(Keyword.SyyNAME, o.toString());
                }
                right = x.right;
            } else {
                throw new IllegalStateException("unmatch: " + e1);
            }
            optStatement();
            Object[] stmt = super.codeBuf.toArray();
            restoreBuffer(backup);
            Keyword forID = hasBreak(breakCount) ?
                    Keyword.SyyBFOR : Keyword.SymFOR;
            gen(new Node.ForIn(forID, left, right, stmt));
        } else {
            Object w = forArgments(2);
            Object e2 = Type.isEmpty(w) ? Boolean.TRUE : w;
            Object e3 = forArgments(3); //  ;TRACE("for("+ e1 +";"+ e2 +";"+ e3 +")")
            eat(")");
            optStatement();
            Object[] stmt = super.codeBuf.toArray();
            restoreBuffer(backup);
            Keyword forID = hasBreak(breakCount) ?
                    Keyword.SyyBFOR : Keyword.SymFOR;
            gen(new Node.For(forID, e1, e2, e3, stmt));
        }
    }

    /**
     * break, continue が出現したかどうかを返す.
     */
    private boolean hasBreak(int count) {
        int breakCount = cBREAK - count;
        cBREAK -= breakCount;
        return 0 < breakCount;
    }

    /**
     * 文 if (logicalop) optStatement else optStatement | if (logicalop)
     * optStatement.
     */
    private void ifStmt() {
        Object[] backup = unloadBuffer();
        eat(Keyword.SymIF);
        Object e = logicalop();
        optStatement();
        Object[] left = unloadBuffer();
        Object[] right;
        if (Keyword.SymELSE == super.tok) { // optional else
            eat(super.tok);
            nl();
            optStatement();
            right = super.codeBuf.toArray();
        } else {
            right = Function.EMPTY_ARRAY;
        }
        restoreBuffer(backup);
        gen(new Node.If(Keyword.SymIF, e, left, right));
    }

    /**
     * logical operation.
     */
    private Object logicalop() {
        return genCompare("(".equals(super.tok) ? parenlist() : expression());
    }

    /**
     * 文 Option Statement.
     */
    private void optStatement() {
        super.lexlinenumber = super.yyLexNumber; // ▼スクリプト行番号を退避
        optSEMI();
        if (rxOPTSTATEMENT.find(super.tok)) {
            nl();
        } else {
            statement();
        }
    }

    /**
     * パターン.
     */
    Object[] pattern(boolean doEmpty) {
        Object stmt;
        if ("{".equals(super.tok)) { // { action }
            action();
            stmt = ((doEmpty && super.codeBuf.isEmpty()) ?
                    EMPTY_PRINTSTMT : super.codeBuf.toArray());
        } else {
            Object p1 = expression(); // pattern
            Object p2 = null;
            if (",".equals(super.tok)) {
                eat(",");
                p2 = expression(); // pattern, pattern {
            }
            if ("{".equals(super.tok)) { // { action }
                action();
            }
            Object ex = (super.codeBuf.isEmpty()) ?
                    EMPTY_PRINTSTMT : super.codeBuf.toArray();
            Object cc;
            if (null == p2) {
                cc = genCompare(p1);
            } else {
                cc = new Node.Comp(Integer.toString(++cP2PID),
                        genCompare(p1), genCompare(p2));
            }
            stmt = new Node.If(Keyword.SymIF, cc, Type.castWrapArray(ex),
                    Function.EMPTY_ARRAY); // if (pattern){ action }{}
        }
        nl();
        super.codeBuf.clear();
        return Type.castWrapArray(stmt);
    }

    /**
     * 単純な文 simple Statement.
     */
    private void simpleStmt() {
        if (super.tok instanceof Node.Annotation) {
            // Annotation インライン注釈を食べる.
//            System.err.println("eat.simpleStmt.Annotation: " + super.tok);
            eat(super.tok);
        } else if (Keyword.SymDELETE == super.tok) { // delete
            advance();
            Object o = term();
            Node.YyVariable x = (Node.YyVariable) o; // NAME, Arr
            if (":".equals(super.tok)) {
                optType(x.name, false); // 削除 配列属性
            }
            updateType(x, Flags.T11ANY, Flags.T16ARRAY | Flags.T20ASSIGN);
            gen(new Node.Del(x));
        } else if (Keyword.SymBREAK == super.tok) { // break
            eat(super.tok);
            cBREAK++; // break, continue が出現
            gen(new Node.Stmt(Keyword.SymBREAK, Function.EMPTY_ARRAY, Flags.T00VOID));
        } else if (Keyword.SymCONTINUE == super.tok) { // continue
            eat(super.tok);
            cBREAK++; // break, continue が出現
            gen(new Node.Stmt(Keyword.SymCONTINUE, Function.EMPTY_ARRAY, Flags.T00VOID));
        } else if (Keyword.SymEXIT == super.tok) { // exit
            advance();
            Object e = rxOPTSTATEMENT.find(super.tok) ? Term.NUMBER_ZERO
                    : expression();
            if (!Flags.isNumber(Type.getNodeType(e))) {
                yyWARNING("ToNUMBER: exit(" + e + ')');
                e = new Node.IncDec(Node.NUMBER_OP, e); // 数値正規化
            }
            gen(new Node.Stmt(Keyword.SymEXIT, e, Flags.T00VOID));
        } else if (Keyword.SymRETURN == super.tok) {
            advance();
            super.cRETURN++; // return
            Object e = (rxOPTSTATEMENT.find(super.tok))
                    ? Function.EMPTY_ARRAY : expression();
            Object ex = assignType("RETURN", super.functionId, e, 0); // Type check: 復帰値(の代入)
            if (Flags.T26NIL == Symbols.getType(super.functionId)) {
                Symbols.setType(super.functionId, Flags.T11ANY | Flags.T20ASSIGN, 0);
            }
            gen(new Node.Stmt(Keyword.SymRETURN, ex, Type.getNodeType(ex)));
        } else if (Keyword.SymNEXT == super.tok) {
            eat(super.tok); // next
            gen(new Node.Stmt(Keyword.SymNEXT, Function.EMPTY_ARRAY, Flags.T00VOID));
        } else if (Keyword.SymNEXTFILE == super.tok) {
            eat(super.tok); // nextfile
            gen(new Node.Stmt(Keyword.SymNEXTFILE, Function.EMPTY_ARRAY,
                    Flags.T00VOID));
        } else if (Keyword.SymTHROW == super.tok) {
            advance(); // throw
            Object e = expression();
            gen(new Node.Throw(e));
        } else {
            gen(expression());
        }
    }

    /**
     * 文 statement.
     */
    private void statement() {
        if (hasComment()) {
            gen(new Node.Comment(getComment().mkString("//", "\n\t//", ""))); // 注釈
        }
        Object x = super.tok;
        if ("{".equals(x)) {
            eat("{");
            nl();
            while ((null != super.tok) && !"}".equals(super.tok)) {
                statement();
            }
            eat("}");
        } else if (Keyword.SymIF == x) {
            ifStmt();
        } else if (Keyword.SymFOR == x) {
            forStmt();
        } else if (Keyword.SymDO == x) {
            doStmt();
        } else if (Keyword.SymWHILE == x) {
            whileStmt();
        } else if (Keyword.SymTRY == x) { // try catch finally
            tryCatchStmt();
        } else if (Keyword.SymFUNCTION == x) {
            eat(super.tok);
            functionDecl();
        } else {
            simpleStmt();
        }
        nl();
    }

    /**
     * 文 try　{} { catch (exception e) {... finally {}.
     */
    private void tryCatchStmt() {
        LexArray<Node.Catch> catchs = new LexArray<>();
        Object[] backup = unloadBuffer();
        eat(Keyword.SymTRY); // `try`
        optStatement(); // {...}
        Object[] trystmt = unloadBuffer();
        while (Keyword.SymCATCH == super.tok) { // catch (ex1 (| ex2...) e) {
            LexArray<String> exceptions = new LexArray<>();
            eat(super.tok);
            eat("(");
            while (true) {
                Object xx = super.tok; // 例外クラス名.
                Term.BOXING claz = classNameOrNull(xx);
                if (null == claz)
                    throw new IllegalArgumentException("Var type mistake! '" + xx + "' " + xx.getClass());
                eat(super.tok);
                exceptions.add(claz.value.toString());
                if ("|".equals(super.tok)) eat("|");
                else break;
            }
            Object xx = super.tok; // 例外変数名.
            if (!(xx instanceof Node.NAME ex))
                throw new IllegalArgumentException("Var type mistake! '" + xx + "' " + xx.getClass());
            eat(super.tok);
            eat(")");
            Symbols.createType(ex.name);
            Symbols.setType(ex.name, Flags.T11ANY, Flags.T20ASSIGN); // local value
            optStatement(); // {...}
            Object[] catstmt = unloadBuffer();
            catchs.add(new Node.Catch(ex.name,
                    exceptions.toArray(new String[0]), catstmt));
        }
        if (Keyword.SymFINALLY == super.tok) { // finally
            eat(super.tok);
            optStatement(); // {...}
        }
        Object[] finalstmt = unloadBuffer();
        Object w = new Node.Try(trystmt, catchs.toArray(new Node.Catch[0]), finalstmt);
        restoreBuffer(backup);
        gen(w);
    }

    /**
     * 文 while (logicalop) optStatement.
     */
    private void whileStmt() {
        Object[] backup = unloadBuffer();
        int breakCount = cBREAK;
        eat(Keyword.SymWHILE);
        Object e = logicalop();
        optStatement();
        Object[] stmt = super.codeBuf.toArray();
        restoreBuffer(backup);
        gen(new Node.While((hasBreak(breakCount)) ?
                Keyword.SyyBWHILE : Keyword.SymWHILE, e, stmt));
    }
}