/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import plus.io.Device;
import plus.io.Io;
import plus.io.IoHelper;
import plus.spawn.system.UtilInterface;
import plus.util.Escape;

import java.io.FilterWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.concurrent.atomic.AtomicReference;

/**
 * [%command%] implementation of the `echo` command.
 * <p>
 * Usage: echo [OPTION]... [STRING]... <br>
 * 文字列を表示する <br>
 * <br>
 * -n 改行を付加しない <br>
 * -e 文字列中のエスケープ文字を有効にする<br>
 * <br>
 * Without -E, the following sequences are recognized and interpolated: <br>
 * \a alert (BEL) <br>
 * \b backspace <br>
 * \f form feed <br>
 * \n new line <br>
 * \r carriage return <br>
 * \t horizontal tab <br>
 * \v vertical tab <br>
 * \\ backslash <br>
 * \uFEFF UNICODE <br>
 * \xFF Hex decimal <br>
 * \0777 Octal decimal <br>
 * </p>
 *
 * @author kunio himei.
 */
public final class Echo extends FilterWriter implements UtilInterface {

    /**
     * USAGE.
     */
    private static final String USAGE = """
            Usage: echo [OPTION]... [STRING]...
              -n    do not output the trailing newline
              -e    enable interpolation of some sequences in Strings
                  --help      display this help and exit (should be alone)

            With -e, the following sequences are recognized and interpolated:
              \\a alert (BEL)
              \\b backspace
              \\f form feed
              \\n new line
              \\r carriage return
              \\t horizontal tab
              \\v vertical tab
              \\\\ backslash
              \\uFEFF UNICODE
              \\xFF Hex decimal
              \\0777 Octal decimal""";
    /**
     * -E disable interpolation of some sequences in Strings.
     */
    private final boolean hasDecode;
    /**
     * -n do not output the trailing newline.
     */
    private final boolean hasNewLine;
    /**
     * output Num.
     */
    private final AtomicReference<String> outputText = new AtomicReference<>();

    /**
     * echo.
     */
    public Echo(String[] args, Writer output) {
        super(output);
        StringBuilder flags = new StringBuilder();
        StringBuilder txt = new StringBuilder();
        boolean isTop = true;
        for (String arg : args) {
            char c = arg.charAt(0);
            if (isTop && ('-' == c) && (2 == arg.length())
                    && (0 <= "neE".indexOf(arg.charAt(1)))) {
                // 先頭の有効なオプションのみ解釈する
                // 手抜き互換のため、オプションは､1文字のみ有効
                flags.append(arg.substring(1));
            } else {
                if (0 != txt.length()) {
                    txt.append(' ');
                }
                txt.append(arg);
                isTop = false;
            }
        }
        this.hasNewLine = (0 > flags.indexOf("n")); // 改行の抑制
        this.hasDecode = (0 <= flags.indexOf("e")); // エスケープを解釈
        this.outputText.set(txt.toString());

        if (IS_VERBOSE) {
            StringBuilder sb = new StringBuilder("`echo");
            if (0 < flags.length()) {
                sb.append(" -").append(flags);
            }
            sb.append(' ').append(this.outputText);
            System.err.println(sb.append('`'));
        }
    }

    /**
     *
     */
    public static void main(String[] args) throws IOException {
        if ((0 != args.length) && args[0].startsWith("--help")) {
            System.out.println(USAGE);
        } else {
            new Echo(args, Device.openOutput("", Io.STDOUT)).close();
        }
    }

    /**
     * このストリームを閉じる.
     */
    @Override
    public void close() throws IOException {
        String x = this.outputText.getAndSet(null);
        if (null != x) {
            writeln(x);
            super.flush();
        }
//        Io.close(super.out); // PowerShell と相性が悪いためリファクタリング
        super.out.flush();
    }

    /**
     * サブプロセスの終了値を返す.
     */
    @Override
    public int exitValue() {
        return 0;
    }

    /**
     *
     */
    @Override
    public boolean hasInput() {
        return false;
    }

    /**
     * 文字列を書き込む.
     */
    private void writeln(String x) throws IOException {
        String txt = (this.hasDecode) ? Escape.outputFilter(x) : x;
        if (this.hasNewLine) {
            IoHelper.writeln(super.out, txt);
        } else {
            super.write(txt);
        }
    }
}