/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.value;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import net.morilib.awk.AwkArithmeticException;
import net.morilib.awk.AwkCastException;
import net.morilib.awk.namespace.AwkNamespace;

public abstract class AwkValue {

	private static final BigInteger MINB =
		BigInteger.valueOf(Integer.MIN_VALUE);
	private static final BigInteger MAXB =
		BigInteger.valueOf(Integer.MAX_VALUE);

	/**
	 * 
	 */
	public static final AwkValue TRUE = AwkInteger.valueOf(1);

	/**
	 * 
	 */
	public static final AwkValue FALSE = AwkInteger.valueOf(0);

	/**
	 * 
	 */
	public static final AwkValue IOERROR = AwkInteger.valueOf(-1);

	/**
	 * 
	 * @return
	 */
	public BigInteger toInteger() {
//		throw new AwkCastException(this + " is not a number");
		return BigInteger.ZERO;
	}

	/**
	 * 
	 * @return
	 */
	public int toIntExact() {
		if(!isIntValue()) {
			throw new AwkArithmeticException(
					"the integer is not small");
		}
		return toInteger().intValue();
	}

	/**
	 * 
	 * @return
	 */
	public double toFloat() {
//		throw new AwkCastException(this + " is not a number");
		return 0.0;
	}

	/**
	 * 
	 * @return
	 */
	public abstract String toString(AwkNamespace ns);

	/**
	 * 
	 * @return
	 */
	public String toString() {
		return toString(null);
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean toBoolean();

	/**
	 * 
	 * @return
	 */
	public Number toReal() {
		throw new AwkCastException(this + " is not a number");
	}

	/**
	 * 
	 * @return
	 */
	public Pattern toRegex() {
		throw new AwkCastException(
				this + " is not a regular expression");
	}

	/**
	 * 
	 * @return
	 */
	public AwkArray toArray() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public abstract Object toObject();

	/**
	 * 
	 * @return
	 */
	public AwkValue referArray(String v) {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public AwkValue putArray(String k, AwkValue v) {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @param k
	 * @return
	 */
	public boolean contains(String k) {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public AwkValue deleteArray(String v) {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public Set<String> keys() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public Collection<AwkValue> values() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public Set<Map.Entry<String, AwkValue>> entrySet() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public Map<String, AwkValue> map() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public void clear() {
		throw new AwkCastException(this + " is not an array");
	}

	/**
	 * 
	 * @return
	 */
	public int size() {
		return 1;
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean isInteger();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFloat();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isReal();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isIntegerValue();

	/**
	 * 
	 * @return
	 */
	public boolean isIntValue() {
		return (isIntegerValue() &&
				toInteger().compareTo(MINB) > 0 &&
				toInteger().compareTo(MAXB) < 0);
	}

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFloatValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isRealValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isPositiveValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isZeroValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isNegativeValue();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isString();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isRegex();

	/**
	 * 
	 * @return
	 */
	public abstract boolean isFunction();

	/**
	 * 
	 * @return
	 */
	public boolean isEmpty() {
		return false;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArray() {
		return false;
	}

	/**
	 * 
	 * @return
	 */
	public AwkNamespace getNamespace() {
		return null;
	}

	/**
	 * 
	 * @param v
	 * @param ns
	 * @return
	 */
	public abstract int compareTo(AwkValue v, AwkNamespace ns);

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static AwkValue cast(String s) {
		try {
			return AwkInteger.valueOf(new BigInteger(s));
		} catch(NumberFormatException e1) {
			try {
				return AwkFloat.valueOf(Double.parseDouble(s));
			} catch(NumberFormatException e2) {
				return AwkString.valueOf(s);
			}
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue add(AwkValue r1, AwkValue r2) {
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined) {
			return r2;
		} else if(r2 instanceof AwkUndefined) {
			return r1;
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> e : r2.entrySet()) {
				s = e.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							add(r1.referArray(s), e.getValue()));
				} else {
					r0.putArray(s, e.getValue());
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, add(z.getValue(), r2));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, add(r1, z.getValue()));
			}
			return r0;
		} else if(r1.isFloat() || r2.isFloat()) {
			return AwkFloat.valueOf(r1.toFloat() + r2.toFloat());
		} else if(r1.isIntegerValue() && r2.isIntegerValue()) {
			return AwkInteger.valueOf(
					r1.toInteger().add(r2.toInteger()));
		} else {
			return AwkFloat.valueOf(r1.toFloat() + r2.toFloat());
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue sub(AwkValue r1, AwkValue r2) {
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined) {
			return sub(AwkInteger.ZERO, r2);
		} else if(r2 instanceof AwkUndefined) {
			return sub(r1, AwkInteger.ZERO);
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> e : r2.entrySet()) {
				s = e.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							sub(r1.referArray(s), e.getValue()));
				} else {
					r0.putArray(s, neg(e.getValue()));
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, sub(z.getValue(), r2));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, sub(r1, z.getValue()));
			}
			return r0;
		} else if(r1.isFloat() || r2.isFloat()) {
			return AwkFloat.valueOf(r1.toFloat() - r2.toFloat());
		} else if(r1.isIntegerValue() && r2.isIntegerValue()) {
			return AwkInteger.valueOf(
					r1.toInteger().subtract(r2.toInteger()));
		} else {
			return AwkFloat.valueOf(r1.toFloat() - r2.toFloat());
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue mul(AwkValue r1, AwkValue r2) {
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined) {
			return mul(AwkInteger.ZERO, r2);
		} else if(r2 instanceof AwkUndefined) {
			return mul(r1, AwkInteger.ZERO);
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							mul(r1.referArray(s), z.getValue()));
				} else if(z.getValue().isFloat()) {
					r0.putArray(s, AwkFloat.valueOf(0.0));
				} else {
					r0.putArray(s, AwkInteger.ZERO);
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					r0.putArray(s, AwkInteger.ZERO);
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, mul(z.getValue(), r2));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, mul(r1, z.getValue()));
			}
			return r0;
		} else if(r1.isFloat() || r2.isFloat()) {
			return AwkFloat.valueOf(r1.toFloat() * r2.toFloat());
		} else if(r1.isIntegerValue() && r2.isIntegerValue()) {
			return AwkInteger.valueOf(
					r1.toInteger().multiply(r2.toInteger()));
		} else {
			return AwkFloat.valueOf(r1.toFloat() * r2.toFloat());
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue div(AwkValue r1, AwkValue r2) {
		BigInteger e, f;
		BigInteger[] a;
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined) {
			return div(AwkInteger.ZERO, r2);
		} else if(r2 instanceof AwkUndefined) {
			return div(r1, AwkInteger.ZERO);
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							div(r1.referArray(s), z.getValue()));
				} else if(z.getValue().isFloat()) {
					r0.putArray(s, AwkFloat.valueOf(0.0));
				} else {
					r0.putArray(s, AwkInteger.ZERO);
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(r2.contains(s)) {
					// do nothing
				} else if(z.getValue().isPositiveValue()) {
					r0.putArray(s, AwkFloat.valueOf(
							Double.POSITIVE_INFINITY));
				} else {
					r0.putArray(s, AwkFloat.valueOf(
							Double.NEGATIVE_INFINITY));
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, div(z.getValue(), r2));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, div(r1, z.getValue()));
			}
			return r0;
		} else if(r1.isFloat() || r2.isFloat()) {
			return AwkFloat.valueOf(r1.toFloat() / r2.toFloat());
		} else if(r1.isIntegerValue() && r2.isIntegerValue()) {
			try {
				e = r1.toInteger();
				f = r2.toInteger();
				a = e.divideAndRemainder(f);
				if(a[1].signum() == 0) {
					return AwkInteger.valueOf(e.divide(f));
				}
			} catch(ArithmeticException z) {
				throw new AwkArithmeticException(z.getMessage());
			}
		}
		return AwkFloat.valueOf(r1.toFloat() / r2.toFloat());
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue pow(AwkValue r1, AwkValue r2) {
		BigInteger f;
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined && r2 instanceof AwkUndefined) {
			throw new ArithmeticException();
		} else if(r1 instanceof AwkUndefined) {
			return pow(AwkInteger.ZERO, r2);
		} else if(r2 instanceof AwkUndefined) {
			return pow(r1, AwkInteger.ZERO);
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							pow(r1.referArray(s), z.getValue()));
				} else {
					r0.putArray(s, pow(AwkInteger.ZERO, z.getValue()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					r0.putArray(s, pow(z.getValue(), AwkInteger.ZERO));
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, pow(z.getValue(), r2));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, pow(r1, z.getValue()));
			}
			return r0;
		} else if(!r1.isInteger() || !r2.isInteger()) {
			return AwkFloat.valueOf(Math.pow(
					r1.toFloat(), r2.toFloat()));
		} else if((f = r2.toInteger()).signum() < 0) {
			return AwkFloat.valueOf(Math.pow(
					r1.toFloat(), f.doubleValue()));
		} else if(f.signum() == 0) {
			if(r2.toInteger().signum() == 0) {
				throw new ArithmeticException();
			}
			return AwkInteger.ONE;
		} else if(f.equals(BigInteger.ONE)) {
			return AwkInteger.valueOf(r1.toInteger());
		} else if(f.compareTo(MAXB) <= 0) {
			return AwkInteger.valueOf(
					r1.toInteger().pow(f.intValue()));
		} else {
			throw new ArithmeticException();
		}
	}

	/**
	 * 
	 * @param v
	 * @return
	 */
	public static AwkValue neg(AwkValue v) {
		AwkArray r0;
		String s;

		if(v.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : v.entrySet()) {
				s = z.getKey();
				r0.putArray(s, neg(z.getValue()));
			}
			return r0;
		} else if(v.isInteger()) {
			return AwkInteger.valueOf(v.toInteger().negate());
		} else if(v.isFloat()) {
			return AwkFloat.valueOf(-v.toFloat());
		} else {
			throw new AwkCastException("number required");
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue concat(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray r0;
		String s;

		if(r1 instanceof AwkUndefined) {
			return r2;
		} else if(r2 instanceof AwkUndefined) {
			return r1;
		} else if(r1.isArray() && r2.isArray()) {
			r0 = new AwkArray(r1);
			for(Map.Entry<String, AwkValue> e : r2.entrySet()) {
				s = e.getKey();
				if(r1.contains(s)) {
					r0.putArray(s,
							concat(r1.referArray(s),
									e.getValue(), ns));
				} else {
					r0.putArray(s, e.getValue());
				}
			}
			return r0;
		} else if(r1.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				r0.putArray(s, concat(z.getValue(), r2, ns));
			}
			return r0;
		} else if(r2.isArray()) {
			r0 = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				r0.putArray(s, concat(r1, z.getValue(), ns));
			}
			return r0;
		} else {
			return AwkString.valueOf(
					r1.toString(ns) + r2.toString(ns));
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue equals(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			a = new AwkArray();
			if(r2.isArray()) {
				for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
					a.putArray(z.getKey(), AwkInteger.booleanOf(
							z.getValue().isEmpty()));
				}
				return a;
			} else {
				return AwkInteger.booleanOf(r2.isEmpty());
			}
		} else if(r2 instanceof AwkUndefined) {
			a = new AwkArray();
			if(r1.isArray()) {
				for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
					a.putArray(z.getKey(), AwkInteger.booleanOf(
							z.getValue().isEmpty()));
				}
				return a;
			} else {
				return AwkInteger.booleanOf(r1.isEmpty());
			}
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = equals(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isEmpty()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isEmpty()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = equals(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = equals(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) == 0);
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @return
	 */
	public static AwkValue ne(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			a = new AwkArray();
			if(r2.isArray()) {
				for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
					a.putArray(z.getKey(), AwkInteger.booleanOf(
							!z.getValue().isEmpty()));
				}
				return a;
			} else {
				return AwkInteger.booleanOf(!r2.isEmpty());
			}
		} else if(r2 instanceof AwkUndefined) {
			a = new AwkArray();
			if(r1.isArray()) {
				for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
					a.putArray(z.getKey(), AwkInteger.booleanOf(
							!z.getValue().isEmpty()));
				}
				return a;
			} else {
				return AwkInteger.booleanOf(!r1.isEmpty());
			}
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = ne(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isEmpty()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isEmpty()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = ne(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = ne(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) != 0);
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @param ns
	 * @return
	 */
	public static AwkValue lt(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			return lt(AwkInteger.ZERO, r2, ns);
		} else if(r2 instanceof AwkUndefined) {
			return lt(r1, AwkInteger.ZERO, ns);
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = lt(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isPositiveValue()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isNegativeValue()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = lt(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = lt(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) < 0);
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @param ns
	 * @return
	 */
	public static AwkValue le(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			return le(AwkInteger.ZERO, r2, ns);
		} else if(r2 instanceof AwkUndefined) {
			return le(r1, AwkInteger.ZERO, ns);
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = le(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isNegativeValue()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isPositiveValue()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = le(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = le(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) <= 0);
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @param ns
	 * @return
	 */
	public static AwkValue gt(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			return gt(AwkInteger.ZERO, r2, ns);
		} else if(r2 instanceof AwkUndefined) {
			return gt(r1, AwkInteger.ZERO, ns);
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = gt(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isNegativeValue()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							z.getValue().isPositiveValue()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = gt(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = gt(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) > 0);
		}
	}

	/**
	 * 
	 * @param r1
	 * @param r2
	 * @param ns
	 * @return
	 */
	public static AwkValue ge(AwkValue r1, AwkValue r2,
			AwkNamespace ns) {
		AwkArray a;
		AwkValue x;
		String s;

		if(r1 instanceof AwkUndefined) {
			return ge(AwkInteger.ZERO, r2, ns);
		} else if(r2 instanceof AwkUndefined) {
			return ge(r1, AwkInteger.ZERO, ns);
		} else if(r1.isArray() && r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				if(r1.contains(s)) {
					x = ge(r1.referArray(s), z.getValue(), ns);
					a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
				} else {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isPositiveValue()));
				}
			}

			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				if(!r2.contains(s)) {
					a.putArray(s, AwkInteger.booleanOf(
							!z.getValue().isNegativeValue()));
				}
			}
			return a;
		} else if(r1.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r1.entrySet()) {
				s = z.getKey();
				x = ge(r1.referArray(s), r2, ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else if(r2.isArray()) {
			a = new AwkArray();
			for(Map.Entry<String, AwkValue> z : r2.entrySet()) {
				s = z.getKey();
				x = ge(r1, r2.referArray(s), ns);
				a.putArray(s, AwkInteger.booleanOf(x.toBoolean()));
			}
			return a;
		} else {
			return AwkInteger.booleanOf(r1.compareTo(r2, ns) >= 0);
		}
	}

}
