/*
    SDL_cfg:  An configuration file parser based on the SDL library
    Copyright (C) 2001-2007 kr2

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    kr2
    ee12054g@hotmail.co.jp
*/
#include "XMLElement.hpp"

#include <string.h>

using namespace std ;

XMLElement::XMLElement ( xmlNode* cnode, XMLElement* prt )
{
    node = cnode ;
    parent = prt ;
    if ( parent != NULL ) parent->clist.push_back ( this ) ;
}

XMLElement::~XMLElement ()
{
    list<XMLElement*>::iterator p = clist.begin () ;
    while ( p != clist.end() ) {
        delete (*p) ;
        p++ ;
    }
}

void XMLElement::Delete ()
{
    if ( node != NULL ) xmlFreeNodeList ( node ) ;
    if ( parent != NULL ) parent->clist.remove ( this ) ;
    delete this ;
}

XMLElement* XMLElement::GetElem ( const char* ename )
{
    return new XMLElement ( SearchElement (ename), this ) ;
}

XMLElement* XMLElement::SetElem ( const char* ename )
{
    XMLElement* elem ;
    xmlNode* a = SearchElement ( ename ) ;
    if ( a != NULL ) {
        elem = GetElementFromNode ( this, a ) ;
        if ( elem == NULL ) elem = new XMLElement ( a, this ) ;
    }else{
        elem = new XMLElement ( xmlNewChild ( node, NULL, BAD_CAST ename, NULL ), this ) ;
    }
    return elem ;
}

XMLElement* XMLElement::CreateElem ( const char* ename )
{
    return new XMLElement ( xmlNewChild ( node, NULL, BAD_CAST ename, NULL ), this ) ;
}

const char* XMLElement::GetName () const
{
    if ( node == NULL ) return NULL ;
    return (const char*)node->name ;
}

void XMLElement::SetName (const char* ename )
{
    if ( node == NULL ) return ;
    xmlNodeSetName ( node, BAD_CAST ename ) ;
}

const char* XMLElement::GetContent()
{
	return (const char*)xmlNodeGetContent(node) ;
}

void XMLElement::SetContent(const char* contentname)
{
    if ( node == NULL ) return ;
	xmlNodeSetContent(node,BAD_CAST contentname) ;
}

const char* XMLElement::GetPropStr ( const char* aname )
{
    if ( node == NULL ) return NULL ;
    return (const char*)xmlGetProp ( node, BAD_CAST aname ) ;
}

void XMLElement::DeleteProp ( const char* propname )
{
    if ( node == NULL ) return ;
    xmlAttr* attr = SearchProp ( propname ) ;
    if ( attr != NULL ) xmlRemoveProp ( attr ) ;
}

void XMLElement::SetPropStr ( const char* aname, const char* name )
{
    if ( node == NULL ) return ;
    if ( SearchProp ( aname ) == NULL ) xmlNewProp ( node, BAD_CAST aname, BAD_CAST name ) ;
	else xmlSetProp ( node, BAD_CAST aname, BAD_CAST name ) ;
}

XMLElement* XMLElement::Next ()
{
    if (node) {
        for (xmlNode* nextNode = node->next; nextNode != NULL; nextNode = nextNode->next) {
            if (nextNode->type == XML_ELEMENT_NODE) {
                XMLElement* elem = GetElementFromNode( parent, nextNode ) ;
                return elem ? elem : new XMLElement( nextNode, parent );
            }
        }
    }
    return NULL;
}

XMLElement* XMLElement::Prev ()
{
    if (node) {
        for (xmlNode* prevNode = node->prev; prevNode != NULL; prevNode = prevNode->prev) {
            if (prevNode->type == XML_ELEMENT_NODE) {
                XMLElement* elem = GetElementFromNode( parent, prevNode ) ;
                return elem ? elem : new XMLElement( prevNode, parent );
            }
        }
    }
    return NULL;
}

XMLElement* XMLElement::Next (const char* aname)
{
    if (node) {
        for (xmlNode* nextNode = node->next; nextNode != NULL; nextNode = nextNode->next) {
            if (nextNode->type == XML_ELEMENT_NODE) {
                XMLElement* elem =
                    GetElementFromNode( parent, SearchElement( nextNode, aname, false ) );
                return elem ? elem : new XMLElement( nextNode, parent );
            }
        }
    }
    return NULL;
}

XMLElement* XMLElement::Prev (const char* aname)
{
    if (node) {
        for (xmlNode* prevNode = node->prev; prevNode != NULL; prevNode = prevNode->prev) {
            if (prevNode->type == XML_ELEMENT_NODE) {
                XMLElement* elem =
                    GetElementFromNode( parent, SearchElement( prevNode, aname, false ) );
                return elem ? elem : new XMLElement( prevNode, parent );
            }
        }
    }
    return NULL;
}

XMLElement* XMLElement::GetChild()
{
    if (node) {
        for (xmlNode* child = node->children; child != NULL; child = child->next) {
            if (child->type == XML_ELEMENT_NODE) {
                return new XMLElement(child, this);
            }
        }
    }
    return NULL;
}

xmlNode* XMLElement::SearchElement ( const char* ename )
{
    if ( node == NULL ) return NULL ;
    return SearchElement ( node->children, ename, false ) ;
}

XMLElement* XMLElement::GetElementFromNode ( XMLElement* from, xmlNode* cnode )
{
    if ( cnode == NULL ) return NULL ;
    list<XMLElement*>::iterator p = from->clist.begin () ;
    while ( p != from->clist.end () ) {
        if ( (*p)->node == cnode ) return (*p);
        p++ ;
    }
    return NULL ;
}

xmlNode* XMLElement::SearchElement ( xmlNode* a_node, const char* ename, bool rev )
{
    xmlNode *ret_node = NULL ;
    xmlNode *cur_node = NULL ;
    for (cur_node = a_node; cur_node; cur_node = rev?cur_node->prev:cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE) {
            if ( strcmp ( (const char*)cur_node->name, ename ) == 0 ) {
                ret_node = cur_node ;
                break ;
            }
        }
        // 下の階層まで再帰的に検索をかけるなら、コメントを外す。
        //ret_node = SearchElement(cur_node->children, ename);
        //if ( ret_node != NULL ) break ;
    }
    return ret_node ;
}

xmlAttr* XMLElement::SearchProp ( const char* aname )
{
    xmlAttr *ret_attr = NULL ;
    xmlAttr *cur_attr = NULL ;
    for (cur_attr = node->properties; cur_attr; cur_attr = cur_attr->next) {
        if ( strcmp ( (const char*)cur_attr->name, aname ) == 0 ) {
            ret_attr = cur_attr ;
            break ;
        }
    }
    return ret_attr ;
}

