/*
 * $Id: ConnectionWrapper.java,v 1.1.1.1 2002/12/10 09:18:12 mitsuhito Exp $
 */


package jp.livewell.baby.pool.jdbc;


import java.lang.ref.WeakReference;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLWarning;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Vector;

import jp.livewell.baby.pool.AbstractWrapper;
import jp.livewell.baby.pool.ComparisonKey;
import jp.livewell.baby.pool.Pool;


/**
 * java.sql.Connection interfaceimplementsConnection˻ѤǤ
 * Wrapper classǤ
 * 
 * @author mitsuhito
 */
class ConnectionWrapper
    extends AbstractWrapper
    implements Connection
{
    // attributes -------------------------------------------------------------
    /** ºݤJDBC ConnectionǤ */
    private Connection conn;

    /** (ưߥ졼Ѥ)Ĥ줿JDBC ConnectionǤ */
    private Connection closedConn;

    /** Factory̤ConnectionνͤǤ */
    private DefaultValues defaults;

    /** StatementWrapperϤʬȤؤReference */
    private WeakReference ownRef;

    /** ǡ١³ǧSQLʸ(å˻) */
    private String command;

    /** Ǹå줿֤ꤳλ(ms)вᤷƤping */
    private long interval;

    /** Connection줿StatementWrapper(sub class)ݻ
	ޤConnection.close()κݤĤޤ*/
    private Vector statements = new Vector();


    // state attributes -------------------------------------------------------
    /** Ǹactivate줿(ms) */
    private long lastAccess = System.currentTimeMillis();

    /** autoCommitѹ */
    private boolean modAutoCommit = false;
    
    /** catalogѹ */
    private boolean modCatalog = false;
    
    /** readOnlyѹ */
    private boolean modReadOnly = false;
    
    /** transactionIsolationѹ */
    private boolean modIsolation = false;
    
    /** typeMapѹ */
    private boolean modTypeMap = false;


    // constructors & init ----------------------------------------------------
    /**
     * ConnectionFactoryƤФpackage scope constructorǤ
     * content˼ºݤ()Connectionꤷޤ
     *
     * @param	conn
     *		WrapJDBC Connection
     * @param	closedConn
     *		ĤJDBC Connectionclose򥨥ߥ졼Ȥݤ˻
     * @param	command
     *		ping˻Ѥǡ١ͭSQLʸ
     * @param	interval
     *		Ǹå줿֤鼡ΥåޤǤκû
     * @param	defaults
     *		JDBC줿ConnectionνͤǼinstance
     * @throws	SQLException
     *		ǡ١³ĤƤ硣
     *		command¹Խʤ硣
     * @throws	IllegalArgumentException
     *		connnullꤷ褦Ȥ硣
     *		closedConnnullꤷ褦Ȥ硣
     *		intervalͤꤷ褦Ȥ硣
     *		defaultsnullꤷ褦Ȥ硣
     */
    ConnectionWrapper(Connection	conn,
		      Connection	closedConn,
		      String		command,
		      long		interval,
		      DefaultValues	defaults)
	throws SQLException, IllegalArgumentException
    {
	if ((conn != null) && (closedConn != null)
		&& (interval >= 0) && (defaults != null)) {
	    this.conn		= conn;
	    this.closedConn	= closedConn;
	    this.command	= command;
	    this.interval	= interval;
	    this.defaults	= defaults;
	    this.ownRef		= new WeakReference(this);
	    ping();
	} else {
	    throw new IllegalArgumentException
		("please check values for create.\n"
		 + "conn = " + conn + "\n"
		 + "closedConn = " + closedConn + "\n"
		 + "command = " + command + "\n"
		 + "interval = " + interval + "\n"
		 + "defaults = " + defaults + "\n");
	}
    }


    // Wrapper implementation & helpers ---------------------------------------
    /**
     * Wrapper̵ȤPoolƤӽФޤtrue
     * ̵Ƥfalseᤵޤkeyפʤ
     * IllegalArgumentExceptionthrowޤ
     *
     * @param	key
     *		WrapperPoolͭComparisonKey
     * @return	true̵Ƥfalse
     * @throws	IllegalStateException
     *		Ǥʤͳ̵ʤ
     * @throws	IllegalArgumentException
     *		keyפʤ
     */
    public synchronized boolean passivate(ComparisonKey key)
	throws IllegalStateException
    {
	if (super.passivate(key)) {
	    closeStatementsAll();
	    
	    boolean isReset = defaults.resetValues
		(conn, modAutoCommit, modCatalog,
		 modReadOnly, modIsolation, modTypeMap);

	    if (isReset) {
		modAutoCommit	= false;
		modCatalog	= false;
		modReadOnly	= false;
		modIsolation	= false;
		modTypeMap	= false;
	    } else {
		throw new IllegalStateException
		    ("can't reset values to default!");
	    } 
	    return true;	    
	} else {
	    return false;
	}
    }

    /**
     * WrapperԲĤˤݤƤ꥽ľ˳
     * isWorkablefalseꤷޤ̾盧Υ᥽åɤPoolƤФޤ
     * destroyƤӽФ줿Wrapperưݾڤޤ
     * keyפʤIllegalArgumentExceptionthrowޤ
     *
     * @param	key
     *		WrapperPoolͭComparisonKey
     * @throws	IllegalArgumentException
     *		keyפʤ
     */
    public synchronized void destroy(ComparisonKey key)
    {
	super.destroy(key);
	try {
	    try {
		if (conn != null) {
		    conn.close();
		}
	    } catch (SQLException e) {
		// do nothing
	    }
	} catch (Exception e) {
	    // do nothing
	} finally {
	    command	= null;
	    statements	= null;
	    conn	= null;
	    closedConn	= null;
	    ownRef	= null;
	}
    }

    /**
     * ConnectionWrapper鿷StatementϿޤ
     *
     * @param	stmt
     *		줿Statement(sub class)
     */
    private void addStatement(Statement stmt)
    {
	if (stmt != null) {
	    statements.add(stmt);
	}
    }

    /**
     * ConnectionWrapper줿StatementWrapper(sub class)
     * Ĥޤ
     */
    private void closeStatementsAll()
    {
	Statement stmt;
	for (int i = statements.size(); i-- > 0;) {
	    stmt = (Statement) statements.remove(i);

	    try {
		stmt.close();
	    } catch (SQLException e) {
		// do nothing
	    }
	}
    }


    // accessor ---------------------------------------------------------------
    /**
     * ߤξ֤activeʤ񤷤Ƥͭ³passiveʤĤ
     * Ƥ³ᤷޤ
     *
     * @return	activeʤͭ³passiveʤ̵³
     */
    private Connection connection()
    {
	return (isWorkable()) ? conn : closedConn;
    }


    // action -----------------------------------------------------------------
    /**
     * raw JDBC ConnectionĤƤʤǧ򤷤ޤ
     */
    protected boolean ping()
    {
	boolean isSuccess = false;
	long curr = System.currentTimeMillis();

	if ((curr - lastAccess) < interval) {
	    isSuccess = true;
	} else {
	    Statement stmt = null;

	    try {
		stmt = conn.createStatement();
		stmt.executeQuery(command);
		isSuccess = true;
	    } catch (SQLException e) {
		// do nothing
	    } finally {
		if (stmt != null) {
		    try {
			stmt.close();
		    } catch (Exception e) {
			// do nothing
		    }
		}
	    }
	}

	lastAccess = curr;
	/* System.out.println("ping done."); // for debug */
	return isSuccess;
    }

    
    // Connection(Jdk ver 1.3) implementation ---------------------------------
    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void clearWarnings()
	throws SQLException
    {
	connection().clearWarnings();
    }

    /**
     * Connectionclose򥪡С饤ɤinstanceConnectionPool
     * ޤºݤˤConnectioncloseޤ
     *
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void close()
	throws SQLException
    {
	if (isWorkable()) {
	    release();
	} else {
	    closedConn.close();
	}
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void commit()
	throws SQLException
    {
	connection().commit();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public Statement createStatement()
	throws SQLException
    {
	try {
	    Statement stmt = new StatementWrapper
		(connection().createStatement(), ownRef);
	    addStatement(stmt);
	    return stmt;
	} catch (IllegalArgumentException e) {
	    throw new SQLException(e.toString());
	}
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	resultSetType
     *		Wrap줿JDBC Connectionؤΰ
     * @param	resultSetConcurrency
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public Statement createStatement(int resultSetType,
				     int resultSetConcurrency)
	throws SQLException
    {
	try {
	    Statement stmt = new StatementWrapper
		(connection().createStatement(resultSetType,
					      resultSetConcurrency), ownRef);
	    addStatement(stmt);
	    return stmt;
	} catch (IllegalArgumentException e) {
	    throw new SQLException(e.toString());
	}
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public String getCatalog()
	throws SQLException
    {
	return connection().getCatalog();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public boolean getAutoCommit()
	throws SQLException
    {
	return connection().getAutoCommit();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public DatabaseMetaData getMetaData()
	throws SQLException
    {
	return connection().getMetaData();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public int getTransactionIsolation()
	throws SQLException
    {
	return connection().getTransactionIsolation();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public Map getTypeMap()
	throws SQLException
    {
	return connection().getTypeMap();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public SQLWarning getWarnings()
	throws SQLException
    {
	return connection().getWarnings();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public boolean isClosed()
	throws SQLException
    {
	return (isWorkable()) ? conn.isClosed() : true;
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public boolean isReadOnly()
	throws SQLException
    {
	return connection().isReadOnly();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	sql
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public String nativeSQL(String sql)
	throws SQLException
    {
	return connection().nativeSQL(sql);
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	sql
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public CallableStatement prepareCall(String sql)
	throws SQLException
    {
	CallableStatement cstmt = new CallableStatementWrapper
	    (connection().prepareCall(sql), ownRef);
	addStatement(cstmt);
	return cstmt;
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	sql
     *		Wrap줿JDBC Connectionؤΰ
     * @param	resultSetType
     *		Wrap줿JDBC Connectionؤΰ
     * @param	resultSetConcurrency
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public CallableStatement prepareCall(String sql,
					 int resultSetType,
					 int resultSetConcurrency)
	throws SQLException
    {
	CallableStatement cstmt = new CallableStatementWrapper
	    (connection().prepareCall
		(sql, resultSetType, resultSetConcurrency), ownRef);
	addStatement(cstmt);
	return cstmt;
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	sql
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public PreparedStatement prepareStatement(String sql) 
	throws SQLException
    {
	PreparedStatement pstmt = new PreparedStatementWrapper
	    (connection().prepareStatement(sql), ownRef);
	addStatement(pstmt);
	return pstmt;
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	sql
     *		Wrap줿JDBC Connectionؤΰ
     * @param	resultSetType
     *		Wrap줿JDBC Connectionؤΰ
     * @param	resultSetConcurrency
     *		Wrap줿JDBC Connectionؤΰ
     * @return	Wrap줿JDBC Connection줿
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public PreparedStatement prepareStatement(String sql,
					      int resultSetType,
					      int resultSetConcurrency)
	throws SQLException
    {
	PreparedStatement pstmt  = new PreparedStatementWrapper
	    (connection().prepareStatement
		(sql, resultSetType, resultSetConcurrency), ownRef);
	addStatement(pstmt);
	return pstmt;
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void rollback()
	throws SQLException
    {
	connection().rollback();
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	autoCommit
     *		Wrap줿JDBC Connectionؤΰ
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void setAutoCommit(boolean autoCommit)
	throws SQLException
    {
	connection().setAutoCommit(autoCommit);
	modAutoCommit = defaults.checkAutoCommit(autoCommit);
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	catalog
     *		Wrap줿JDBC Connectionؤΰ
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void setCatalog(String catalog)
	throws SQLException
    {
	connection().setCatalog(catalog);
	modCatalog = defaults.checkCatalog(catalog);
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	readOnly
     *		Wrap줿JDBC Connectionؤΰ
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void setReadOnly(boolean readOnly)
	throws SQLException
    {
	connection().setReadOnly(readOnly);
	modReadOnly = defaults.checkReadOnly(readOnly);
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	level
     *		Wrap줿JDBC Connectionؤΰ
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void setTransactionIsolation(int level)
	throws SQLException
    { 
	connection().setTransactionIsolation(level);
	modIsolation = defaults.checkTransactionIsolation(level);
    }

    /**
     * Wrap줿JDBC ConnectionƱ쥷ͥ᥽åɤ׵žޤ
     *
     * @param	map
     *		Wrap줿JDBC Connectionؤΰ
     * @throws	SQLException
     *		Wrap줿JDBC Connectionthrow
     */
    public void setTypeMap(Map map)
	throws SQLException
    {
	connection().setTypeMap(map);
	modTypeMap = defaults.checkTypeMap(map);
    }


    // helper class -----------------------------------------------------------
    /**
     * JDBC Driver줿ConnectionνͤǼclassǤ
     */
    static class DefaultValues 
    {
	// raw Connection's default values & status ---------------------------
	/** */
	private final boolean autoCommit;

	/** */
	private String catalog;

	/** */
	private final boolean readOnly;
	
	/** */
	private final int isolation;
	

	/** */
	private final Map typeMap;
	

	// constructors & init ------------------------------------------------
	/**
	 * ȤʤConnectionꤷDefaultValuesۤޤ
	 *
	 * @param	conn
	 *		ȤʤConnection
	 * @throws	SQLException
	 *		ȤʤͤǤʤ
	 */
	DefaultValues(Connection conn)
	    throws SQLException
	{
	    autoCommit	= conn.getAutoCommit();
	    catalog	= conn.getCatalog();
	    readOnly	= conn.isReadOnly();
	    isolation	= conn.getTransactionIsolation();
	    typeMap	= conn.getTypeMap();
	}
	

	// action -------------------------------------------------------------
	/**
	 * connγƼѥ᡼ͤᤷޤ
	 * 
	 * @param	conn
	 *		ͤꤹJDBC Connection
	 * @return	ꥻåȤtrue
	 */
	boolean resetValues(Connection conn,
			    boolean modAutoCommit,
			    boolean modCatalog,
			    boolean modReadOnly,
			    boolean modIsolation,
			    boolean modTypeMap)
	{
	    boolean isSuccess = false;
	    try {
		if (!conn.getAutoCommit()) {
		    conn.rollback();
		}

		if (modAutoCommit) {
		    conn.setAutoCommit(autoCommit);
		}
		
		if (modCatalog) {
		    conn.setCatalog(catalog);
		}
		
		if (modReadOnly) {
		    conn.setReadOnly(readOnly);
		}
		
		if (modIsolation) {
		    conn.setTransactionIsolation(isolation);
		}
		
		if (modTypeMap) {
		    conn.setTypeMap(typeMap);
		}
		isSuccess = true;
	    } catch (SQLException e) {
		// please logging this trace!
	    } finally {
		try {
		    conn.clearWarnings();
		} catch (SQLException e) {
		    // do nothing
		}
	    }
	    return isSuccess;
	}

	/**
	 *
	 */
	boolean checkAutoCommit(boolean val)
	{
	    return (val == autoCommit);
	}
	
	/**
	 *
	 */
	boolean checkCatalog(String val)
	{
	    if (val != null) {
		return val.equals(catalog);
	    } else {
		return (catalog == null);
	    }
	}
	
	/**
	 *
	 */
	boolean checkReadOnly(boolean val)
	{
	    return (val == readOnly);
	}
	
	/**
	 *
	 */
	boolean checkTransactionIsolation(int val)
	{
	    return (val == isolation);
	}
	
	/**
	 *
	 */
	boolean checkTypeMap(Map val)
	{
	    if (val != null) {
		return val.equals(typeMap);
	    } else {
		// ꤨʤȤϻפǰΤ
		return (typeMap == null);
	    }
	}


	// over rides ---------------------------------------------------------
	/**
	 *
	 */
	public boolean equals(Object o) 
	{
	    if (o instanceof DefaultValues) {
		DefaultValues vals = (DefaultValues) o;
		
		return (vals.checkAutoCommit(autoCommit) 
			&& vals.checkCatalog(catalog)
			&& vals.checkReadOnly(readOnly)
			&& vals.checkTransactionIsolation(isolation)
			&& vals.checkTypeMap(typeMap));
	    } else {
		return false;
	    }
	}

	/**
	 *
	 */
	public String toString()
	{
	    return new StringBuffer()
		.append("autoCommit = ").append(autoCommit).append("\n")
		.append("catalog = ").append(catalog).append("\n")
		.append("readOnly = ").append(readOnly).append("\n")
		.append("transactionIsolation = ").append(isolation)
		.append("\n")
		.append("typeMap = ").append(typeMap).append("\n")
		.toString();
	}
    }
}
