/*
-------------------------------------------------------------------------------
  delete.c - Delete folder or bookmark
-------------------------------------------------------------------------------
*/


#include <unistd.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "ibb.h"


typedef struct {
  GtkWidget *dialog;
  gchar *fileName;
  GList *selection;
  gint row;
} typeDelete;


void delete_bookmark_ok(GtkWidget *, typeDelete *);
void delete_folder_ok(GtkWidget *, typeDelete *);
gint recursive_rmdir(gchar *);


/*
-------------------------------------------------------------------------------
  Recursively deleting files and folders
-------------------------------------------------------------------------------
*/
gint recursive_rmdir(gchar *dirName)
{
  DIR *dirp;
  struct dirent *entry;
  struct stat statBuf;
  gchar *fileName;
  gchar *newFileName;
  gint result;

  dirp = opendir(dirName);
  if (dirp == NULL) {
    return FALSE;
  }

  while ((entry = readdir(dirp)) != NULL) {
    fileName = entry->d_name;
    newFileName = g_strconcat(dirName, "/", fileName, NULL);

    lstat(newFileName, &statBuf);

    if (!g_strcasecmp(fileName, ".") || !g_strcasecmp(fileName, "..")) {
      continue;
    }

    if (S_ISREG (statBuf.st_mode)) {
      result = unlink(newFileName);
      if (result) {
	return FALSE;
      }
      continue;
    }

    if (S_ISDIR (statBuf.st_mode)) {
      result = recursive_rmdir(newFileName);
      if (!result) {
	return FALSE;
      }
      rmdir(newFileName);
      continue;
    }
  }

  result = rmdir(dirName);
  if (result) {
    return FALSE;
  }
  return TRUE;
}


/*
-------------------------------------------------------------------------------
  Delete folder
-------------------------------------------------------------------------------
*/
void delete_folder_ok(GtkWidget *button, typeDelete *data)
{
  gint result;

  result = recursive_rmdir(data->fileName);
  if(!result) {
    error_dialog("ERROR: Cannot delete!\n");
    return;
  }

  gtk_tree_remove_items(GTK_TREE (treeGlobal), data->selection);
  gtk_clist_clear(GTK_CLIST (clistGlobal));
  gtk_widget_destroy(data->dialog);
  g_free(data);
}


/*
-------------------------------------------------------------------------------
  Delete bookmark
-------------------------------------------------------------------------------
*/
void delete_bookmark_ok(GtkWidget *button, typeDelete *data)
{
  gint result;
  
  result = unlink(data->fileName);
  if(result) {
    error_dialog("ERROR: Cannot delete!\n");
    return;
  }

  gtk_clist_remove(GTK_CLIST (clistGlobal), data->row);
  gtk_widget_destroy(data->dialog);
  g_free(data);
}


/*
-------------------------------------------------------------------------------
  Popup "Deleting Bookmark" dialog
-------------------------------------------------------------------------------
*/
void cb_delete_bookmark(GtkWidget *toolBar, gpointer data)
{
  GtkWidget *dialog;
  GtkWidget *okButton;
  GList *selection;
  gchar *fileTemp;
  gchar *message;
  gint row;

  typeDelete *passData;

  passData = g_malloc(sizeof(typeDelete));
  selection = GTK_CLIST (clistGlobal)->selection;

  if (!selection) {
    error_dialog("ERROR: Select a bookmark!\n");
    return;
  }

  row = GPOINTER_TO_INT (selection->data);
  gtk_clist_get_text(GTK_CLIST (clistGlobal),
		     GPOINTER_TO_INT (selection->data), 0, &fileTemp);
  message = g_strconcat("Deleting [", fileTemp, "]\nAre you sure?", NULL); 
  passData->fileName = g_strconcat(dirNameGlobal, "/", fileTemp,
				  ".url", NULL);

  okButton = gtk_button_new_with_label("Ok");
  dialog = dialog_with_label(okButton, "IBB: Deleting Bookmark", message);
  passData->dialog = dialog;

  passData->row = row;

  gtk_signal_connect(GTK_OBJECT (okButton), "clicked",
		     GTK_SIGNAL_FUNC (delete_bookmark_ok),
		     passData);

  gtk_widget_show(okButton);
  gtk_widget_show(dialog);
}


/*
-------------------------------------------------------------------------------
  Popup "Deleting Folder" dialog
-------------------------------------------------------------------------------
*/
void cb_delete_folder(GtkWidget *toolBar, gpointer data)
{
  GtkWidget *dialog;
  GtkWidget *okButton;
  gchar *fileTemp;
  gchar *message;
  GList *selection;
  typeDelete *passData;

  passData = g_malloc(sizeof(typeDelete));

  selection = GTK_TREE_SELECTION (treeGlobal);
  if (!selection) {
    error_dialog("ERROR: Select a folder!\n");
    return;
  }
  if (GTK_IS_ROOT_TREE (GTK_WIDGET (selection->data)->parent)) {
    error_dialog("ERROR: Cannot delte ROOT folder!\n");
    return;
  }
  passData->selection = selection;

  fileTemp = g_basename(dirNameGlobal);
  message = g_strconcat("Recursively deleting [", fileTemp,
			"]\nAre you sure?", NULL); 
  passData->fileName = g_strdup(dirNameGlobal);

  okButton = gtk_button_new_with_label("Ok");
  dialog = dialog_with_label(okButton, "IBB: Deleting Folder", message);
  passData->dialog = dialog;

  gtk_signal_connect(GTK_OBJECT (okButton), "clicked",
		     GTK_SIGNAL_FUNC (delete_folder_ok),
		     passData);

  gtk_widget_show(okButton);
  gtk_widget_show(dialog);
}
