/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.ui.views;

import java.util.ArrayList;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IWorkbenchActionConstants;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.cat.tm.ui.CatTmUiPlugin;
import benten.cat.tm.ui.actions.CopyTranslationSourceAction;
import benten.cat.tm.ui.actions.CopyTranslationTargetAction;
import benten.cat.tm.ui.views.messages.TranslationViewMessages;
import benten.ui.views.AutoColumnWidthTableViewer;
import benten.ui.views.CopySupportTableView;
import benten.ui.views.TableColumnSortAdapter;
import benten.ui.views.TableLabelProvider;

/**
 * 翻訳ビュー。
 *
 * <UL>
 * <LI>エディターなどで選択された文字列に対し、TM を表示するためのビュー・クラスです。
 * <LI>TM の検索は {@link BentenTmEngine} へ委譲されます。
 * </UL>
 *
 * ★基本設計「翻訳支援機能: 翻訳メモリー検索機能」に対応します。
 *
 * @author YAMAMOTO Koji
 * @author KASHIHARA Shinji
 */
public class TranslationView extends CopySupportTableView {

	/**
	 * 翻訳ビューのためのメッセージ。
	 */
	protected static final TranslationViewMessages fMsg = new TranslationViewMessages();

	/** コピー・ソース・アクション */
	private Action copySourceAction;

	/** コピー・ターゲット・アクション */
	private Action copyTargetAction;

	/**
	 * コンストラクター。
	 */
	public TranslationView() {
	}

	@Override
	public void createTableViewer(final AutoColumnWidthTableViewer viewer) {
		viewer.addColumn(fMsg.getOrigin(), 8).addSelectionListener(new TmOriginSortListener(0));
		viewer.addColumn(fMsg.getQuality(), 5).addSelectionListener(new TmMatchQualitySortListener(1));
		viewer.addColumn(fMsg.getSource(), 20).addSelectionListener(new TmSourceSortListener(2));
		viewer.addColumn(fMsg.getTarget(), 20).addSelectionListener(new TmTargetSortListener(3));

		viewer.setLabelProvider(new ViewLabelProvider());
		viewer.setInput(new ArrayList<BentenTmSearchResult>());

		makeActions();
		createContextMenu();
	}

	/**
	 * コンテキスト・メニューの作成。
	 */
	protected void createContextMenu() {
		final MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(final IMenuManager manager) {
				TranslationView.this.fillContextMenu(manager);
			}
		});
		final Menu menu = menuMgr.createContextMenu(getViewer().getControl());
		getViewer().getControl().setMenu(menu);
		getSite().registerContextMenu(menuMgr, getViewer());
	}

	/**
	 * アクション・バーのコントリビュート。
	 */
	private void fillContextMenu(final IMenuManager manager) {
		final boolean isEmpty = getViewer().getSelection().isEmpty();
		copySourceAction.setEnabled(!isEmpty);
		manager.add(copySourceAction);
		copyTargetAction.setEnabled(!isEmpty);
		manager.add(copyTargetAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	/**
	 * 翻訳ビューの表示。
	 * @param string 対象文字列
	 */
	public void showTranslation(final String string) {
		final CatTmUiPlugin plugin = CatTmUiPlugin.getDefault();
		getViewer().setInput(plugin.fuzzyMatch(string));
	}

	/**
	 * アクションの作成。
	 */
	private void makeActions() {
		copySourceAction = new CopyTranslationSourceAction(this, fMsg.getCopySource());
		copyTargetAction = new CopyTranslationTargetAction(this, fMsg.getCopyTarget());
	}

	/**
	 * ビューのラベル・プロバイダー。
	 */
	private static class ViewLabelProvider extends TableLabelProvider {
		public String getColumnText(final Object obj, final int index) {
			final BentenTmSearchResult entry = (BentenTmSearchResult) obj;
			switch (index) {
			case 0:
				return entry.getOrigin();
			case 1:
				return entry.getMatchQuality();
			case 2:
				return entry.getSource();
			case 3:
				return entry.getTarget();
			}
			return getText(obj);
		}
	}

	//-------------------------------------------------------------------------
	// ソート・リスナー・クラス

	protected class TmOriginSortListener extends TableColumnSortAdapter<BentenTmSearchResult> {
		public TmOriginSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenTmSearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenTmSearchResult>() {
				@Override
				public int compare(final BentenTmSearchResult tm1, final BentenTmSearchResult tm2) {
					return tm1.getOrigin().compareToIgnoreCase(tm2.getOrigin());
				}
			};
		}
	}

	protected class TmMatchQualitySortListener extends TableColumnSortAdapter<BentenTmSearchResult> {
		public TmMatchQualitySortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenTmSearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenTmSearchResult>() {
				@Override
				public int compare(final BentenTmSearchResult tm1, final BentenTmSearchResult tm2) {
					return compareInteger(tm1.getMatchQuality(), tm2.getMatchQuality());
				}
			};
		}
	}

	protected class TmSourceSortListener extends TableColumnSortAdapter<BentenTmSearchResult> {
		public TmSourceSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenTmSearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenTmSearchResult>() {
				@Override
				public int compare(final BentenTmSearchResult tm1, final BentenTmSearchResult tm2) {
					return tm1.getSource().compareToIgnoreCase(tm2.getSource());
				}
			};
		}
	}

	protected class TmTargetSortListener extends TableColumnSortAdapter<BentenTmSearchResult> {
		public TmTargetSortListener(final int columnIndex) {
			super(getViewer(), columnIndex);
		}

		@Override
		public TableColumnSorter<BentenTmSearchResult> createTableColumnSorter() {
			return new TableColumnSorter<BentenTmSearchResult>() {
				@Override
				public int compare(final BentenTmSearchResult tm1, final BentenTmSearchResult tm2) {
					return tm1.getTarget().compareToIgnoreCase(tm2.getTarget());
				}
			};
		}
	}
}
