/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.tmx.core;

import java.io.IOException;

import benten.twa.tmx.core.valueobject.BentenExtractTmxFragmentProcessInput;

/**
 * バッチ処理クラス [BentenExtractTmxFragmentBatchProcess]。
 *
 * <P>バッチ処理の呼び出し例。</P>
 * <code>
 * java -classpath (クラスパス) benten.twa.tmx.core.BentenExtractTmxFragmentBatchProcess -help
 * </code>
 */
public class BentenExtractTmxFragmentBatchProcess {
    /**
     * 正常終了。
     */
    public static final int END_SUCCESS = 0;

    /**
     * 入力異常終了。内部的にjava.lang.IllegalArgumentExceptionが発生した場合。
     */
    public static final int END_ILLEGAL_ARGUMENT_EXCEPTION = 7;

    /**
     * 入出力例外終了。内部的にjava.io.IOExceptionが発生した場合。
     */
    public static final int END_IO_EXCEPTION = 8;

    /**
     * 異常終了。バッチの処理開始に失敗した場合、および内部的にjava.lang.Errorまたはjava.lang.RuntimeExceptionが発生した場合。
     */
    public static final int END_ERROR = 9;

    /**
     * コマンドラインから実行された際のエントリポイントです。
     *
     * @param args コンソールから引き継がれた引数。
     */
    public static final void main(final String[] args) {
        final BentenExtractTmxFragmentBatchProcess batchProcess = new BentenExtractTmxFragmentBatchProcess();

        // バッチ処理の引数。
        final BentenExtractTmxFragmentProcessInput input = new BentenExtractTmxFragmentProcessInput();

        boolean isNeedUsage = false;
        boolean isFieldXliffdirProcessed = false;
        boolean isFieldTmxdirProcessed = false;
        boolean isFieldTranstargetidProcessed = false;

        // コマンドライン引数の解析をおこないます。
        for (int index = 0; index < args.length; index++) {
            String arg = args[index];
            if (arg.startsWith("-verbose=")) {
                input.setVerbose(Boolean.valueOf(arg.substring(9)).booleanValue());
            } else if (arg.startsWith("-xliffdir=")) {
                input.setXliffdir(arg.substring(10));
                isFieldXliffdirProcessed = true;
            } else if (arg.startsWith("-tmxdir=")) {
                input.setTmxdir(arg.substring(8));
                isFieldTmxdirProcessed = true;
            } else if (arg.startsWith("-transtargetid=")) {
                input.setTranstargetid(arg.substring(15));
                isFieldTranstargetidProcessed = true;
            } else if (arg.startsWith("-extractbystate=")) {
                input.setExtractbystate(arg.substring(16));
            } else if (arg.startsWith("-includedate=")) {
                input.setIncludedate(Boolean.valueOf(arg.substring(13)).booleanValue());
            } else if (arg.startsWith("-ignorewhitespacetmxextract=")) {
                input.setIgnorewhitespacetmxextract(Boolean.valueOf(arg.substring(28)).booleanValue());
            } else if (arg.startsWith("-ignoremnemonickeytmxextract=")) {
                input.setIgnoremnemonickeytmxextract(Boolean.valueOf(arg.substring(29)).booleanValue());
            } else if (arg.startsWith("-transsourcelang=")) {
                input.setTranssourcelang(arg.substring(17));
            } else if (arg.equals("-?") || arg.equals("-help")) {
                usage();
                System.exit(END_SUCCESS);
            } else {
                System.out.println("BentenExtractTmxFragmentBatchProcess: 入力パラメータ[" + arg + "]は無視されました。");
                isNeedUsage = true;
            }
        }

        if (isNeedUsage) {
            usage();
        }

        if( isFieldXliffdirProcessed == false) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[xliffdir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldTmxdirProcessed == false) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[tmxdir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldTranstargetidProcessed == false) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[transtargetid]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }

        int retCode = batchProcess.execute(input);

        // 終了コードを戻します。
        // ※注意：System.exit()を呼び出している点に注意してください。
        System.exit(retCode);
    }

    /**
     * 具体的なバッチ処理内容を記述するためのメソッドです。
     *
     * このメソッドに実際の処理内容を記述します。
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IOException 入出力例外が発生した場合。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public int process(final BentenExtractTmxFragmentProcessInput input) throws IOException, IllegalArgumentException {
        // 入力パラメータをチェックします。
        validateInput(input);

        // この箇所でコンパイルエラーが発生する場合、BentenExtractTmxFragmentProcessインタフェースを実装して benten.twa.tmx.coreパッケージに BentenExtractTmxFragmentProcessImplクラスを作成することにより解決できる場合があります。
        final BentenExtractTmxFragmentProcess process = new BentenExtractTmxFragmentProcessImpl();

        // 処理の本体を実行します。
        final int retCode = process.execute(input);

        return retCode;
    }

    /**
     * クラスをインスタンス化してバッチを実行する際のエントリポイントです。
     *
     * このメソッドは下記の仕様を提供します。
     * <ul>
     * <li>メソッドの入力パラメータの内容チェック。
     * <li>IllegalArgumentException, RuntimeException, Errorなどの例外をcatchして戻り値へと変換。
     * </ul>
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public final int execute(final BentenExtractTmxFragmentProcessInput input) throws IllegalArgumentException {
        try {
            // バッチ処理の本体を実行します。
            int retCode = process(input);

            return retCode;
        } catch (IllegalArgumentException ex) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: 入力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_ILLEGAL_ARGUMENT_EXCEPTION;
        } catch (IOException ex) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: 入出力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_IO_EXCEPTION;
        } catch (RuntimeException ex) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: ランタイム例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            ex.printStackTrace();
            // 異常終了。
            return END_ERROR;
        } catch (Error er) {
            System.out.println("BentenExtractTmxFragmentBatchProcess: ランタイムエラーが発生しました。バッチ処理を中断します。:" + er.toString());
            er.printStackTrace();
            // 異常終了。
            return END_ERROR;
        }
    }

    /**
     * このバッチ処理クラスの使い方の説明を標準出力に示すためのメソッドです。
     */
    public static final void usage() {
        System.out.println("BentenExtractTmxFragmentBatchProcess: Usage:");
        System.out.println("  java benten.twa.tmx.core.BentenExtractTmxFragmentBatchProcess -verbose=値1 -xliffdir=値2 -tmxdir=値3 -transtargetid=値4 -extractbystate=値5 -includedate=値6 -ignorewhitespacetmxextract=値7 -ignoremnemonickeytmxextract=値8 -transsourcelang=値9");
        System.out.println("    -verbose");
        System.out.println("      説明[verboseモードで動作させるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -xliffdir");
        System.out.println("      説明[抽出元となる XLIFF ディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -tmxdir");
        System.out.println("      説明[抽出先となる TMX を格納するディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -transtargetid");
        System.out.println("      説明[翻訳対象 id  を格納します。この id はファイル名の一部に利用されます。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -extractbystate");
        System.out.println("      説明[状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。]");
        System.out.println("      型[文字列]");
        System.out.println("    -includedate");
        System.out.println("      説明[出力ファイル名に日付を含めるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -ignorewhitespacetmxextract");
        System.out.println("      説明[ホワイトスペースを無視した内容の TMX を抽出するかどうか。HTML の場合には true を推奨。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -ignoremnemonickeytmxextract");
        System.out.println("      説明[ニーモニック・キーを無視した内容の TMX を抽出するかどうか。日本風のニーモニックのみ対応。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -transsourcelang");
        System.out.println("      説明[翻訳元言語。TMX ヘッダーに使われます。]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[en-US]");
        System.out.println("    -? , -help");
        System.out.println("      説明[使い方を表示します。]");
    }

    /**
     * このバッチ処理クラスの入力パラメータの妥当性チェックを実施するためのメソッドです。
     *
     * @param input バッチ処理の入力パラメータ。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public void validateInput(final BentenExtractTmxFragmentProcessInput input) throws IllegalArgumentException {
        if (input == null) {
            throw new IllegalArgumentException("BlancoBatchProcessBatchProcess: 処理開始失敗。入力パラメータ[input]にnullが与えられました。");
        }
        if (input.getXliffdir() == null) {
            throw new IllegalArgumentException("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[xliffdir]に値が設定されていません。");
        }
        if (input.getTmxdir() == null) {
            throw new IllegalArgumentException("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[tmxdir]に値が設定されていません。");
        }
        if (input.getTranstargetid() == null) {
            throw new IllegalArgumentException("BentenExtractTmxFragmentBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[transtargetid]に値が設定されていません。");
        }
    }
}
