/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.filter.core;

import java.io.IOException;

import benten.twa.filter.core.valueobject.BentenConvertToXliffProcessInput;

/**
 * バッチ処理クラス [BentenConvertToXliffBatchProcess]。
 *
 * <P>バッチ処理の呼び出し例。</P>
 * <code>
 * java -classpath (クラスパス) benten.twa.filter.core.BentenConvertToXliffBatchProcess -help
 * </code>
 */
public class BentenConvertToXliffBatchProcess {
    /**
     * 正常終了。
     */
    public static final int END_SUCCESS = 0;

    /**
     * 入力異常終了。内部的にjava.lang.IllegalArgumentExceptionが発生した場合。
     */
    public static final int END_ILLEGAL_ARGUMENT_EXCEPTION = 7;

    /**
     * 入出力例外終了。内部的にjava.io.IOExceptionが発生した場合。
     */
    public static final int END_IO_EXCEPTION = 8;

    /**
     * 異常終了。バッチの処理開始に失敗した場合、および内部的にjava.lang.Errorまたはjava.lang.RuntimeExceptionが発生した場合。
     */
    public static final int END_ERROR = 9;

    /**
     * コマンドラインから実行された際のエントリポイントです。
     *
     * @param args コンソールから引き継がれた引数。
     */
    public static final void main(final String[] args) {
        final BentenConvertToXliffBatchProcess batchProcess = new BentenConvertToXliffBatchProcess();

        // バッチ処理の引数。
        final BentenConvertToXliffProcessInput input = new BentenConvertToXliffProcessInput();

        boolean isNeedUsage = false;
        boolean isFieldSourcedirProcessed = false;
        boolean isFieldTargetdirProcessed = false;
        boolean isFieldTranstargetidProcessed = false;

        // コマンドライン引数の解析をおこないます。
        for (int index = 0; index < args.length; index++) {
            String arg = args[index];
            if (arg.startsWith("-verbose=")) {
                input.setVerbose(Boolean.valueOf(arg.substring(9)).booleanValue());
            } else if (arg.startsWith("-sourcedir=")) {
                input.setSourcedir(arg.substring(11));
                isFieldSourcedirProcessed = true;
            } else if (arg.startsWith("-targetdir=")) {
                input.setTargetdir(arg.substring(11));
                isFieldTargetdirProcessed = true;
            } else if (arg.startsWith("-transtargetid=")) {
                input.setTranstargetid(arg.substring(15));
                isFieldTranstargetidProcessed = true;
            } else if (arg.startsWith("-transsourcelang=")) {
                input.setTranssourcelang(arg.substring(17));
            } else if (arg.startsWith("-transtargetlang=")) {
                input.setTranstargetlang(arg.substring(17));
            } else if (arg.equals("-?") || arg.equals("-help")) {
                usage();
                System.exit(END_SUCCESS);
            } else {
                System.out.println("BentenConvertToXliffBatchProcess: 入力パラメータ[" + arg + "]は無視されました。");
                isNeedUsage = true;
            }
        }

        if (isNeedUsage) {
            usage();
        }

        if( isFieldSourcedirProcessed == false) {
            System.out.println("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcedir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldTargetdirProcessed == false) {
            System.out.println("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetdir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldTranstargetidProcessed == false) {
            System.out.println("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[transtargetid]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }

        int retCode = batchProcess.execute(input);

        // 終了コードを戻します。
        // ※注意：System.exit()を呼び出している点に注意してください。
        System.exit(retCode);
    }

    /**
     * 具体的なバッチ処理内容を記述するためのメソッドです。
     *
     * このメソッドに実際の処理内容を記述します。
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IOException 入出力例外が発生した場合。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public int process(final BentenConvertToXliffProcessInput input) throws IOException, IllegalArgumentException {
        // 入力パラメータをチェックします。
        validateInput(input);

        // この箇所でコンパイルエラーが発生する場合、BentenConvertToXliffProcessインタフェースを実装して benten.twa.filter.coreパッケージに BentenConvertToXliffProcessImplクラスを作成することにより解決できる場合があります。
        final BentenConvertToXliffProcess process = new BentenConvertToXliffProcessImpl();

        // 処理の本体を実行します。
        final int retCode = process.execute(input);

        return retCode;
    }

    /**
     * クラスをインスタンス化してバッチを実行する際のエントリポイントです。
     *
     * このメソッドは下記の仕様を提供します。
     * <ul>
     * <li>メソッドの入力パラメータの内容チェック。
     * <li>IllegalArgumentException, RuntimeException, Errorなどの例外をcatchして戻り値へと変換。
     * </ul>
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public final int execute(final BentenConvertToXliffProcessInput input) throws IllegalArgumentException {
        try {
            // バッチ処理の本体を実行します。
            int retCode = process(input);

            return retCode;
        } catch (IllegalArgumentException ex) {
            System.out.println("BentenConvertToXliffBatchProcess: 入力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_ILLEGAL_ARGUMENT_EXCEPTION;
        } catch (IOException ex) {
            System.out.println("BentenConvertToXliffBatchProcess: 入出力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_IO_EXCEPTION;
        } catch (RuntimeException ex) {
            System.out.println("BentenConvertToXliffBatchProcess: ランタイム例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            ex.printStackTrace();
            // 異常終了。
            return END_ERROR;
        } catch (Error er) {
            System.out.println("BentenConvertToXliffBatchProcess: ランタイムエラーが発生しました。バッチ処理を中断します。:" + er.toString());
            er.printStackTrace();
            // 異常終了。
            return END_ERROR;
        }
    }

    /**
     * このバッチ処理クラスの使い方の説明を標準出力に示すためのメソッドです。
     */
    public static final void usage() {
        System.out.println("BentenConvertToXliffBatchProcess: Usage:");
        System.out.println("  java benten.twa.filter.core.BentenConvertToXliffBatchProcess -verbose=値1 -sourcedir=値2 -targetdir=値3 -transtargetid=値4 -transsourcelang=値5 -transtargetlang=値6");
        System.out.println("    -verbose");
        System.out.println("      説明[verboseモードで動作させるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -sourcedir");
        System.out.println("      説明[変換元となる HTML が格納されたディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -targetdir");
        System.out.println("      説明[変換先となる XLIFF を格納するディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -transtargetid");
        System.out.println("      説明[翻訳対象 id  を格納します。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -transsourcelang");
        System.out.println("      説明[翻訳元言語]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[en-US]");
        System.out.println("    -transtargetlang");
        System.out.println("      説明[翻訳先言語]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[ja-JP]");
        System.out.println("    -? , -help");
        System.out.println("      説明[使い方を表示します。]");
    }

    /**
     * このバッチ処理クラスの入力パラメータの妥当性チェックを実施するためのメソッドです。
     *
     * @param input バッチ処理の入力パラメータ。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public void validateInput(final BentenConvertToXliffProcessInput input) throws IllegalArgumentException {
        if (input == null) {
            throw new IllegalArgumentException("BlancoBatchProcessBatchProcess: 処理開始失敗。入力パラメータ[input]にnullが与えられました。");
        }
        if (input.getSourcedir() == null) {
            throw new IllegalArgumentException("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcedir]に値が設定されていません。");
        }
        if (input.getTargetdir() == null) {
            throw new IllegalArgumentException("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetdir]に値が設定されていません。");
        }
        if (input.getTranstargetid() == null) {
            throw new IllegalArgumentException("BentenConvertToXliffBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[transtargetid]に値が設定されていません。");
        }
    }
}
