/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.fields;

import java.io.File;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.swt.widgets.Composite;

import benten.core.io.ResourceFile;
import benten.ui.UiPlugin;
import benten.ui.dialogs.AbstractResourceSelectionDialog;
import benten.ui.dialogs.ResourceFileSelectionDialog;

/**
 * リソース・ファイルを選択するためのテキスト・フィールド。
 *
 * <UL>
 * <LI>Eclipse のリソース・ファイルを選択するためのフィールド・クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class ResourceFileTextField extends AbstractBrowseTextField {

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param labelText ラベル・テキスト
	 * @param extensions 拡張子
	 */
	public ResourceFileTextField(final Composite parent, final String labelText, final String... extensions) {
		super(parent, labelText, extensions);
	}

	@Override
	protected void openDialog() {
		final AbstractResourceSelectionDialog dialog = createResourceSelectionDialog();
		final String s = text.getText();
		if (s.length() > 0) {
			final IResource res = getResource(s);
			if (res != null && res.exists()) {
				dialog.setInitialSelections(new Object[] { res });
			}
		}
		dialog.open();
		final Object[] result = dialog.getResult();
		if (result != null && result.length > 0) {
			final IResource resource = (IResource) result[0];
			setResource(resource);
		}
	}

	/**
	 * リソース選択ダイアログの作成。
	 * @return リソース選択ダイアログ
	 */
	protected AbstractResourceSelectionDialog createResourceSelectionDialog() {
		final IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		final ResourceFileSelectionDialog dialog = new ResourceFileSelectionDialog(parent.getShell(), projects);
		dialog.setExtensions(extensions);
		return dialog;
	}

	@Override
	public void setText(final String path) {
		try {
			if (path != null && !path.equals("")) { //$NON-NLS-1$
				text.setText(path.replaceFirst("^/", "")); //$NON-NLS-1$ //$NON-NLS-2$
			} else {
				final String prevDir = previousInputMap.get(labelText);
				text.setText(prevDir == null ? "" : prevDir); //$NON-NLS-1$
			}
		} catch (final Exception e) {
			UiPlugin.getDefault().log(e);
		}
	}

	@Override
	public File getFile() {
		final IResource resource = getResource();
		if (resource != null) {
			final IPath location = resource.getLocation();
			if (location != null) {
				return new File(location.toOSString());
			}
		}
		return null;
	}

	/**
	 * リソースのセット。
	 * @param resource リソース
	 */
	public void setResource(final IResource resource) {
		text.setText(ResourceFile.relativePath(resource));
	}

	/**
	 * リソースの取得。
	 * @return リソース。取得できない場合は null。
	 */
	public IResource getResource() {
		return getResource(getText());
	}

	/**
	 * リソースの取得。
	 * @param relativePath 相対パス
	 * @return リソース。取得できない場合は null。
	 */
	protected IResource getResource(final String relativePath) {
		final String[] segments = relativePath.replace("\\", "/").replaceFirst("^/", "").split("/"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		if (segments.length == 1) {
			return null;
		}
		final String projectName = segments[0];
		final IProject project = getProject(projectName);
		if (project == null) {
			return null;
		}
		final String path = relativePath.replaceFirst(Pattern.quote(projectName) + "/", ""); //$NON-NLS-1$ //$NON-NLS-2$
		return project.getFile(path);
	}

	/**
	 * プロジェクトの取得
	 * @param projectName プロジェクト名
	 * @return プロジェクト
	 */
	protected IProject getProject(final String projectName) {
		try {
			final IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
			return workspaceRoot.getProject(projectName);
		} catch (final Exception e) {
			// パス不正
			return null;
		}
	}
}
