/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.ui.fields;

import org.eclipse.jface.fieldassist.ControlDecoration;
import org.eclipse.jface.fieldassist.FieldDecoration;
import org.eclipse.jface.fieldassist.FieldDecorationRegistry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Text;

import benten.ui.views.UiToolkit;

/**
 * ラベル付きテキスト・フィールド。
 *
 * <UL>
 * <LI>{@link Label} と {@link Text} コントロールを持つフィールド・クラスです。
 * </UL>
 *
 * @author KASHIHARA Shinji
 */
public class TextField {

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	private final UiToolkit toolkit = new UiToolkit();

	/** 親コンポジット */
	protected final Composite parent;

	/** ラベル・テキスト */
	protected final String labelText;

	/** テキスト */
	protected final Text text;

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param labelText ラベル・テキスト
	 */
	public TextField(final Composite parent, final String labelText) {
		final Layout layout = parent.getLayout();
		if (layout instanceof GridLayout) {
			final GridLayout gd = (GridLayout) layout;
			gd.horizontalSpacing = 10;
		}
		this.parent = parent;

		this.labelText = labelText;

		final String labelString = (labelText == null || labelText.equals("")) ? "" : labelText + ":"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		toolkit.createLabel(parent, labelString);

		final GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.widthHint = 450;
		text = new Text(parent, SWT.SINGLE | SWT.BORDER);
		text.setLayoutData(gd);
	}

	/**
	 * 横幅ヒントのセット。
	 * @param widthHint 横幅ヒント
	 */
	public void setWidthHint(final int widthHint) {
		final GridData gd = (GridData) text.getLayoutData();
		gd.widthHint = widthHint;
	}

	/**
	 * 変更リスナーの追加。
	 * @param listener リスナー
	 */
	public void addModifyListener(final ModifyListener listener) {
		text.addModifyListener(listener);
	}

	/**
	 * テキストのセット。
	 * @param value 値
	 */
	public void setText(final String value) {
		text.setText(value == null ? "" : value); //$NON-NLS-1$
	}

	/**
	 * テキストの取得。
	 * @return テキスト
	 */
	public String getText() {
		return text.getText() == null ? "" : text.getText().trim(); //$NON-NLS-1$
	}

	/**
	 * ラベル・テキストの取得・
	 * @return ラベル・テキスト
	 */
	public String getLabelText() {
		return labelText;
	}

	/**
	 * 情報装飾のセット。
	 * @param descriptionText 説明テキスト
	 */
	public void setInfoDecoration(final String descriptionText) {
		final ControlDecoration decoration = new ControlDecoration(text, SWT.RIGHT | SWT.BOTTOM);
		decoration.setDescriptionText(descriptionText);
		final FieldDecoration fieldDecoration = FieldDecorationRegistry.getDefault().getFieldDecoration(
				FieldDecorationRegistry.DEC_INFORMATION);
		decoration.setImage(fieldDecoration.getImage());
		decoration.show();
	}

	/**
	 * 使用可能かセット。
	 * @param enabled 使用可能にする場合は true
	 */
	public void setEnabled(final boolean enabled) {
		text.setEnabled(enabled);
	}

	/**
	 * 使用可能か取得。
	 * @return 使用可能な場合は true
	 */
	public boolean getEnabled() {
		return text.getEnabled();
	}
}
