/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.cat.core;

import java.io.IOException;

import benten.twa.cat.core.valueobject.BentenApplyExistentTranslationProcessInput;

/**
 * バッチ処理クラス [BentenApplyExistentTranslationBatchProcess]。
 *
 * <P>バッチ処理の呼び出し例。</P>
 * <code>
 * java -classpath (クラスパス) benten.twa.cat.core.BentenApplyExistentTranslationBatchProcess -help
 * </code>
 */
public class BentenApplyExistentTranslationBatchProcess {
    /**
     * 正常終了。
     */
    public static final int END_SUCCESS = 0;

    /**
     * 入力異常終了。内部的にjava.lang.IllegalArgumentExceptionが発生した場合。
     */
    public static final int END_ILLEGAL_ARGUMENT_EXCEPTION = 7;

    /**
     * 入出力例外終了。内部的にjava.io.IOExceptionが発生した場合。
     */
    public static final int END_IO_EXCEPTION = 8;

    /**
     * 異常終了。バッチの処理開始に失敗した場合、および内部的にjava.lang.Errorまたはjava.lang.RuntimeExceptionが発生した場合。
     */
    public static final int END_ERROR = 9;

    /**
     * コマンドラインから実行された際のエントリポイントです。
     *
     * @param args コンソールから引き継がれた引数。
     */
    public static final void main(final String[] args) {
        final BentenApplyExistentTranslationBatchProcess batchProcess = new BentenApplyExistentTranslationBatchProcess();

        // バッチ処理の引数。
        final BentenApplyExistentTranslationProcessInput input = new BentenApplyExistentTranslationProcessInput();

        boolean isNeedUsage = false;
        boolean isFieldTargetdirProcessed = false;
        boolean isFieldSourcedirProcessed = false;

        // コマンドライン引数の解析をおこないます。
        for (int index = 0; index < args.length; index++) {
            String arg = args[index];
            if (arg.startsWith("-verbose=")) {
                input.setVerbose(Boolean.valueOf(arg.substring(9)).booleanValue());
            } else if (arg.startsWith("-targetdir=")) {
                input.setTargetdir(arg.substring(11));
                isFieldTargetdirProcessed = true;
            } else if (arg.startsWith("-sourcedir=")) {
                input.setSourcedir(arg.substring(11));
                isFieldSourcedirProcessed = true;
            } else if (arg.startsWith("-transsourcelang=")) {
                input.setTranssourcelang(arg.substring(17));
            } else if (arg.startsWith("-transtargetlang=")) {
                input.setTranstargetlang(arg.substring(17));
            } else if (arg.startsWith("-tmdriverclassname=")) {
                input.setTmdriverclassname(arg.substring(19));
            } else if (arg.startsWith("-ignorewhitespacetmreference=")) {
                input.setIgnorewhitespacetmreference(Boolean.valueOf(arg.substring(29)).booleanValue());
            } else if (arg.startsWith("-ignoremnemonickeytmreference=")) {
                input.setIgnoremnemonickeytmreference(Boolean.valueOf(arg.substring(30)).booleanValue());
            } else if (arg.startsWith("-tmpdir=")) {
                input.setTmpdir(arg.substring(8));
            } else if (arg.equals("-?") || arg.equals("-help")) {
                usage();
                System.exit(END_SUCCESS);
            } else {
                System.out.println("BentenApplyExistentTranslationBatchProcess: 入力パラメータ[" + arg + "]は無視されました。");
                isNeedUsage = true;
            }
        }

        if (isNeedUsage) {
            usage();
        }

        if( isFieldTargetdirProcessed == false) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetdir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }
        if( isFieldSourcedirProcessed == false) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcedir]に値が設定されていません。");
            System.exit(END_ILLEGAL_ARGUMENT_EXCEPTION);
        }

        int retCode = batchProcess.execute(input);

        // 終了コードを戻します。
        // ※注意：System.exit()を呼び出している点に注意してください。
        System.exit(retCode);
    }

    /**
     * 具体的なバッチ処理内容を記述するためのメソッドです。
     *
     * このメソッドに実際の処理内容を記述します。
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IOException 入出力例外が発生した場合。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public int process(final BentenApplyExistentTranslationProcessInput input) throws IOException, IllegalArgumentException {
        // 入力パラメータをチェックします。
        validateInput(input);

        // この箇所でコンパイルエラーが発生する場合、BentenApplyExistentTranslationProcessインタフェースを実装して benten.twa.cat.coreパッケージに BentenApplyExistentTranslationProcessImplクラスを作成することにより解決できる場合があります。
        final BentenApplyExistentTranslationProcess process = new BentenApplyExistentTranslationProcessImpl();

        // 処理の本体を実行します。
        final int retCode = process.execute(input);

        return retCode;
    }

    /**
     * クラスをインスタンス化してバッチを実行する際のエントリポイントです。
     *
     * このメソッドは下記の仕様を提供します。
     * <ul>
     * <li>メソッドの入力パラメータの内容チェック。
     * <li>IllegalArgumentException, RuntimeException, Errorなどの例外をcatchして戻り値へと変換。
     * </ul>
     *
     * @param input バッチ処理の入力パラメータ。
     * @return バッチ処理の終了コード。END_SUCCESS, END_ILLEGAL_ARGUMENT_EXCEPTION, END_IO_EXCEPTION, END_ERROR のいずれかの値を戻します。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public final int execute(final BentenApplyExistentTranslationProcessInput input) throws IllegalArgumentException {
        try {
            // バッチ処理の本体を実行します。
            int retCode = process(input);

            return retCode;
        } catch (IllegalArgumentException ex) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: 入力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_ILLEGAL_ARGUMENT_EXCEPTION;
        } catch (IOException ex) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: 入出力例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            // 入力異常終了。
            return END_IO_EXCEPTION;
        } catch (RuntimeException ex) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: ランタイム例外が発生しました。バッチ処理を中断します。:" + ex.toString());
            ex.printStackTrace();
            // 異常終了。
            return END_ERROR;
        } catch (Error er) {
            System.out.println("BentenApplyExistentTranslationBatchProcess: ランタイムエラーが発生しました。バッチ処理を中断します。:" + er.toString());
            er.printStackTrace();
            // 異常終了。
            return END_ERROR;
        }
    }

    /**
     * このバッチ処理クラスの使い方の説明を標準出力に示すためのメソッドです。
     */
    public static final void usage() {
        System.out.println("BentenApplyExistentTranslationBatchProcess: Usage:");
        System.out.println("  java benten.twa.cat.core.BentenApplyExistentTranslationBatchProcess -verbose=値1 -targetdir=値2 -sourcedir=値3 -transsourcelang=値4 -transtargetlang=値5 -tmdriverclassname=値6 -ignorewhitespacetmreference=値7 -ignoremnemonickeytmreference=値8 -tmpdir=値9");
        System.out.println("    -verbose");
        System.out.println("      説明[verboseモードで動作させるかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -targetdir");
        System.out.println("      説明[適用先 XLIFF が格納されているディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -sourcedir");
        System.out.println("      説明[適用元(過去訳) XLIFF が格納されているディレクトリ。]");
        System.out.println("      型[文字列]");
        System.out.println("      必須パラメータ");
        System.out.println("    -transsourcelang");
        System.out.println("      説明[翻訳元言語]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[en-US]");
        System.out.println("    -transtargetlang");
        System.out.println("      説明[翻訳先言語]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[ja-JP]");
        System.out.println("    -tmdriverclassname");
        System.out.println("      説明[Ant タスクの場合には、TM ドライバーのクラス名を指定します。]");
        System.out.println("      型[文字列]");
        System.out.println("    -ignorewhitespacetmreference");
        System.out.println("      説明[TMX を検索する際にホワイトスペースを無視するかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -ignoremnemonickeytmreference");
        System.out.println("      説明[TMX を検索する際にニーモニックを無視するかどうか。]");
        System.out.println("      型[真偽]");
        System.out.println("      デフォルト値[false]");
        System.out.println("    -tmpdir");
        System.out.println("      説明[テンポラリ・ディレクトリ]");
        System.out.println("      型[文字列]");
        System.out.println("      デフォルト値[\"./tmp\"]");
        System.out.println("    -? , -help");
        System.out.println("      説明[使い方を表示します。]");
    }

    /**
     * このバッチ処理クラスの入力パラメータの妥当性チェックを実施するためのメソッドです。
     *
     * @param input バッチ処理の入力パラメータ。
     * @throws IllegalArgumentException 入力値に不正が見つかった場合。
     */
    public void validateInput(final BentenApplyExistentTranslationProcessInput input) throws IllegalArgumentException {
        if (input == null) {
            throw new IllegalArgumentException("BlancoBatchProcessBatchProcess: 処理開始失敗。入力パラメータ[input]にnullが与えられました。");
        }
        if (input.getTargetdir() == null) {
            throw new IllegalArgumentException("BentenApplyExistentTranslationBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[targetdir]に値が設定されていません。");
        }
        if (input.getSourcedir() == null) {
            throw new IllegalArgumentException("BentenApplyExistentTranslationBatchProcess: 処理開始失敗。入力パラメータ[input]の必須フィールド値[sourcedir]に値が設定されていません。");
        }
    }
}
