/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;

import benten.cat.tm.core.BentenTmDriverManager;
import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.ui.CatTmUiPlugin;
import benten.core.BentenConstants;
import benten.core.text.Strings;
import benten.twa.cat.core.valueobject.BentenApplyExactMatchProcessInput;
import benten.twa.cat.messages.BentenApplyExactMatchMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.BlancoXliffParser;
import blanco.xliff.BlancoXliffSerializer;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * 完全一致訳の適用
 *
 * <pre>
 * TMX の完全一致訳を XLIFF に適用します。
 *    1.  未翻訳の翻訳対象に対して訳の適用を試みます。
 *    2.  翻訳単位と完全一致する TMX を探し、翻訳単位に適用します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式機械支援翻訳機能: 完全一致訳適用機能」に対応します。
 *
 * @author IGA Tosiki
 */
public class BentenApplyExactMatchProcessImpl extends AbstractTraverseDir implements BentenApplyExactMatchProcess {
	/**
	 * 完全一致訳適用機能のためのメッセージ。
	 */
	protected static final BentenApplyExactMatchMessages fMsg = new BentenApplyExactMatchMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenApplyExactMatchProcessInput fInput;

	/**
	 * この処理の実行結果情報。
	 */
	protected final BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この翻訳で利用する翻訳メモリー。
	 */
	protected BentenTmEngine fTm;

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenApplyExactMatchProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenApplyExactMatchProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenApplyExactMatchProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		File dirTmx = null;
		if (fInput.getTmxdir() != null && fInput.getTmxdir().length() > 0) {
			dirTmx = new File(fInput.getTmxdir());
			if (dirTmx.exists() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE006(fInput.getTmxdir()));
			}
			if (dirTmx.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE007(fInput.getTmxdir()));
			}
		}

		final File dirXliff = new File(fInput.getXliffdir());
		if (dirXliff.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getXliffdir()));
		}
		if (dirXliff.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getXliffdir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenApplyExactMatchProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) {
				fCounter += 2;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
			}

			/**
			 * カウント数の取得。
			 * @return カウント数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(fInput);
		inner.processDir(new File(input.getXliffdir()));
		beginTask(inner.getCounter() + 4);

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		if (fInput.getTmdriverclassname() == null || fInput.getTmdriverclassname().trim().length() == 0) {
			// プラグインの場合のコース。
			fTm = CatTmUiPlugin.getDefault().getTmEngine();
		} else {
			// Ant タスクの場合のコース。
			try {
				Class.forName(fInput.getTmdriverclassname());
			} catch (final ClassNotFoundException e) {
				throw new IOException(fMsg.getCoreE001(fInput.getTmdriverclassname(), e.toString()));
			}
			if (BentenTmDriverManager.getDrivers().length == 0) {
				throw new IllegalArgumentException(fMsg.getCoreE002());
			}
			fTm = BentenTmDriverManager.getDrivers()[0].getEngineInstance();
		}
		if (fTm == null) {
			throw new IOException(fMsg.getCoreE003());
		}

		if (dirTmx != null) {
			if (progress(fMsg.getCoreP003())) {
				return 6;
			}

			fTm.loadTmx(dirTmx);
		}

		if (progress(fMsg.getCoreP004())) {
			return 6;
		}

		processDir(dirXliff);

		if (progress(fMsg.getCoreP005(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * 処理の内部処理でアイテムが処理されるたびに進捗報告としてコールバック。
	 *
	 * @param argProgressMessage 現在処理しているアイテムに関するメッセージ。
	 * @return 処理をそのまま継続する場合は false。処理中断をリクエストしたい場合は true。
	 */
	public boolean progressInner(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_XLIFF);
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenApplyExactMatchProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		final BlancoXliffParser parser = new BlancoXliffParser();

		BlancoXliff xliff = null;
		try {
			xliff = parser.parse(file);
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE011(file.getName(), e.toString()));
		}

		int transUnitTotalCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			transUnitTotalCount += xliffFile.getBody().getTransUnitList().size();
		}

		int transUnitCount = 0;
		for (final BlancoXliffFile xliffFile : xliff.getFileList()) {
			for (final BlancoXliffTransUnit transUnit : xliffFile.getBody().getTransUnitList()) {
				if (progressInner(fMsg.getCoreP101(file.getName(), BigDecimal.valueOf(++transUnitCount), BigDecimal
						.valueOf(transUnitTotalCount)))) {
					break;
				}

				if (transUnit.getTranslate() == false) {
					continue;
				}
				if (transUnit.getSource() == null || transUnit.getSource().length() == 0) {
					continue;
				}
				if (transUnit.getTarget() != null && transUnit.getTarget().getTarget() != null
						&& transUnit.getTarget().getTarget().length() >= 0) {
					// target は既に存在しています。

					if (transUnit.getTarget().getState() != null
							&& transUnit.getTarget().getState().trim().length() > 0) {
						// state が設定されています。これは既に翻訳確定済みと判断します。
					} else {
						if (transUnit.getSource().equals(transUnit.getTarget().getTarget())) {
							// source と target とが一致しています。

							// state が無指定または空で、しかも source と target とが一致している場合にのみ、訳を上書きします。
							processTransUnit(transUnit);
							continue;
						}
					}

					// target がすでに存在する場合、訳を上書きしてしまうことを避けるために、
					// ここで中断します。state のレベルによる詳細な判断は実施しません。
					continue;
				}

				// trans-unit を処理します。
				processTransUnit(transUnit);
				continue;
			}
		}

		try {
			final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
			new BlancoXliffSerializer().serialize(xliff, outStream);
			outStream.flush();

			if (BlancoFileUtil.bytes2FileIfNecessary(outStream.toByteArray(), file) == 0) {
				// 変更無し。
				getResultInfo().setSkipCount(getResultInfo().getSkipCount() + 1);
			} else {
				// 更新のあったファイル数をカウントします。
				getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
			}
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE012(file.getName(), e.toString()));
		}

		if (progress(fMsg.getCoreP013(file.getName()))) {
			return;
		}
	}

	/**
	 * trans-unit の完全一致訳を処理。
	 *
	 * <UL>
	 * <LI>設定にかかわらず、最初に指定の文字で完全一致訳の検索を実施します。
	 * <LI>検索で見つからなかった場合に、ホワイトスペース無視の指定がある場合にはホワイトスペースを無視した完全一致訳の検索を実施します。
	 * <LI>ホワイトスペース無視の場合の 2 パス検索は、完全一致訳導出のための特殊な動きです。類似訳の場合には考慮しません。
	 * </UL>
	 *
	 * 前提条件
	 * <UL>
	 * <LI>execute または setInput が事前に呼び出されていること。
	 * </UL>
	 *
	 * @param transUnit 翻訳単位。
	 */
	protected void processTransUnit(final BlancoXliffTransUnit transUnit) {
		if (processTransUnitInternal(transUnit, false, false)) {
			return;
		}

		if (fInput.getIgnorewhitespacetmreference()) {
			// ホワイトスペース無視の設定がある場合には、ホワイトスペース無視で再度検索を実施します。
			processTransUnitInternal(transUnit, true, false);
		}

		if (fInput.getIgnoremnemonickeytmreference()) {
			// ニーモニック・キー無視の設定がある場合には、ニーモニック・キー無視で再度検索を実施します。
			processTransUnitInternal(transUnit, false, true);
		}
	}

	/**
	 * trans-unit の完全一致訳の内部処理。
	 *
	 * <UL>
	 * <LI>fuzzySearch に与える文字列をプリファランスの設定に従って加工するかどうかの判断は引数によって与えられます。
	 * </UL>
	 *
	 * @param transUnit 翻訳単位。
	 * @param isIgnorewhitespacetmreference ホワイトスペースを無視するかどうか。
	 * @param isIgnoreMnemonicKeyTmReference ニーモニック・キーを無視するかどうか。
	 * @return 処理したかどうか。
	 */
	protected boolean processTransUnitInternal(final BlancoXliffTransUnit transUnit,
			final boolean isIgnorewhitespacetmreference, final boolean isIgnoreMnemonicKeyTmReference) {
		String source = transUnit.getSource();
		if (isIgnorewhitespacetmreference) {
			source = Strings.removeRedundantWhitespace(source);
		}

		if (isIgnoreMnemonicKeyTmReference) {
			source = Strings.stripMnemonicKey(source);
		}

		final String targetString = BentenTmUtil.searchExactMatch(fTm, source);
		if (targetString == null) {
			// 完全一致訳はありませんでした。
			return false;
		}

		// 完全一致訳が見つかりました。

		fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

		final BlancoXliffTarget target = new BlancoXliffTarget();
		transUnit.setTarget(target);

		// 完全一致訳をセットします。
		target.setTarget(targetString);

		if (isIgnoreMnemonicKeyTmReference) {
			// ニーモニック・キー部分を追加。
			final String strMnemonicKey = transUnit.getSource().substring(source.length());
			target.setTarget(target.getTarget() + strMnemonicKey);
		}

		// 完全一致を target にセットする場合には、state に translated をセットとします。
		target.setState("translated"); //$NON-NLS-1$
		return true;
	}
}
