/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.preferences.AbstractPreferenceInitializer;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import benten.cat.glossary.ui.CatGlossaryUiPlugin;
import benten.cat.tm.ui.CatTmUiPlugin;
import benten.cat.ui.CatUiPlugin.CatUiPreference;
import benten.cat.ui.messages.CatPreferencePageMessages;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.fields.ControlUtil;
import benten.ui.preference.BentenPreference;
import benten.ui.views.UiToolkit;

/**
 * CAT プラグイン設定ページ。
 * 
 * @author KASHIHARA Shinji
 */
public class CatPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {
	/**
	 * CAT プラグイン設定ページのためのメッセージ。
	 */
	protected static final CatPreferencePageMessages fMsg = new CatPreferencePageMessages();

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	private final UiToolkit toolkit = new UiToolkit();

	// 画面項目。

	/** 翻訳元の言語・コンボボックス */
	private Combo transSourceLangCombo;

	/** 翻訳先の言語・コンボボックス */
	private Combo transTargetLangCombo;

	/** 翻訳の基軸言語・コンボボックス */
	private Combo transPivotLangCombo;

	/**
	 * 「改行、タブ、半角空白を可視化」ボタン。
	 */
	private Button showLineBreakCheck;

	@Override
	protected Control createContents(final Composite parent) {
		final Composite composite = toolkit.createComposite(parent, 1);

		final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
		final IPreferenceStore bentenStore = BentenPreference.getStore();

		{
			final Group transGroup = new Group(composite, SWT.BOLD);
			transGroup.setLayout(toolkit.createGridLayout(3, 5));
			transGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			transGroup.setText(fMsg.getLabelGroupTransLang());

			{
				toolkit.createLabel(transGroup, fMsg.getLabelSourceLang());

				String lang = bentenStore.getString(BentenPreference.Preference.TRANS_SOURCE_LANG.name());
				if (lang == null || lang.length() == 0) {
					lang = fMsg.getDefaultTransSourceLang();
				}

				transSourceLangCombo = new Combo(transGroup, SWT.READ_ONLY);
				for (String locale : getLocaleList()) {
					transSourceLangCombo.add(locale);
				}
				transSourceLangCombo.setText(lang);

				final Button button = toolkit.createButton(transGroup, fMsg.getButtonReverse(), SWT.DEFAULT);
				button.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(final SelectionEvent e) {
						final String origSource = transSourceLangCombo.getText();
						transSourceLangCombo.setText(transTargetLangCombo.getText());
						transTargetLangCombo.setText(origSource);
					}
				});
			}

			{
				toolkit.createLabel(transGroup, fMsg.getLabelTargetLang());

				String lang = bentenStore.getString(BentenPreference.Preference.TRANS_TARGET_LANG.name());
				if (lang == null || lang.length() == 0) {
					lang = fMsg.getDefaultTransTargetLang();
				}

				transTargetLangCombo = new Combo(transGroup, SWT.READ_ONLY);
				for (String locale : getLocaleList()) {
					transTargetLangCombo.add(locale);
				}
				transTargetLangCombo.setText(lang);

				toolkit.createLabel(transGroup, ""); //$NON-NLS-1$
			}

			{
				toolkit.createLabel(transGroup, fMsg.getLabelPivotLang());

				String pivotLang = bentenStore.getString(BentenPreference.Preference.TRANS_PIVOT_LANG.name());
				if (pivotLang == null || pivotLang.length() == 0) {
					pivotLang = fMsg.getDefaultTransPivotLang();
				}

				transPivotLangCombo = new Combo(transGroup, SWT.READ_ONLY);
				for (String locale : getLocaleList()) {
					transPivotLangCombo.add(locale);
				}
				transPivotLangCombo.setText(pivotLang);
				transPivotLangCombo.setToolTipText(fMsg.getLabelPivotLangTooltip());
				ControlUtil.setInfoDecoration(transPivotLangCombo, fMsg.getLabelPivotLangTooltip());
			}
		}

		{
			final Group g = new Group(composite, SWT.BOLD);
			g.setLayout(toolkit.createGridLayout(1, 5));
			g.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			g.setText(fMsg.getLabel1());

			showLineBreakCheck = new Button(g, SWT.CHECK);
			showLineBreakCheck.setText(fMsg.getLabel2());
			showLineBreakCheck.setSelection(store.getBoolean(CatUiPreference.SHOW_LINE_BREAK.name()));
		}

		return composite;
	}

	/**
	 * {@inheritDoc}
	 */
	public void init(final IWorkbench workbench) {
	}

	@Override
	public boolean performOk() {
		if (transSourceLangCombo.getText().equals(transTargetLangCombo.getText())) {
			final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError0());
			UiPlugin.openDialog(fMsg.getDlgErrorLangTitle(), s);
			return false;
		}
		if (transPivotLangCombo.getText().equals(transSourceLangCombo.getText()) == false
				&& transPivotLangCombo.getText().equals(transTargetLangCombo.getText()) == false) {
			final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError1());
			UiPlugin.openDialog(fMsg.getDlgErrorLangTitle(), s);
			return false;
		}

		boolean isLangChanged = false;

		final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
		final IPreferenceStore bentenStore = BentenPreference.getStore();

		if (false == bentenStore.getString(BentenPreference.Preference.TRANS_SOURCE_LANG.name()).equals(
				transSourceLangCombo.getText())) {
			isLangChanged = true;
			bentenStore.setValue(BentenPreference.Preference.TRANS_SOURCE_LANG.name(), transSourceLangCombo.getText());
		}
		if (false == bentenStore.getString(BentenPreference.Preference.TRANS_TARGET_LANG.name()).equals(
				transTargetLangCombo.getText())) {
			isLangChanged = true;
			bentenStore.setValue(BentenPreference.Preference.TRANS_TARGET_LANG.name(), transTargetLangCombo.getText());
		}
		if (false == bentenStore.getString(BentenPreference.Preference.TRANS_PIVOT_LANG.name()).equals(
				transPivotLangCombo.getText())) {
			isLangChanged = true;
			bentenStore.setValue(BentenPreference.Preference.TRANS_PIVOT_LANG.name(), transPivotLangCombo.getText());
		}

		store.setValue(CatUiPreference.SHOW_LINE_BREAK.name(), showLineBreakCheck.getSelection());

		CatUiPlugin.getDefault().firePreferenceChange();

		// 設定変更の影響がある TM および用語集をリロードします。

		if (isLangChanged) {
			// 影響がある場合にのみリロード。
			try {
				CatTmUiPlugin.getDefault().loadExtensions();
			} catch (final IllegalArgumentException e) {
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError2(e.getMessage()));
				UiPlugin.openDialog(fMsg.getDlgErrorLangTitle(), s);
				return false;
			}

			try {
				CatGlossaryUiPlugin.getDefault().loadExtensions();
			} catch (final IllegalArgumentException e) {
				final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getDlgError3(e.getMessage()));
				UiPlugin.openDialog(fMsg.getDlgErrorLangTitle(), s);
				return false;
			}
		}

		return true;
	}

	@Override
	protected void performDefaults() {
		final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();

		transPivotLangCombo.setText(fMsg.getDefaultTransPivotLang());
		transSourceLangCombo.setText(fMsg.getDefaultTransSourceLang());
		transTargetLangCombo.setText(fMsg.getDefaultTransTargetLang());

		showLineBreakCheck.setSelection(store.getDefaultBoolean(CatUiPreference.SHOW_LINE_BREAK.name()));
	}

	/**
	 * 設定初期化クラス。
	 */
	public static class Initializer extends AbstractPreferenceInitializer {
		@Override
		public void initializeDefaultPreferences() {
			final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();

			store.setDefault(CatUiPreference.SHOW_LINE_BREAK.name(), true);
		}
	}

	/**
	 * ロケール一覧を取得します。
	 * 
	 * @return ロケールの一覧。
	 */
	List<String> getLocaleList() {
		final List<String> localeList = new LinkedList<String>();

		// 一覧を取得。
		final Locale[] availableLocales = Locale.getAvailableLocales();
		for (Locale loc : availableLocales) {
			final String language = loc.getLanguage();
			final String country = loc.getCountry();
			if (country == null || country.trim().length() == 0) {
				// 国が指定されていないものはパス。
				continue;
			}

			final String localeString = language + "-" + country;

			boolean isAlreadyExist = false;
			for (String look : localeList) {
				if (localeString.equals(look)) {
					isAlreadyExist = true;
					break;
				}
			}
			if (isAlreadyExist) {
				// すでに登録されているものはパス。
				continue;
			}

			localeList.add(localeString);
		}

		// 名称順でソート。
		Collections.sort(localeList);

		return localeList;
	}
}
