/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.filter.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.filter.core.valueobject.BentenExportHtmlProcessInput;

/**
 * Apache Antタスク [BentenExportHtml]のクラス。
 *
 * HTML エクスポートのための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.filter.core]にBentenExportHtmlBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenexporthtml&quot; classname=&quot;benten.twa.filter.core.BentenExportHtmlTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenExportHtmlTask extends Task {
    /**
     * HTML エクスポートのための Ant タスク。
     */
    protected BentenExportHtmlProcessInput fInput = new BentenExportHtmlProcessInput();

    /**
     * フィールド [sourcexliffdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcexliffdirProcessed = false;

    /**
     * フィールド [sourcehtmldir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcehtmldirProcessed = false;

    /**
     * フィールド [targetdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetdirProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcexliffdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * エクスポート元となる XLIFF が格納されたディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcexliffdir(final String arg) {
        fInput.setSourcexliffdir(arg);
        fIsFieldSourcexliffdirProcessed = true;
    }

    /**
     * Antタスクの[sourcexliffdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * エクスポート元となる XLIFF が格納されたディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcexliffdir() {
        return fInput.getSourcexliffdir();
    }

    /**
     * Antタスクの[sourcehtmldir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * エクスポート元となる HTML が格納されたディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcehtmldir(final String arg) {
        fInput.setSourcehtmldir(arg);
        fIsFieldSourcehtmldirProcessed = true;
    }

    /**
     * Antタスクの[sourcehtmldir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * エクスポート元となる HTML が格納されたディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcehtmldir() {
        return fInput.getSourcehtmldir();
    }

    /**
     * Antタスクの[targetdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * エクスポート先となる HTML を格納するディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetdir(final String arg) {
        fInput.setTargetdir(arg);
        fIsFieldTargetdirProcessed = true;
    }

    /**
     * Antタスクの[targetdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * エクスポート先となる HTML を格納するディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetdir() {
        return fInput.getTargetdir();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenExportHtmlTask begin.");

        // 項目番号[1]、アトリビュート[sourcexliffdir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcexliffdirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcexliffdir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[sourcehtmldir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcehtmldirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcehtmldir]が設定されていません。処理を中断します。");
        }
        // 項目番号[3]、アトリビュート[targetdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcexliffdir:[" + getSourcexliffdir() + "]");
            System.out.println("- sourcehtmldir:[" + getSourcehtmldir() + "]");
            System.out.println("- targetdir:[" + getTargetdir() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenExportHtmlProcessインタフェースを実装して benten.twa.filter.coreパッケージに BentenExportHtmlProcessImplクラスを作成することにより解決できる場合があります。
            final BentenExportHtmlProcess proc = new BentenExportHtmlProcessImpl();
            if (proc.execute(fInput) != BentenExportHtmlBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
