/*******************************************************************************
 * Copyright (c) 2009-2010 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;

import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.twa.filter.model.HtmlInlineTextBuilder;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.filter.model.HtmlInlineTextBuilder.InlineText;
import benten.twa.process.BentenProcessResultInfo;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * HTML を処理します。
 * @author IGA Tosiki
 */
class BentenTwaFilterHtmlProcessor implements BentenTwaFilterProcessor {
	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".html") || file.getName().toLowerCase().endsWith(".htm")) { //$NON-NLS-1$ //$NON-NLS-2$
			return BentenTwaFilterHtmlUtil.isHtmlTagExist(file);
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliffFromSource(final File sourceFile, final BlancoXliffFile xliffFile,
			final HelpTransUnitId id, final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffBody body = xliffFile.getBody();

		// HTML 依存箇所。
		xliffFile.setDatatype("html"); //$NON-NLS-1$

		// HTML 固有箇所。
		for (final InlineText inline : new HtmlInlineTextBuilder(sourceFile)) {
			final String text = inline.getText();
			final SentencePartitionList sentenceList = new SentencePartitionList(text);
			if (sentenceList.size() > 1) {
				for (final String sentence : sentenceList) {
					id.incrementSubSeq(sentenceList.size());
					createTransUnit(body, id, sentence, resultInfo);
				}
			} else {
				createTransUnit(body, id, text, resultInfo);
			}
			id.incrementSeq();
		}
	}

	/**
	 * trans-unit の作成。
	 *
	 * <UL>
	 * <LI>このメソッドは HTML -> XLIFF で利用されます。
	 * </UL>
	 *
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source,
			final BentenProcessResultInfo resultInfo) {
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliffAndSource(final File sourceFile, final File xliffFile,
			final BentenProcessResultInfo resultInfo) throws IOException {

		final BentenXliff xliff = BentenXliff.loadInstance(xliffFile);
		final Iterator<BlancoXliffTransUnit> units = xliff.getAllTransUnitList().iterator();
		final HtmlInlineTextBuilder htmlBuilder = new HtmlInlineTextBuilder(sourceFile);

		for (final InlineText inline : htmlBuilder) {
			final SentencePartitionList targetList = new SentencePartitionList();

			while (units.hasNext()) {
				resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

				final BlancoXliffTransUnit unit = units.next();
				final BlancoXliffTarget t = unit.getTarget();

				if (t != null && t.getTarget() != null && !t.getTarget().equals("")) { //$NON-NLS-1$
					targetList.add(t.getTarget());
				} else {
					targetList.add(unit.getSource());
				}

				if (!HelpTransUnitId.hasContinue(unit.getId())) {
					break;
				}
			}

			final String target = targetList.join();

			if (target.length() > 0) {
				inline.setText(target);
			} else {
				inline.setText(inline.getText());
			}
		}

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		htmlBuilder.writeTo(outStream, "UTF-8"); //$NON-NLS-1$
		outStream.flush();
		byte[] bytes = outStream.toByteArray();

		// Content-type 指定が存在しない場合には、これを追加します。
		bytes = resolveContentType(bytes);

		return bytes;
	}

	/**
	 * コンテンツ・タイプの解決。
	 *
	 * <UL>
	 * <LI>このメソッドは XLIFF -> HTML で利用されます。
	 * </UL>
	 *
	 * @param bytes HTML バイト・データ
	 * @return HTML バイト・データ
	 * @throws IOException 入出力例外が発生した場合
	 */
	byte[] resolveContentType(byte[] bytes) throws IOException {
		// エンコーディングは UTF-8 固定です。
		final String encoding = "UTF-8"; //$NON-NLS-1$
		final String html = new String(bytes, encoding);

		// HTML の meta charset 指定が存在しない場合 (HTML が正規化されている前提)
		if (!html
				.replaceAll("<!--.*?-->", "").matches("(?i)(?s).+?<meta[^>]*?\\s+content=\"text/html;\\s+charset=.+?\".+")) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

			// meta タグの作成
			final StringBuilder metaTag = new StringBuilder();
			metaTag.append("\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset="); //$NON-NLS-1$
			metaTag.append(encoding);
			metaTag.append("\">\n"); //$NON-NLS-1$

			// head タグの後に meta タグを挿入
			String resultHtml = insertMetaTag(html, "<head>", metaTag.toString()); //$NON-NLS-1$

			// head タグがない場合は作成
			if (resultHtml == null) {
				resultHtml = insertMetaTag(html, "<html>", "\n<head>" + metaTag.toString() + "</head>\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
			bytes = resultHtml.getBytes(encoding);
		}
		return bytes;
	}

	/**
	 * meta タグの挿入。
	 *
	 * <UL>
	 * <LI>このメソッドは XLIFF -> HTML で利用されます。
	 * </UL>
	 *
	 * @param html html 文字列
	 * @param beforeTag 挿入する前のタグ
	 * @param metaTag メタタグ
	 * @return meta タグ挿入後の html 文字列。指定された beforeTag が見つからない場合は null。
	 */
	private String insertMetaTag(final String html, final String beforeTag, final String metaTag) {
		final StringBuilder sb = new StringBuilder();
		final int beforeTagIndex = html.toLowerCase().indexOf(beforeTag.toLowerCase());
		if (beforeTagIndex != -1) {
			sb.append(html.substring(0, beforeTagIndex + beforeTag.length()));
			sb.append(metaTag);
			sb.append(html.substring(beforeTagIndex + beforeTag.length()));
			return sb.toString();
		}
		return null;
	}
}
