/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.ui;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IEclipsePreferences.IPreferenceChangeListener;
import org.eclipse.jface.preference.IPreferenceStore;
import org.osgi.framework.BundleContext;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.cat.tm.engine.AbstractBentenTmEnginePlugin;
import benten.cat.tm.ui.messages.CatTmUiPluginMessages;
import benten.core.text.Strings;
import benten.ui.AbstractBentenUiPlugin;
import benten.ui.preference.BentenPreference;
import benten.ui.preference.BentenPreference.Preference;

/**
 * CAT TM UI プラグイン。
 *
 * <UL>
 * <LI>このプラグインを Eclipse 上で有効化するためのプラグイン・クラスです。
 * </UL>
 *
 * @author YAMAMOTO Koji
 * @author IGA Tosiki
 * @author KASHIHARA Shinji
 */
public class CatTmUiPlugin extends AbstractBentenUiPlugin {

	/**
	 * CAT TM UI プラグインのためのメッセージ。
	 */
	protected static final CatTmUiPluginMessages fMsg = new CatTmUiPluginMessages();

	/** 共用インスタンス */
	private static CatTmUiPlugin plugin;

	/** loadExtensions メソッドが 1 度でも実行済みかどうか。 */
	private boolean fLoadExtensionsCalled = false;

	/**
	 * このプラグインが保持している TM エンジン・プラグイン。
	 */
	private AbstractBentenTmEnginePlugin fTmEnginePlugin = null;

	/** 設定変更リスナー。 */
	private final List<IPreferenceChangeListener> preferenceChangeListners = new LinkedList<IPreferenceChangeListener>();

	@Override
	public void start(final BundleContext context) throws Exception {
		super.start(context);
		plugin = this;
	}

	@Override
	public void stop(final BundleContext context) throws Exception {
		plugin = null;
		super.stop(context);
	}

	/**
	 * このプラグインの共用インスタンスを取得。
	 * @return このプラグインの共用インスタンス
	 */
	public static CatTmUiPlugin getDefault() {
		return plugin;
	}

	/**
	 * このプラグインが利用する拡張をロード。
	 *
	 * <UL>
	 * <LI>デフォルトの TM ンジン・プラグインのインスタンスを取得します。
	 * <LI>TMX をロードします。
	 * </UL>
	 * 
	 * @throws IllegalArgumentException ロードに失敗した場合。
	 */
	public void loadExtensions() {
		try {
			fTmEnginePlugin = newTmEnginePluginInstance();
		} catch (final Exception e) {
			log(e);
			throw new IllegalArgumentException(fMsg.getMsg002(e.getMessage()));
		}

		// 翻訳元・先の言語設定は、プロジェクト毎ではなく全体で設定します。
		// なぜなら、TM のロードは Eclipse 単位だからです。
		final IPreferenceStore store = BentenPreference.getStore();
		fTmEnginePlugin.getEngine().setLang(store.getString(BentenPreference.Preference.TRANS_SOURCE_LANG.toString()),
				store.getString(BentenPreference.Preference.TRANS_TARGET_LANG.toString()));

		try {
			fTmEnginePlugin.load();
		} catch (final Exception e) {
			log(e);
			throw new IllegalArgumentException(fMsg.getMsg001(e.getMessage()));
		}

		fLoadExtensionsCalled = true;
	}

	/**
	 * あいまい一致 TM 検索結果のリストを取得。
	 *
	 * <UL>
	 * <LI>あいまい一致 TM 検索結果のリストを取得します。
	 * </UL>
	 *
	 * @param string ターゲット。
	 * @return あいまい一致 TM 検索結果のリスト。
	 * @throws IllegalArgumentException ロードに失敗した場合。
	 */
	public List<BentenTmSearchResult> fuzzyMatch(String string) {
		synchronized (this) {
			if (fLoadExtensionsCalled == false) {
				loadExtensions();
			}
			final IPreferenceStore store = BentenPreference.getStore();

			final boolean ignoreWhitespace = store.getBoolean(Preference.IGNORE_WHITESPACE_TM_REFERENCE.name());
			if (ignoreWhitespace) {
				string = Strings.removeRedundantWhitespace(string);
			}

			final boolean ignoreMnemonicKey = store.getBoolean(Preference.IGNORE_MNEMONIC_KEY_TM_REFERENCE.name());
			String strMnemonicKey = ""; //$NON-NLS-1$
			if (ignoreMnemonicKey) {
				// ニーモニック・キーにまつわる事前処理。
				final String origString = string;
				string = Strings.stripMnemonicKey(string);
				strMnemonicKey = origString.substring(string.length());
			}

			final List<BentenTmSearchResult> result = getTmEngine().fuzzySearch(string);

			if (ignoreMnemonicKey) {
				// ニーモニック・キーにまつわる事後処理。
				for (final BentenTmSearchResult item : result) {
					item.setSource(item.getSource() + strMnemonicKey);
					item.setTarget(item.getTarget() + strMnemonicKey);
				}
			}

			return result;
		}
	}

	/**
	 * このプラグインが保持している TM エンジンを取得。
	 * 
	 * <UL>
	 * <LI>高速に TM エンジンを取得したい場合に利用します。
	 * <LI>「設定」で指定した TM がすでに読み込み済みのエンジンを取得できます。
	 * </UL>
	 *
	 * @return TM エンジン
	 * @throws IllegalArgumentException ロードに失敗した場合。
	 */
	public BentenTmEngine getTmEngine() {
		if (fLoadExtensionsCalled == false) {
			loadExtensions();
		}

		return fTmEnginePlugin.getEngine();
	}

	/**
	 * TM エンジン・プラグインのインスタンスを新規に作成。
	 * 
	 * <UL>
	 * <LI>自力で TM エンジンに TM を読み込ませたい場合に利用します。
	 * <LI>一般的にはこのメソッドは利用せずに {@link #getTmEngine} を利用してエンジンを取得するほうが手軽で高速です。
	 * <LI>デフォルトでプラグインが保持している TM とは別の TM を利用したい場合などにのみ、このメソッドを利用します。
	 * <LI>このメソッドでは、所定のルールに従って拡張ポイントを検索して TM エンジン・プラグインのインスタンスを新規作成します。
	 * </UL>
	 * 
	 * @return 新規作成された TM エンジン・プラグインのインスタンス。必ず null 以外が戻ります。
	 * @throws IllegalArgumentException TM エンジン・プラグインのインスタンスが取得できなかった場合。
	 */
	public AbstractBentenTmEnginePlugin newTmEnginePluginInstance() {
		final Map<String, AbstractBentenTmEnginePlugin> tmEnginePluginNameMap = getTmEnginePluginMap();

		// 「設定」から翻訳メモリー･エンジンの名称を取得。
		// 該当する翻訳メモリー・エンジンの翻訳メモリーをロードする。
		final IPreferenceStore store = BentenPreference.getStore();
		final String tmName = store.getString(Preference.TM_NAME.name());
		final AbstractBentenTmEnginePlugin enginePlugin = tmEnginePluginNameMap.get(tmName);
		if (enginePlugin == null) {
			throw new IllegalArgumentException(fMsg.getGetTmEnginePluginE002());
		}

		return enginePlugin;
	}

	/**
	 * TM エンジン・プラグイン名のセットを取得。
	 * 
	 * <UL>
	 * <LI>利用可能な用語集エンジン・プラグイン名のリストを表示したい場合などに利用します。
	 * </UL>
	 * 
	 * @return TM エンジン・プラグイン名のセット
	 */
	public Set<String> getTmEnginePluginNameSet() {
		return getTmEnginePluginMap().keySet();
	}

	/**
	 * 拡張ポイントから知ることができる TM エンジン・プラグイン名のマップを取得。
	 * 
	 * @return TM エンジン・プラグイン名のマップ
	 */
	private Map<String, AbstractBentenTmEnginePlugin> getTmEnginePluginMap() {
		final Map<String, AbstractBentenTmEnginePlugin> tmEnginePluginNameMap = new HashMap<String, AbstractBentenTmEnginePlugin>();

		final IExtensionRegistry registry = Platform.getExtensionRegistry();
		final IExtensionPoint point = registry.getExtensionPoint("benten.cat.tm.engine"); //$NON-NLS-1$
		if (point == null) {
			throw new IllegalArgumentException(fMsg.getGetTmEnginePluginE001());
		}

		// このシステムに登録されている、すべての翻訳メモリー・エンジンの一覧を取得。
		// 取得された翻訳メモリー･エンジンの名前を記憶。
		for (final IExtension extension : point.getExtensions()) {
			for (final IConfigurationElement cfgElem : extension.getConfigurationElements()) {
				try {
					final String name = cfgElem.getAttribute("name"); //$NON-NLS-1$
					final Object obj = cfgElem.createExecutableExtension("class"); //$NON-NLS-1$
					if (obj instanceof AbstractBentenTmEnginePlugin) {
						tmEnginePluginNameMap.put(name, (AbstractBentenTmEnginePlugin) obj);
					}
				} catch (final CoreException e) {
					log(e);
				}
			}
		}

		return tmEnginePluginNameMap;
	}

	/**
	 * 設定変更の通知。
	 */
	public void firePreferenceChange() {
		for (final IPreferenceChangeListener listener : preferenceChangeListners) {
			listener.preferenceChange(null);
		}
	}

	/**
	 * 設定変更リスナーの追加。
	 * @param listner 設定変更リスナー
	 */
	public void addPreferenceChangeListner(final IPreferenceChangeListener listner) {
		preferenceChangeListners.add(listner);
	}

	/**
	 * 設定変更リスナーの除去。
	 * @param listner 設定変更リスナー
	 */
	public void removePreferenceChangeListner(final IPreferenceChangeListener listner) {
		preferenceChangeListners.remove(listner);
	}
}
