/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
/*
 * Copyright (C) 2010 IGA Tosiki.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package benten.twa.filter.engine.po;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterEngine;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.commons.util.BlancoStringUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffAltTrans;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffNote;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * PO ファイルの処理。
 * 
 * @author IGA Tosiki
 * @see "http://www.gnu.org/software/hello/manual/gettext/PO-Files.html"
 */
public class BentenTwaFilterPoEngine implements BentenTwaFilterEngine {

	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		final String fileName = file.getName().toLowerCase();
		if (fileName.endsWith(".po")) { //$NON-NLS-1$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliff(final File sourceFile, final BlancoXliff xliff, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffFile xliffFile = xliff.getFileList().get(0);

		xliffFile.setDatatype("po"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final List<SimplePOEntry> entryList = new SimplePOReader().process(new ByteArrayInputStream(bytesFile));

		BlancoXliffTransUnit unit = new BlancoXliffTransUnit();

		String pushedPlural = ""; //$NON-NLS-1$
		for (SimplePOEntry entry : entryList) {
			final String literal = getLiteral(entry.getStringLiteralList());
			String value = entry.getCommand();

			if ("x-benten-whitespace".equals(entry.getCommand())) { //$NON-NLS-1$
				if (unit.getSource() != null) {
					addTransUnit(xliffFile, id, resultInfo, unit);

					// 次の翻訳単位を生成。
					unit = new BlancoXliffTransUnit();
				}
			} else if ("msgid".equals(value)) { //$NON-NLS-1$
				unit.setSource(unescapeString(literal));
			} else if ("msgid_plural".equals(value)) { //$NON-NLS-1$
				pushedPlural = pushedPlural + unescapeString(literal);
			} else if (value.startsWith("msgstr")) { //$NON-NLS-1$
				if (value.equals("msgstr") || value.equals("msgstr[0]")) { //$NON-NLS-1$ //$NON-NLS-2$
					// そのままOK。						
				} else {

					addTransUnit(xliffFile, id, resultInfo, unit);

					// 次の翻訳単位を生成。
					unit = new BlancoXliffTransUnit();
					// source などを引継ぎ。
					unit.setSource(pushedPlural);
					// TODO source 以外の引継ぎ

					{
						final BlancoXliffNote note = new BlancoXliffNote();
						unit.getNoteList().add(note);
						note.setFrom("benten-po-plural"); //$NON-NLS-1$
						note.setText(value);
					}
				}

				if (unit.getTarget() == null) {
					unit.setTarget(new BlancoXliffTarget());
				}
				if (unit.getTarget().getTarget() == null) {
					unit.getTarget().setTarget(""); //$NON-NLS-1$
				}
				unit.getTarget().setTarget(unit.getTarget().getTarget() + unescapeString(literal));
			} else if ("msgctxt".equals(value)) { //$NON-NLS-1$
				final BlancoXliffNote note = new BlancoXliffNote();
				unit.getNoteList().add(note);
				note.setFrom("benten-po-msgctxt"); //$NON-NLS-1$
				note.setText(unescapeString(literal));
			} else {
				throw new IllegalArgumentException("Unknown id[" + value + "] is taken while processing PO."); //$NON-NLS-1$ //$NON-NLS-2$
			}

			// PO のコメントを XLIFF のコメントに移送。
			for (String comment : entry.getCommentList()) {
				final BlancoXliffNote note = new BlancoXliffNote();
				unit.getNoteList().add(note);
				if (comment.startsWith(": ")) { //$NON-NLS-1$
					note.setFrom("benten-po-reference"); //$NON-NLS-1$
				} else if (comment.startsWith(". ")) { //$NON-NLS-1$
					note.setFrom("benten-po-extracted-comment"); //$NON-NLS-1$
				} else if (comment.startsWith(", ")) { //$NON-NLS-1$
					note.setFrom("benten-po-flag"); //$NON-NLS-1$
				} else if (comment.startsWith("| ")) { //$NON-NLS-1$
					note.setFrom("benten-po-previous"); //$NON-NLS-1$
				} else {
					note.setFrom("benten-po-comment"); //$NON-NLS-1$
				}
				note.setText(comment);
			}
		}

		if (unit.getSource() != null) {
			addTransUnit(xliffFile, id, resultInfo, unit);
		}

		// 後整理。
		for (BlancoXliffTransUnit unit2 : xliffFile.getBody().getTransUnitList()) {
			// PO のステータスを移送
			String newState = null;
			for (BlancoXliffNote note : unit2.getNoteList()) {
				if (note.getText() != null && note.getText().trim().startsWith(", ") //$NON-NLS-1$
						&& note.getText().trim().indexOf(" fuzzy") >= 0) { //$NON-NLS-1$
					newState = "needs-review-translation"; //$NON-NLS-1$
				}
			}
			if (newState != null && unit2.getTarget() != null) {
				unit2.getTarget().setState(newState);

				final BlancoXliffAltTrans altTrans = new BlancoXliffAltTrans();
				unit2.getAltTransList().add(altTrans);
				altTrans.setToolId("benten-po"); //$NON-NLS-1$
				altTrans.setMatchQuality("75%"); //$NON-NLS-1$
				altTrans.setOrigin("PO"); //$NON-NLS-1$
				altTrans.setSource(unit.getSource());
				final BlancoXliffTarget altTarget = new BlancoXliffTarget();
				altTarget.setTarget(unit2.getTarget().getTarget());
				altTrans.setTarget(altTarget);
			}
		}
	}

	private String getLiteral(List<String> list) {
		StringBuilder strbuf = new StringBuilder();
		for (String literal : list) {
			strbuf.append(literal);
		}
		return strbuf.toString();
	}

	private void addTransUnit(final BlancoXliffFile xliffFile, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo, final BlancoXliffTransUnit unit) {
		xliffFile.getBody().getTransUnitList().add(unit);
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);
		unit.setId(id.toString());
		id.incrementSeq();

		// status の設定
		if (unit.getTarget() != null && unit.getTarget().getTarget() != null) {
			if (unit.getTarget().getTarget().length() == 0) {
				unit.getTarget().setState("new"); //$NON-NLS-1$
			} else {
				unit.getTarget().setState("translated"); //$NON-NLS-1$
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliff(final File sourceFile, final BlancoXliff xliff, final BentenProcessResultInfo resultInfo)
			throws IOException {

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();

		final List<SimplePOEntry> originalEntryList = new SimplePOReader().process(new ByteArrayInputStream(bytesFile));
		int entryIndex = 0;

		for (BlancoXliffFile file : xliff.getFileList()) {
			for (BlancoXliffTransUnit unit : file.getBody().getTransUnitList()) {
				boolean isFirstEmptyStringLiteralExist = false;

				for (entryIndex++;; entryIndex++) {
					if (originalEntryList.get(entryIndex).getCommand().startsWith("msgstr")) { //$NON-NLS-1$
						// 該当する msgstr を発見しました。
						break;
					}
				}

				if (unit.getTarget() != null && unit.getTarget().getTarget() != null) {
					// 展開
					final List<String> literalLines = escapeString(BlancoStringUtil.null2Blank(unit.getTarget()
							.getTarget()));

					final StringBuilder strbufCompareTransUnit = new StringBuilder();
					for (String line : literalLines) {
						strbufCompareTransUnit.append(line);
					}

					final StringBuilder strbufComparePO = new StringBuilder();
					final SimplePOEntry entry = originalEntryList.get(entryIndex);
					for (String line : entry.getStringLiteralList()) {
						strbufComparePO.append(line);
					}

					if (strbufCompareTransUnit.toString().equals(strbufComparePO.toString())) {
						// 何もしません。
						// これは、取り込んだままの状況だからです。
					} else {
						// 文字列リテラル部分を一旦クリアします。
						entry.getStringLiteralList().clear();

						boolean isFirstTarget = true;

						for (String line : literalLines) {
							if (isFirstTarget) {
								isFirstTarget = false;

								if (isFirstEmptyStringLiteralExist) {
									if (line.length() > 0) {
										// po ファイルの元の空文字列による空行がある場合があります。これを再現するための特殊なコードです。
										entry.getStringLiteralList().add(""); //$NON-NLS-1$
									}
								}
							}

							entry.getStringLiteralList().add(line);
						}
					}

					// state にまつわる fuzzy 除去処理。
					int entryIndexRemove = entryIndex;
					for (; entryIndexRemove >= 0; entryIndexRemove--) {
						SimplePOEntry justPrevEntry = originalEntryList.get(entryIndexRemove);
						if (justPrevEntry.getCommand().equals("x-benten-whitespace")) { //$NON-NLS-1$
							// ホワイトスペースの箇所に到達しました。これより前を処理しません。
							break;
						}

						// 翻訳単位のコメントを PO に転送。
						if (justPrevEntry.getCommentList().size() > 0) {
							for (int index = unit.getNoteList().size() - 1; index > 0; index--) {
								final BlancoXliffNote note = unit.getNoteList().get(index);
								if (note.getFrom() == null || note.getFrom().startsWith("benten-po") == false) { //$NON-NLS-1$
									justPrevEntry.getCommentList().add(0,
											"  " + (note.getFrom() == null ? "" : "<" + note.getFrom() + ">") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
													+ (note.getText() == null ? "" : " " + note.getText())); //$NON-NLS-1$ //$NON-NLS-2$
								}
							}
						}

						// PO の fuzzy を処理します。
						// その entry 中のコメントから fuzzy の行を除去します。
						boolean isFuzzyExist = false;
						for (int index = justPrevEntry.getCommentList().size() - 1; index >= 0; index--) {
							final String comment = justPrevEntry.getCommentList().get(index);
							if (comment != null && comment.trim().startsWith(", ") //$NON-NLS-1$
									&& comment.trim().indexOf(" fuzzy") >= 0) { //$NON-NLS-1$
								// もはや fuzzy ではなくなっています。, fuzzy の行を除去します。
								isFuzzyExist = true;
								if ("needs-review-translation".equals(unit.getTarget().getState()) == false) { //$NON-NLS-1$
									justPrevEntry.getCommentList().remove(index);
								}
							}
						}
						// 今回 needs-review-translation に変更されたのであれば、, fuzzy を追加します。
						if (justPrevEntry.getCommentList().size() > 0) {
							if ("needs-review-translation".equals(unit.getTarget().getState()) && isFuzzyExist == false) { //$NON-NLS-1$
								justPrevEntry.getCommentList().add(", fuzzy"); //$NON-NLS-1$
							}
						}
					}
				}
			}
		}

		new SimplePOWriter().process(originalEntryList, outStream);

		return outStream.toByteArray();
	}

	/**
	 * 文字列のエスケープを解除します。
	 * 
	 * @param input 入力文字列。
	 * @return エスケープ解除後の文字列。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	String unescapeString(final String input) throws IOException {
		final StringReader reader = new StringReader(input);
		final StringWriter writer = new StringWriter();

		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				break;
			}
			final char cRead = (char) iRead;
			switch (cRead) {
			case '\\':
				reader.mark(1);
				final int iRead2 = reader.read();
				if (iRead2 < 0) {
					break;
				}
				final char cRead2 = (char) iRead2;
				switch (cRead2) {
				case 'n':
					writer.write("\n"); //$NON-NLS-1$
					break;
				case 'r':
					writer.write("\r"); //$NON-NLS-1$
					break;
				case 't':
					writer.write("\t"); //$NON-NLS-1$
					break;
				case '\\':
				case '"':
					writer.write(cRead2);
					break;
				default:
					reader.reset();
					writer.write(cRead2);
					break;
				}
				break;
			default:
				writer.write(cRead);
				break;
			}
		}

		writer.flush();
		return writer.toString();
	}

	/**
	 * 文字列をエスケープします。
	 * @param input 入力文字列。
	 * @return エスケープ後の文字列。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	List<String> escapeString(final String input) throws IOException {
		final List<String> result = new ArrayList<String>();
		final StringReader reader = new StringReader(input);
		StringWriter writer = new StringWriter();

		for (;;) {
			reader.mark(1);
			final int iRead = reader.read();
			if (iRead < 0) {
				break;
			}
			final char cRead = (char) iRead;
			switch (cRead) {
			case '\\':
			case '"':
				writer.write("\\" + cRead); //$NON-NLS-1$
				break;
			case '\t':
				writer.write("\\t"); //$NON-NLS-1$
				break;
			case '\n':
			case '\r':
				if (cRead == '\n') {
					writer.write("\\n"); //$NON-NLS-1$
				} else {
				}
				// 行変
				writer.flush();
				result.add(writer.toString());
				writer = new StringWriter();
				break;
			default:
				writer.write(cRead);
				break;
			}
		}

		writer.flush();
		final String line = writer.toString();
		if (result.size() == 0 || line.length() > 0) {
			result.add(writer.toString());
		}

		return result;
	}
}
