/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.dialogs;

import java.io.File;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

import benten.twa.ui.dialogs.messages.AbstractWorkflowDialogMessages;
import benten.ui.UiStatusException;
import benten.ui.fields.AbstractBrowseTextField;
import benten.ui.fields.IBrowseTextField;
import benten.ui.fields.IFolderTextField;
import benten.ui.fields.TextField;

/**
 * ワークフローで使用するダイアログの基底クラス。
 *
 * <UL>
 * <LI>翻訳ワークフローで必要な OK ボタン押下時の処理や検証が抽象化されています。
 * <UL>
 *
 * @author KASHIHARA Shinji
 */
public abstract class AbstractWorkflowDialog extends StatusDialog implements ModifyListener {
	/**
	 * ワークフローで使用するダイアログの基底クラスのためのメッセージ。
	 */
	protected static final AbstractWorkflowDialogMessages fMsg = new AbstractWorkflowDialogMessages();

	/** 選択。 */
	private final IStructuredSelection fSelection;
	/** 変更リスナー。 */
	private ModifyListener modifyListener;

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param selection 選択
	 * @param title タイトル
	 */
	public AbstractWorkflowDialog(final Shell parentShell, final IStructuredSelection selection, final String title) {
		super(parentShell);
		setTitle(title);
		setHelpAvailable(false);
		fSelection = selection;
	}

	@Override
	public void create() {
		super.create();
		updateStatus(getValidateStatus());
	}

	@Override
	protected Control createDialogArea(final Composite parent) {
		final Composite composite = (Composite) super.createDialogArea(parent);
		createControls(composite);
		return composite;
	}

	/**
	 * コントロールを作成するテンプレート・メソッド。
	 * @param composite コンポジット
	 */
	abstract public void createControls(final Composite composite);

	/**
	 * 選択されたリソースを取得。
	 * @return 選択されたリソース
	 */
	protected IResource getSelectionResource() {
		final Object o = fSelection.getFirstElement();
		if (o != null && o instanceof IResource) {
			return (IResource) o;
		}
		return null;
	}

	/**
	 * 選択されたリソース文字列を取得。
	 * @return 選択されたリソース文字列
	 */
	protected String getSelectionResourceString() {
		final IResource resource = getSelectionResource();
		if (resource != null) {
			return resource.getLocation().toOSString();
		}
		return ""; //$NON-NLS-1$
	}

	@Override
	final protected boolean isResizable() {
		return true;
	}

	@Override
	final protected void okPressed() {
		if (hasError()) {
			return;
		}
		if (okPressedPrompt()) {
			final IStatus status = getResultStatus();
			updateStatus(status);
			super.okPressed();
		}
	}

	/**
	 * このダイアログに入力された値にエラーがあるか判定。
	 * @return エラーがある場合は true
	 */
	public boolean hasError() {
		modifyText(null);
		return getStatus().matches(IStatus.ERROR);
	}

	/**
	 * OK 押下時に確認ダイアログなどを表示する場合にオーバーライドするメソッド。
	 * @return 処理を続行する場合は true
	 */
	public boolean okPressedPrompt() {
		return true;
	}

	/**
	 * 入力値ステータスの取得。
	 * @return 入力値ステータス
	 */
	abstract public IStatus getResultStatus();

	/**
	 * 変更リスナーをセット。
	 * @param modifyListener 変更リスナー
	 */
	public void setModifyListner(final ModifyListener modifyListener) {
		this.modifyListener = modifyListener;
	}

	/**
	 * {@inheritDoc}
	 */
	public void modifyText(final ModifyEvent e) {
		if (modifyListener != null) {
			modifyListener.modifyText(e);
		}
		updateStatus(getValidateStatus());
	}

	/**
	 * 検証ステータスの取得。
	 * @return 検証ステータス
	 */
	public IStatus getValidateStatus() {
		try {
			return validate();
		} catch (final UiStatusException e) {
			return e.getStatus();
		}
	}

	/**
	 * 検証を行うテンプレート・メソッド。
	 * @return 検証ステータス
	 * @throws UiStatusException 検証エラーが発生した場合
	 */
	abstract public IStatus validate() throws UiStatusException;

	//-------------------------------------------------------------------------
	// 検証メソッド

	/**
	 * フィールドのパスを検証。
	 * @param field フィールド
	 * @throws UiStatusException 検証エラーが発生した場合
	 */
	protected void validatePath(final IBrowseTextField field) throws UiStatusException {
		if (field == null) {
			return;
		}
		if (field instanceof TextField && !((TextField) field).getEnabled()) {
			return;
		}
		final String name = field.getLabelText();
		final String path = field.getText();
		if (path.length() == 0) {
			throw new UiStatusException(fMsg.getValidatePathE01(name));
		}
		if (path.startsWith(".") || path.endsWith(".")) { //$NON-NLS-1$ //$NON-NLS-2$
			throw new UiStatusException(fMsg.getValidatePathE02(name));
		}
		final File file = field.getFile();
		if (file == null || !file.exists()) {
			throw new UiStatusException(fMsg.getValidatePathE03(name));
		}
		if (field instanceof IFolderTextField) {
			if (file.isFile()) {
				throw new UiStatusException(fMsg.getValidatePathE04(name));
			}
		} else {
			if (file.isDirectory()) {
				throw new UiStatusException(fMsg.getValidatePathE05(name));
			}
			if (field instanceof AbstractBrowseTextField && ((AbstractBrowseTextField) field).invalidExtension()) {
				throw new UiStatusException(fMsg.getValidatePathE06(name));
			}
		}
	}

	/**
	 * フィールドの循環関係を検証。
	 * @param fromField 入力元フィールド
	 * @param toField 出力先フィールド
	 * @throws UiStatusException 検証エラーが発生した場合
	 */
	protected void validateFromTo(final IBrowseTextField fromField, final IBrowseTextField toField)
			throws UiStatusException {
		if (fromField == null || toField == null) {
			return;
		}
		if (fromField instanceof TextField && !((TextField) fromField).getEnabled() && toField instanceof TextField
				&& !((TextField) toField).getEnabled()) {
			return;
		}
		String from = fromField.getFile().getAbsolutePath().replace('\\', '/') + "/"; //$NON-NLS-1$
		final String to = toField.getFile().getAbsolutePath().replace('\\', '/') + "/"; //$NON-NLS-1$
		if (from.equals(to)) {
			throw new UiStatusException(fMsg.getValidateFromToE01(fromField.getLabelText(), toField.getLabelText()));
		}
		if (toField instanceof IFolderTextField) {
			if (!(fromField instanceof IFolderTextField)) {
				from = fromField.getFile().getParentFile().getAbsolutePath().replace('\\', '/') + "/"; //$NON-NLS-1$
			}
			if (from.startsWith(to)) {
				throw new UiStatusException(fMsg.getValidateFromToE02(toField.getLabelText(), fromField.getLabelText()));
			}
			if (to.startsWith(from)) {
				throw new UiStatusException(fMsg.getValidateFromToE03(toField.getLabelText(), fromField.getLabelText()));
			}
		}
	}

	/**
	 * フィールドの必須チェック。
	 * @param field フィールド
	 * @throws UiStatusException 検証エラーが発生した場合
	 */
	protected void validateRequire(final TextField field) throws UiStatusException {
		if (field == null) {
			return;
		}
		if (field.getText().equals("")) { //$NON-NLS-1$
			throw new UiStatusException(fMsg.getValidateRequireE01(field.getLabelText()));
		}
	}

	/**
	 * フィールドの正規表現チェック。
	 *
	 * <UL>
	 * <LI>okPattern、ngPattern に含まれる正規表現を文字列として扱う場合は、正規表現エスケープする必要があります。
	 * </UL>
	 *
	 * @param field フィールド
	 * @param okPattern OK パターン。null の場合はチェックされない。
	 * @param ngPattern NG パターン。null の場合はチェックされない。
	 * @throws UiStatusException 検証エラーが発生した場合
	 */
	protected void validatePattern(final TextField field, final String okPattern, final String ngPattern)
			throws UiStatusException {
		if (field == null) {
			return;
		}
		if (okPattern != null && !field.getText().matches(okPattern)) {
			throw new UiStatusException(fMsg.getValidatePatternE01(field.getLabelText()));
		}
		if (ngPattern != null && field.getText().matches(ngPattern)) {
			throw new UiStatusException(fMsg.getValidatePatternE02(field.getLabelText()));
		}
	}
}
