/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.ui.jobs;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.swt.widgets.Display;

import benten.twa.TwaPlugin;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.ui.jobs.messages.WorkflowJobMessages;
import benten.ui.UiMultiStatus;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.jobs.BentenSchedulingRule;

/**
 * 翻訳ワークフロー・ジョブ。
 *
 * @author KASHIHARA Shinji
 */
public class WorkflowJob extends Job {

	/**
	 * 翻訳ワークフロー・ジョブのためのメッセージ。
	 */
	protected static final WorkflowJobMessages fMsg = new WorkflowJobMessages();

	private String completDialogTitle;
	private String completeMessage;
	private final LinkedHashMap<AbstractWorkflowHandler, IStatus> handlerStatusMap =
		new LinkedHashMap<AbstractWorkflowHandler, IStatus>();

	/**
	 * コンストラクター。
	 * @param name ジョブ名。
	 */
	public WorkflowJob(final String name) {
		super(name);
		setUser(true);

		// ワークフロー・ジョブの同時実行はすべてブロックします。
		setRule(BentenSchedulingRule.INSTANCE);
	}

	/**
	 * ハンドラーとステータスの追加。
	 * @param handler ハンドラー。
	 * @param status ステータス (処理パラメーター)。
	 */
	public void addHandler(final AbstractWorkflowHandler handler, final IStatus status) {
		handlerStatusMap.put(handler, status);
	}

	@Override
	protected IStatus run(final IProgressMonitor monitor) {
		IStatus status = null;
		try {
			for (final Entry<AbstractWorkflowHandler, IStatus> entry : handlerStatusMap.entrySet()) {
				final AbstractWorkflowHandler handler = entry.getKey();
				status = entry.getValue();
				handler.monitor = monitor;

				// ハンドラーを実行します。
				try {
					handler.run(status);
				} catch (final Exception e) {

					// 最初にロギングします。
					TwaPlugin.getDefault().log(e);

					// ロギング後、メッセージを表示したいが、これは外側のコードで実施します。
					throw new IllegalArgumentException(e);
				}

				// ユーザーによるキャンセルがあった場合、処理を中断します。
				if (handler.monitor.isCanceled()) {
					return Status.CANCEL_STATUS;
				}
				// ハンドラー固有の完了処理を実施します。
				handler.complete(status);
			}

			// 完了メッセージを表示します。
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					showCompleteMessage();
				}
			});
			return Status.OK_STATUS;

		} finally {

			if (status instanceof UiStatus) {
				final IProject project = ((UiStatus) status).getProject();
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {

						// プロジェクトのリフレッシュ。異常終了時、キャンセル時でもリフレッシュします。
						if (project != null) {
							try {
								project.refreshLocal(IResource.DEPTH_INFINITE, null);
							} catch (final CoreException e) {
								TwaPlugin.getDefault().log(e);
							}
						}
						// エディターをリフレッシュします。
						UiPlugin.getDefault().fireResourceChange();
					}
				});
			}
		}
	}

	/**
	 * この処理の実行結果情報を取得。
	 * @return 処理結果情報
	 */
	public BentenProcessResultInfo getResultInfo() {

		// 今のところ、複数処理がある場合、最後の結果情報を返します。
		return new LinkedList<AbstractWorkflowHandler>(handlerStatusMap.keySet()).getLast().getResultInfo();
	}

	/**
	 * 完了メッセージの表示。
	 */
	protected void showCompleteMessage() {

		// ダイアログに表示するメッセージを作成します。
		final StringBuffer sb = new StringBuffer();
		sb.append(getCompleteMessage());
		sb.append("\n"); //$NON-NLS-1$
		sb.append(BentenTwaProcessUtil.getResultMessage(getResultInfo()));

		// 詳細ボタン押下時のメッセージを作成します。
		IStatus s = new UiStatus(IStatus.INFO, sb.toString());
		final List<String> messageList = getResultInfo().getMessageList();
		if (messageList.size() > 0) {
			final UiMultiStatus multiStatus = new UiMultiStatus(sb.toString());
			for (final String message : getResultInfo().getMessageList()) {
				multiStatus.add(new UiStatus(IStatus.INFO, message));
			}
			s = multiStatus;
		}

		// ダイアログを開きます。
		UiPlugin.openDialog(getCompleteDialogTitle(), s);
	}

	/**
	 * 完了メッセージのセット。
	 * @param dialogTitle ダイアログ・タイトル。
	 * @param message メッセージ。
	 */
	public void setCompleteMessage(final String dialogTitle, final String message) {
		this.completDialogTitle = dialogTitle;
		this.completeMessage = message;
	}

	/**
	 * 完了ダイアログのタイトルを取得。<br>
	 * セットされていない場合はデフォルトのタイトルを返します。
	 * 現状、ウィザードでは個別、ダイアログではデフォルトのタイトルが使用されています。
	 *
	 * @return 完了ダイアログのタイトル。
	 */
	protected String getCompleteDialogTitle() {
		return completDialogTitle == null ? fMsg.getMsgDefaultCompleteMessage() : completDialogTitle;
	}

	/**
	 * 完了メッセージの取得。<br>
	 * セットされていない場合はデフォルトのメッセージを返します。
	 * 現状、ウィザードでは個別、ダイアログではデフォルトのメッセージが使用されています。
	 *
	 * @return 完了メッセージ。
	 */
	protected String getCompleteMessage() {
		return completeMessage == null ? fMsg.getMsgDefaultCompleteMessage() : completeMessage;
	}
}
