/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.tm.engine.regex;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.core.BentenConstants;
import benten.core.io.Files;
import blanco.tmx.BlancoTmxParser;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;

/**
 * このクラスは、TM エンジンの切り替えテストのためのテストクラスです。
 * 実際の Benten 基本セットには含まれません。
 * 
 * @author IGA Tosiki
 */
public class RegexTmEngine implements BentenTmEngine {
	/**
	 * 翻訳元の言語。
	 */
	protected String fSourceLang = "en-US";//$NON-NLS-1$

	/**
	 * 翻訳先の言語。
	 */
	protected String fTargetLang = "ja-JP";//$NON-NLS-1$

	/**
	 * 翻訳言語を設定。
	 * 
	 * @param source 翻訳元の言語。例: en-US。
	 * @param target 翻訳先の言語。例: ja-JP。
	 */
	public void setLang(final String source, final String target) {
		fSourceLang = source;
		fTargetLang = target;
	}

	/**
	 * TM データのリスト。
	 */
	private List<TmTuvEntry> fListTuvEntry = Collections.synchronizedList(new ArrayList<TmTuvEntry>());

	/**
	 * 検索を継続するかどうか判断する際の最大検索数。
	 * 
	 * <UL>
	 * <LI>この数を超える場合には、検索処理を中断して検索結果を戻します。
	 * </UL>
	 */
	private static final int MAX_RESULT_SIZE = 3000;

	/**
	 * {@inheritDoc}
	 */
	public void loadTmx(final File dirTmx) throws IOException {
		final File[] fileTmxs = dirTmx.listFiles();
		if (fileTmxs == null) {
			throw new IllegalArgumentException("The pathname does not denote directory: " + dirTmx.toString()); //$NON-NLS-1$
		}

		for (int index = 0; index < fileTmxs.length; index++) {
			final File fileTmx = fileTmxs[index];
			if (fileTmx.isFile() == false) {
				continue;
			}
			if (fileTmx.getName().toLowerCase().endsWith(BentenConstants.FILE_EXT_TMX) == false) {
				// ファイルの拡張子が .tmx のものだけ処理します。
				continue;
			}

			final BlancoTmxParser parser = new BlancoTmxParser();
			final BlancoTmx tmx = parser.parse(fileTmx);

			for (BlancoTmxTu tu : tmx.getBody().getTuList()) {
				final TmTuvEntry entry = new TmTuvEntry();

				final List<String> sources = new ArrayList<String>();
				final List<String> targets = new ArrayList<String>();
				for (BlancoTmxTuv tuv : tu.getTuvList()) {
					// TODO 現在は、アンダースコアは対応していない。
					if (tuv.getLang().toLowerCase().startsWith(fSourceLang.toLowerCase())) {
						// 先頭に追加。
						sources.add(0, tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().equals(getLanguageFromLocale(fSourceLang.toLowerCase()))) {
						// 末尾に追加。
						sources.add(tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().startsWith(fTargetLang.toLowerCase())) {
						// 先頭に追加。
						targets.add(0, tuv.getSeg());
					} else if (tuv.getLang().toLowerCase().equals(getLanguageFromLocale(fTargetLang.toLowerCase()))) {
						targets.add(tuv.getSeg());
						// 末尾に追加。
					}
				}
				if (sources.size() == 0) {
					continue;
				}
				if (targets.size() == 0) {
					continue;
				}

				fListTuvEntry.add(entry);
				entry.source = sources.get(0);
				entry.sourceLower = sources.get(0).toLowerCase();
				entry.target = targets.get(0);
				entry.origin = Files.baseName(fileTmx);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public List<BentenTmSearchResult> fuzzySearch(final String target) {
		String regPatternString;
		boolean isRegexMode = false;
		if (target.startsWith("!!")) { //$NON-NLS-1$
			isRegexMode = true;
			// !! の場合に、特殊モードに入ります。
			regPatternString = target.substring(2);
		} else {
			regPatternString = ".*" + Pattern.quote(target.toLowerCase()) + ".*"; //$NON-NLS-1$ //$NON-NLS-2$
		}

		final Pattern regPattern = Pattern.compile(regPatternString);

		final List<BentenTmSearchResult> result = new ArrayList<BentenTmSearchResult>();
		for (int indexTm = 0; indexTm < fListTuvEntry.size(); indexTm++) {
			final TmTuvEntry entry = fListTuvEntry.get(indexTm);

			String searchString = null;
			if (isRegexMode) {
				searchString = entry.source;
			} else {
				searchString = entry.sourceLower;
			}
			final Matcher regMatcher = regPattern.matcher(searchString);
			if (regMatcher.matches()) {
				// ヒット
				final BentenTmSearchResult resultItem = new BentenTmSearchResult();
				result.add(resultItem);
				resultItem.setSource(entry.source);
				resultItem.setTarget(entry.target);
				resultItem.setOrigin(entry.origin);

				int quality = (int) (((double) target.length()) / resultItem.getSource().length() * 100);
				if (resultItem.getSource().equals(target)) {
					quality = 100;
				} else {
					if (quality >= 100) {
						quality = 99;
					}
				}

				resultItem.setMatchQuality(Integer.toString(quality) + "%"); //$NON-NLS-1$
			}
			if (result.size() > MAX_RESULT_SIZE) {
				break;
			}
		}

		Collections.sort(result, new Comparator<BentenTmSearchResult>() {
			public int compare(final BentenTmSearchResult tuLeft, final BentenTmSearchResult tuRight) {
				String left = tuLeft.getMatchQuality().trim();
				String right = tuRight.getMatchQuality().trim();
				if (left.endsWith("%")) { //$NON-NLS-1$
					left = left.substring(0, left.length() - 1);
				}
				if (right.endsWith("%")) { //$NON-NLS-1$
					right = right.substring(0, right.length() - 1);
				}
				return Integer.parseInt(right) - Integer.parseInt(left);
			}
		});

		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	public void unload() {
		fListTuvEntry.clear();
	}

	/**
	 * ロケール文字列から言語部分を抽出。
	 * @param locale en-US など。
	 * @return 言語部分。en など。
	 */
	String getLanguageFromLocale(final String locale) {
		StringTokenizer tok = new StringTokenizer(locale, "-_"); //$NON-NLS-1$
		return tok.nextToken();
	}
}

/**
 * TM の tuv エントリーをあらわすクラス。
 */
class TmTuvEntry {
	/** ソースを小文字化したもの。 */
	public String sourceLower;
	/** ソース。 */
	public String source;
	/** ターゲット。 */
	public String target;
	/** 翻訳メモリーの由来。 */
	public String origin;
}