/*
**  bifdp_a.c
**  bif-c
**
**  Created by Joel Rees on 2009/07/22.
**  Copyright 2009 __Reiisi_Kenkyuu__. All rights reserved.
**
** Translated to C from BIFDP/A, as mechanically as possible.
**
*/


#include "bifdp_a.h"
#include "bif5b_a.h" /* for mERROR() */
#include "bif7b_a.h"


/*
** I am not going to try to emulate what I did with the direct page.
** Wouldn't really make sense.
** It's all absolute addressed with short absolute addresses, anyway.
00020 * Direct Page definitions for BIF
00030 * BIF Copyright 1989 Joel Matthew Rees (see BIF/ASM)
00060 * This file is dependent on BIF/M definitions
00100 DPAGE	EQU *
00105 VDP	EQU DPAGE/256
00110 * ORG needs to leave space for this code
00120 	SETDP VDP
00130 * COLD needs to load VDP into DP register
00500 *
00510 * Direct Page variables
*/


/* Since this is where most of the inner interpreter got defined, 
** I'll define the virtual machine registers here, too.
** Notes from BIFDOC.TXT:
*******************************************************************************
                        The BIF Virtual Machine

fig     6809
UP      [DP]    pointer to the per-USER variable table (USER Pointer)
IP      Y       pointer to the next definition (Instruction Pointer)
RP      S       return/control stack pointer
SP      U       parameter/data stack pointer
W       [S]     pointer to executing definition's parameter field
*/


/* This is a pointer to the taskrecord_s structure for the current task.
** It will be set in COLD, in bifst_a.
** But this is a prime candidate for early refactoring. Or maybe not.
00600 UP	RMB 2
*/
cell_u	UP;	/* [DP]    pointer to the per-USER variable table (USER Pointer) */
cell_u	* IP;	/* Y       Pointer to the next definition to execute (Instruction Pointer). */
cell_u	* RP;	/* S       return/control stack pointer */
cell_u	* SP;	/* U       parameter/data stack pointer */
/* W was originally (ephemeral) on the 6809 S stack because there weren't enough registers. */
cell_u	W;	/* [S]     pointer to executing definition's parameter field */
cell_u	volatile sysSIG;	/* Added as a way to break out of the inner interpreter. */

/* No need for these spares in the C runtime environment? Well, maybe.
00610 	RMB 8 spares
*/
cell_u spares[ SPARE_REG_COUNT ];


#if defined KERNEL_VOCABULARY_JUNK
/* Had this idea late last night, then thought better of it.
**
** The fig FORTH does not provide symbol table entries for these, 
** but, since I'm using C conventions and linking to the bases of the structures,
** I'm going to provide symbol table entries in a separate vocabulary.
** This could be convenient for debugging the symbol table early, as well.
*/
static character_t sxKERNEL[] = "\x7" "xKERNEL";
definition_header_s hxKERNEL =
{	{ (natural_t) sxKERNEL },
	{ MIMM },
	{ 0 },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ 0 },
	{ 0 },
	{ (natural_t) XVOC }, 
	{ { (natural_t) &hxKERNEL } }
};


static character_t sXNEXT[] = "\x5" "XNEXT";
definition_header_s hXNEXT =	
{	{ (natural_t) sXNEXT },
	{ MCOMP },
	{ (natural_t) &hxKERNEL },
	{ MFORE },
	{ (natural_t) &hxKERNEL },
	{ 0 },
	{ 0 },
	{ (natural_t) XNEXT }
};
#endif
/* My 6809 implementation had no specific inner interpreter. 
** See the NEXT macro in BIF/M (bif_m.h).
** Since we can't jump outside the function definition in C, we can't do that here.
** I'll provide sysSIG as a way to return to main().
** Note that the fig forth linked directly to fields inside the definition header structure, 
** saving offset calculations, and so did my 6809 implementation.
** It could be done with C, but I'm not going to. I'll use C conventions, instead.
** See XCOL for more on what's happening here.
**
** It's tempting to want to use a direct function call, but then how does one get anything 
** but the function from the i-code? How do you find the entry in the symbol table or even 
** extract the address being called without actually calling it and then trying to look in 
** the CPU's program counter?
*/
/* Do not absorb this into the WARM boot code:
*/
#if defined DBG_TRACE_NEXT
#define DUMP_W( tag )	\
	fprintf( standardError, "{%s} W:{%p:%s} ", tag, \
	W.bytep, W.definitionp->nameLink.chString + 1 )

static void dumpStack(void)
{	cell_u * i;
	fprintf( standardError, "SP:{%p:", SP );
	for ( i = UP.task->dataStackBase.cellp - 1; i >= SP; --i )
	{	fprintf( standardError, "%08lx", (unsigned long) ( * i ).integer );
		if ( i > SP )
		{	fputc( ',', standardError );
		}
	}
	fputc( '}', standardError );
}

void dumpState( char * tag )
{	DUMP_W( tag );
	dumpStack();
	fputc( '\n', standardError );
}

static int tracing = 0;

int isTracing( void )
{	return tracing;
}

static character_t sTRACEON[] = "\x7" "TRACEON";
definition_header_s hTRACEON =	
{	{ (natural_t) sTRACEON },
	{ 0 },
	{ (natural_t) NULL },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) NULL },
	{ 0 },
	{ (natural_t) TRACEON }
};
extern void TRACEON( void )
{	tracing = 1;
}

static character_t sTRACEOFF[] = "\x8" "TRACEOFF";
definition_header_s hTRACEOFF =	
{	{ (natural_t) sTRACEOFF },
	{ 0 },
	{ (natural_t) NULL },
	{ MFORE },
	{ (natural_t) &hBIF },
	{ (natural_t) NULL },
	{ 0 },
	{ (natural_t) TRACEOFF }
};
extern void TRACEOFF( void )
{	tracing = 0;
}


#endif

inline void XNEXT (void)
{
#if defined DBG_TRACE_NEXT
	cell_u localW = W;
#endif
	while ( ( sysSIG.integer == ICODE_LIST_CONTINUE ) && ( ( W = ( * IP++ ) ).icode != (icode_f) 0 ) )
	{	/* W = ( * IP++ ); */
#if defined DBG_TRACE_NEXT
if ( tracing )
{
	DUMP_W( "NEXT" );
	dumpStack();
	fprintf( standardError, " IP:{%p} callee:{%p:%s}\n", 
		  IP, localW.bytep, localW.definitionp->nameLink.chString + 1 );
}
#endif
		( * ( * W.definitionp ).codeLink.icode )();
	}
#if defined DBG_TRACE_NEXT
if ( tracing )
{	fprintf( standardError, "exiting list:{%p:%s} with exception code {%lX}", 
		  localW.bytep, localW.definitionp->nameLink.chString + 1, (unsigned long) sysSIG.integer );
	dumpStack();
	fputc( '\n', standardError );
}
#endif
	if ( sysSIG.integer > ICODE_LIST_CONTINUE )
	{	--sysSIG.integer;
	}
}
/* Another possibility would be to return a continuation flag and pass the object parameter:
void XNEXT (void)
{	natural_t flag;
	do
	{	W = ( * IP++ );
		flag = ( * ( * W.definitionp ).codeLink.icode )( W );
	} while ( ( sysSIG.integer = flag ) == 0 )
	if ( ( flag > 0 ) && ( flag < INTERPRETER_NEST_LIMIT ) )
	{	--flag;	/ * or something like this, negative for errors, positive to terminate nested interp loops n levels.
	}
	return flag;
}
*/

/* 
01000 *
01010 * nest into icode list
01020 * call by JSR
01030 XCOL	LDX ,S	list ptr
01040 	STY ,S	nest old
01050 	TFR X,Y	interpret
01060 	NEXT
** In my 6809 implementation, 
** the first thing in the parameter field was machine code 
** to subroutine jump to the machine-level characteristic code.
** The result was that the top address on RP pointed to the parameter field.
** That won't work in C (and wasn't really a great idea, after all).
** So I must do it another way.
** So, I could put the parameter field pointer a little deeper in the stack
** (and get rid of W) by having the inner interpreter call the characteristic 
** with the address of the parameter field as a C parameter,
** or I could be fig-conventional and store the address of the parameter field in W.
** I will be conventional, to keep from hiding things.
** See XNEXT for more.
*/
void XCOL(void)
{	cell_u * IPsave = IP;
	/* ( * --RP ).cellp = IP */
	IP = &( ( * W.definitionp ).parameterLink[ 0 ] );
	XNEXT();
	IP = IPsave;
}
/* 01070 * */

/*
01080 * PUSH FOLLOWING WORD
01090 * call by JSR for CONSTANT
01100 XCON	LDD [,S++]
01110 	PSHU D
01120 	NEXT
*/
void XCON(void)
{	( * --SP ) = ( * W.definitionp ).parameterLink[ 0 ];
}
/* 01130 * */

/*
01140 * push following address
01150 * call by JSR for VARIABLE address
01160 XVAR	PULS D
01170 	PSHU D
01180 	NEXT
01190 * (same code as R>, but in DP)
*/
void XVAR(void)
{	( * --SP ).cellp = &( ( * W.definitionp ).parameterLink[ 0 ] );
}
/* 01200 * */

/*
01210 * push address of USER variable
01215 * call by JSR
01220 XUSER	LDB [,S++]	one byte unsigned offset
01225 	CLRA
01230 	ADDD <UP
01240 	PSHU D
01250 	NEXT
** The 6809 source was optimized to straight byte-addressing, byte-packed,
** but the C source will be optimized to cell_u boundaries where it makes sense.
*/
void XUSER(void)
{	( * --SP ).bytep = UP.bytep + ( * W.definitionp ).parameterLink[ 0 ].integer;
}
/* 01260 * */

/*
01300 * push address of LINEAR array entry
01305 * call by JSR
01310 X1ARR	PULS X
01320 	LDD ,U index
01330 	SUBD ,X base
01340 	CMPD 2,X width
01350 	BLO X1ARRS
01360 	LDD #$0C
01365 	PSHU D
01370 	JMP ERROR
01380 X1ARRS	PSHS A
01390 	LDA 4,X size
01400 	MUL
01410 	STD ,U
01420 	PULS A ms byte
01430 	LDB 4,X
01440 	MUL
01450 	TFR B,A
01460 	ADDA ,U
01470 	LDB 1,U
01480 	LEAX 5,X past header
01490 	LEAX D,X
01500 	STX ,U
01510 	NEXT
01520 * LINEAR array header format:
01530 *	FDB BASE (lowest index)
01540 *	FDB WIDTH (# of elements)
01550 *	FCB SIZE (< 256 bytes / element)
01560 *	RMB SIZE*WIDTH (data)
*/
/* untested, 2009.09.14 */
void X1ARR(void)
{	cell_u * base = &( ( * W.definitionp ).parameterLink[ 0 ] );
	snatural_t index = ( * SP ).sinteger;
	index -= ( base[ 0 ] ).sinteger;	/* Adjust by lower limit. */
	if ( index < 0 || index > base[ 1 ].sinteger )	/* Check bounds. */
	{	mERROR( ARRAY_REFERENCE_OUT_OF_BOUNDS );
	}
	else
	{	byte_p arraybase = (byte_p) &( base[ 3 ] );	/* ergo, LINEARRAY_DATAOFFSET, which should not be externally visible, anyway. */
		( * SP ).bytep = &( arraybase[ index * base[ 2 ].integer ] );
	}
}
/* 01700 * */

/*
01710 * push content of USER variable
01715 * call by JSR
01720 XUCON	LDX <UP
01722 	LDB [,S++]	one byte unsigned offset
01725 	CLRA
01730 	LDD D,X
01740 	PSHU D
01750 	NEXT
** The 6809 source was optimized to straight byte-addressing, byte-packed,
** but the C source will be optimized to cell_u boundaries.
*/
void XUCON(void)
{	( * --SP ) = * ( (cell_u *) ( UP.bytep + ( * W.definitionp ).parameterLink[ 0 ].integer ) );
}
/* 01800 * */

/*
01802 * store VOCABULARY pointer in root
01805 * call by JSR
01810 XVOC	LDX <UP
01820 	PULS D
01830 	STD UROOT,X
01840 	NEXT
*/
/* untested, 2009.09.14 */
void XVOC(void)
{	( * UP.task ).searchContextRoot.cellp = (cell_u *) &( ( * W.definitionp ).parameterLink[ 0 ] );
}
/* 01850 * */

/*
01860 * indirect into icode list following
01870 * DOES> in defining definition
01880 * call by JSR
01890 XDOES	LDX ,S pfa
01900 	LDD ,X++	list ptr
01920 	PSHU X	parameters
01930 	STY ,S	nest old
01940 	TFR D,Y	interpret
01950 	NEXT
*/
/* untested, 2009.09.14 */
void XDOES(void)
{
/* Refer to XCOL 
// and rememeber that a <BUILDS ... DOES> definition is not the sames as a colon definition.
*/ 
	cell_u * IPsave = IP;
/* XDOES	LDX ,S pfa ; get the characteristing pointer (definition elsewhere to execute) */
	definition_header_s * definitionp = W.definitionp;
/* 	LDD ,X++	list ptr ; get the definition list (i-code) pointer, point to the parameters */
	cell_u * listp = definitionp->parameterLink[ 0 ].cellp;
/* 	PSHU X	parameters ; push the pointer to the parameters on the parameter stack */
	( * --SP ).cellp = & ( definitionp->parameterLink[ 1 ] );
/* 	STY ,S	nest old ; Replace the characteristic pointer with the address to return to */
	/* IPsave = IP; did this in the initialization. */
/* 	TFR D,Y	interpret */
	IP = listp;
/* 	NEXT */
	XNEXT();
	IP = IPsave;
/* After I test the above, this should also do the trick:
	cell_u * IPsave = IP;
	cell_u * parametersp = &( ( * W.definitionp ).parameterLink[ 0 ] );
	( * --SP ).cellp = parametersp + 1;
	IP = parametersp[ 0 ].cellp;
	XNEXT();
	IP = IPsave;
//
// tested with 
// ( base limit barray name )
//
// : barray <BUILDS OVER - DUP >R SWAP , , 0 , R> ALLOT 0 , DOES> DUP >R @ - R> CELL-SIZE 3 * + + ;
on 2011.01.18:JMR
*/
}
/* 01960 * */

/*
02000 * push double constant
02010 * call by JSR
02020 XDCON	PULS X
02030 	LDD ,X++
02040 	LDX ,X++
02050 	PSHU D,X
02060 	NEXT
*/
/* untested, 2009.09.14 */
void XDCON(void)
{	register dblnatural_t * stack = (dblnatural_t *) ( (char *) SP );
	* --stack = * (dblnatural_t *) ( (char *) ( * W.definitionp ).parameterLink );
	SP = (cell_u *) ( (char *) stack );
	/* SP -= 2;
	// SP[ 1 ] = ( * W.definitionp ).parameterLink[ 1 ];
	// SP[ 0 ] = ( * W.definitionp ).parameterLink[ 0 ];
	*/
}
/* 02070 * */

/*
40000 	ORG DPAGE+256 if code fits
40010 	SETDP 0 not yet valid
*/

/* Need this for shifting between modes?
** No, use mEXEC(), the macro-wrapped EXEC(), instead.
void callDefinition( definition_header_s * headerp )
{	cell_u saveW = W;
	W.definitionp = headerp;
	( * ( * W.definitionp ).codeLink.icode )();
	W = saveW;
}
*/

