/*
 * blancoIg
 * Copyright (C) 2004-2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.ig.expander;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import blanco.ig.expander.field.FieldExpander;
import blanco.ig.expander.implementor.ImplementData;
import blanco.ig.expander.javadoc.ClassJavaDoc;
import blanco.ig.expander.method.MethodExpander;
import blanco.ig.generator.GeneratorSetting;
import blanco.ig.generator.ImplementGenerator;

/**
 * @author Yasuo Nakanishi
 */
public abstract class ClassExpander extends Expander {
    private Type _type = null;

    private ImportList _importList2 = new ImportList();

    private ImplementData _staticImplement = new ImplementData();

    private List _fieldList = new ArrayList();

    private Map _filedMap = new Hashtable();

    private List _interfaceList = new ArrayList();

    private List _methodList = new ArrayList();

    private List _code = null;

    private boolean _existSuperClass = false;

    private Type _superClass = null;

    private ClassJavaDoc _javaDoc = new ClassJavaDoc();

    private List _fileComments = new ArrayList();

    /**
     * 4󔒂ŃCfgƂ܂B
     */
    public static final String INDENT = "    ";

    protected boolean isInterface() {
        return false;
    }

    protected final boolean isStatic() {
        return false;
    }

    public static final Type adjust(Type type) {
        String nameSpace = type.getNameSpace();
        String className = type.getName();
        Type result = new Type(nameSpace, className);
        return result;
    }

    public ClassExpander(Type type) {
        _type = adjust(type);

        createFileComment();
    }

    public void createFileComment() {
        addFileComment("This code is generated by blanco Framework.");
    }

    public void addFileComment(String comment) {
        _fileComments.add(comment);
    }

    public ClassJavaDoc getJavaDoc() {
        return _javaDoc;
    }

    public void addImport(Type type) {
        _importList2.add(type);
    }

    public void addImport(List listType) {
        for (Iterator ite = listType.iterator(); ite.hasNext();) {
            addImport((Type) ite.next());
        }
    }

    public void addField(FieldExpander field) {
        addImport(field.getTypeValue().getType());
        addImport(field.getUsingTypeList());
        _fieldList.add(field);
        _filedMap.put(field.getTypeValue().getName(), field);
    }

    public void setSuperClass(Type type) {
        _existSuperClass = true;
        addImport(type);
        _superClass = type;
    }

    protected boolean existSuperClass() {
        return _existSuperClass;
    }

    protected FieldExpander getField(int index) {
        return (FieldExpander) _fieldList.get(index);
    }

    protected FieldExpander getField(String name) {
        return (FieldExpander) _filedMap.get(name);
    }

    public Value getFieldValue(String fieldName) {
        Value result = null;
        Iterator i = _fieldList.iterator();
        FieldExpander field = null;
        while (i.hasNext()) {
            field = (FieldExpander) i.next();
            if (field.getTypeValue().getName().equals(fieldName)) {
                result = field.getFieldValue();
                break;
            }
        }
        return result;
    }

    protected int getFieldCount() {
        return _fieldList.size();
    }

    public void addMethod(MethodExpander method) {
        _methodList.add(method);
        method.setupSignature();
        method.setClassSource(this);
    }

    protected MethodExpander getMethod(int index) {
        return (MethodExpander) _methodList.get(index);
    }

    protected int getMethodCount() {
        return _methodList.size();
    }

    public void addInterface(Type type) {
        addImport(type);
        _interfaceList.add(type);
    }

    public void addInterface(Class type) {
        addInterface(new Type(type));
    }

    protected void implementStaticBlock() {
    }

    protected abstract void expandClassStruct();

    private void implementAll() {
        implementStaticBlock();
        addImport(_staticImplement.getUsingTypeList());

        MethodExpander m = null;
        for (int i = 0; i < getMethodCount(); i++) {
            m = getMethod(i);
            m.implement();
            addImport(m.getUsingTypeList());
        }
    }

    public final List expand() {

        expandClassStruct();
        implementAll();

        _code = new ArrayList();

        if (_fileComments.size() != 0) {
            _code.add("/*");
            for (int i = 0; i < _fileComments.size(); i++) {
                _code.add(" * " + _fileComments.get(i));
            }
            _code.add(" */");
        }

        if (!_type.getNameSpace().equals("")) {
            _code.add("package " + getNameSpace() + ";");
            _code.add("");
        }

        _code.addAll(_importList2.expand());
        _code.add("");

        if (!_javaDoc.isEmpty()) {
            List list = _javaDoc.expand();
            for (int i = 0; i < list.size(); i++) {
                _code.add(list.get(i));
            }
        }
        if (getAnnotationList().isEmpty() == false) {
            // NX̃Ame[VWJ܂B
            for (int index = 0; index < getAnnotationList().size(); index++) {
                _code.add("@" + (String) getAnnotationList().get(index));
            }
        }
        _code.add(getClassDeclaration());

        expandAllFields();
        _code.add("");

        expandStaticBlock();
        _code.add("");

        expandAllMethod();
        _code.add("}");
        return _code;
    }

    private void expandStaticBlock() {
        if (!_staticImplement.isEmpty()) {
            _code.add(INDENT + "static {");
            List codes = _staticImplement.getImplementList();
            for (int i = 0; i < codes.size(); i++) {
                _code.add(codes.get(i));
            }
            _code.add(INDENT + "}");

        }
    }

    private void expandAllMethod() {
        for (int i = 0; i < getMethodCount(); i++) {
            if (i != 0) {
                _code.add("");
            }
            makeMethodCode(getMethod(i));
        }
    }

    private void expandAllFields() {
        for (int i = 0; i < getFieldCount(); i++) {
            if (i != 0) {
                _code.add("");
            }
            makeFieldCode(getField(i));
        }
    }

    private String getClassDeclaration() {
        StringBuffer sb = new StringBuffer();
        sb.append("public");
        if (isFinal()) {
            sb.append(" final");
        }
        if (isAbstract()) {
            sb.append(" abstract");
        }
        if (isInterface()) {
            sb.append(" interface ");
        } else {
            sb.append(" class ");
        }
        sb.append(_type.getName());
        if (existSuperClass()) {
            sb.append(" extends ");
            sb.append(_superClass.getName());
        }

        if (!_interfaceList.isEmpty()) {
            sb.append(" implements");
            Iterator i = _interfaceList.iterator();
            Type type = null;

            if (i.hasNext()) {
                type = (Type) i.next();
                sb.append(" ");
                sb.append(type.getName());
            }

            while (i.hasNext()) {
                type = (Type) i.next();
                sb.append(", ");
                sb.append(type.getName());
            }
        }

        sb.append(" {");
        return new String(sb);
    }

    private void makeMethodCode(MethodExpander method) {
        List methodCode = method.expand();
        for (int i = 0; i < methodCode.size(); i++) {
            _code.add(methodCode.get(i));
        }
    }

    private void makeFieldCode(FieldExpander field) {
        List fieldCode = field.expand();

        for (int i = 0; i < fieldCode.size(); i++) {
            _code.add(fieldCode.get(i));
        }
    }

    public String getName() {
        return _type.getName();
    }

    public String getNameSpace() {
        return _type.getNameSpace();
    }

    public Type getType() {
        return _type;
    }

    protected ImplementData getStaticImplementData() {
        return _staticImplement;
    }

    public void clear() {
        _fieldList.clear();

        MethodExpander method = null;
        Iterator i = _methodList.iterator();
        while (i.hasNext()) {
            method = (MethodExpander) i.next();
            method.clear();
        }
        _methodList.clear();
        _fileComments.clear();
        _javaDoc = null;
        _staticImplement = null;
        _code.clear();
        _importList2.clear();
    }

    /**
     * ClassExpander͂ƂJava\[XR[ho͂܂B
     * 
     * @param classExpander
     * @param directoryTarget
     */
    public static void generateJavaSource(final ClassExpander classExpander,
            final File directoryTarget) {
        final GeneratorSetting setting = new GeneratorSetting();
        // o͐fBNgݒ肵܂B
        setting.setWorkDirectory(directoryTarget.getAbsolutePath());
        final ImplementGenerator implementGenerator = new ImplementGenerator(
                setting);
        // 쐬NXo^܂B
        implementGenerator.addMain(classExpander);
        try {
            // ۂɃ\[XR[hs܂B
            implementGenerator.generate();
            // System.out.println("\[XR[h܂B");
        } catch (IOException e) {
            e.printStackTrace();
            throw new IllegalArgumentException("\[XR[hɓo͗O܂B:"
                    + e.toString());
        }
    }
}
