/*
 * blanco Framework
 * Copyright (C) 2004-2005 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.constants;

import java.io.File;
import java.io.IOException;

import javax.xml.transform.dom.DOMResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import blanco.commons.util.BlancoJavaSourceUtil;
import blanco.commons.util.BlancoNameUtil;
import blanco.commons.util.BlancoXmlUtil;
import blanco.constants.resourcebundle.BlancoConstantsResourceBundle;
import blanco.constants.valueobject.BlancoConstantsFieldStructure;
import blanco.constants.valueobject.BlancoConstantsStructure;
import blanco.ig.expander.ClassExpander;
import blanco.ig.expander.Scope;
import blanco.ig.expander.Type;
import blanco.ig.expander.field.FieldExpander;
import blanco.ig.expander.implementor.Statement;

/**
 * XMLt@CJava\[XR[hNXłB
 * 
 * 萔`(XLS)\[XR[hvWFNg̈ꕔłB
 * 
 * @author IGA Tosiki
 */
public class BlancoConstantsXml2JavaClass {
    /**
     * \[Xohւ̃ANZTNX̃IuWFNgB
     */
    private final BlancoConstantsResourceBundle bundle = new BlancoConstantsResourceBundle();

    /**
     * ʏ̃GgB
     */
    private static final String ELEMENT_COMMON = "blancoconstants-common";

    /**
     * Xg̃GgB
     */
    private static final String ELEMENT_LIST = "blancoconstants-list";

    /**
     * XMLt@CJava\[XR[h܂B
     * 
     * @param metaXmlSourceFile
     *            ValueObjectɊւ郁^񂪊܂܂ĂXMLt@C
     * @param directoryTarget
     *            \[XR[hfBNg
     * @throws IOException
     *             o͗Oꍇ
     */
    public void process(final File metaXmlSourceFile, final File directoryTarget)
            throws IOException {

        final DOMResult result = BlancoXmlUtil
                .transformFile2Dom(metaXmlSourceFile);

        final Node rootNode = result.getNode();
        if (rootNode instanceof Document) {
            // ꂪnBhLg[g擾
            final Document rootDocument = (Document) rootNode;
            final NodeList listSheet = rootDocument
                    .getElementsByTagName("sheet");
            final int sizeListSheet = listSheet.getLength();
            for (int index = 0; index < sizeListSheet; index++) {
                final Element elementCommon = BlancoXmlUtil.getElement(
                        listSheet.item(index), ELEMENT_COMMON);
                if (elementCommon == null) {
                    // commonꍇɂ̓XLbv܂B
                    continue;
                }

                final String name = BlancoXmlUtil.getTextContent(elementCommon,
                        "name");
                if (name == null || name.trim().length() == 0) {
                    continue;
                }

                final Element elementList = BlancoXmlUtil.getElement(listSheet
                        .item(index), ELEMENT_LIST);

                expandSheet(elementCommon, elementList, directoryTarget);
            }
        }
    }

    /**
     * V[gGgWJ܂B
     * 
     * @param elementCommon
     *            ݏĂCommonm[hB
     * @param elementList
     *            ݏĂListm[hB
     * @param directoryTarget
     *            o͐tH_B
     */
    private void expandSheet(final Element elementCommon,
            final Element elementList, final File directoryTarget) {
        final BlancoConstantsStructure processStructure = new BlancoConstantsStructure();
        processStructure.setName(BlancoXmlUtil.getTextContent(elementCommon,
                "name"));
        processStructure.setPackage(BlancoXmlUtil.getTextContent(elementCommon,
                "package"));
        if (processStructure.getPackage() == null
                || processStructure.getPackage().trim().length() == 0) {
            throw new IllegalArgumentException(bundle
                    .getXml2javaclassErr001(processStructure.getName()));
        }

        if (BlancoXmlUtil.getTextContent(elementCommon, "description") != null) {
            processStructure.setDescription(BlancoXmlUtil.getTextContent(
                    elementCommon, "description"));
        }
        if (BlancoXmlUtil.getTextContent(elementCommon, "fileDescription") != null) {
            processStructure.setFileDescription(BlancoXmlUtil.getTextContent(
                    elementCommon, "fileDescription"));
        }

        if (elementList != null) {
            final NodeList listField = elementList
                    .getElementsByTagName("field");
            for (int indexField = 0; indexField < listField.getLength(); indexField++) {
                final Node nodeField = listField.item(indexField);
                if (nodeField instanceof Element == false) {
                    continue;
                }
                final Element elementField = (Element) nodeField;
                BlancoConstantsFieldStructure field = new BlancoConstantsFieldStructure();
                field.setNo(BlancoXmlUtil.getTextContent(elementField, "no"));

                field.setName(BlancoXmlUtil
                        .getTextContent(elementField, "name"));
                if (field.getName() == null) {
                    continue;
                }

                field.setType(BlancoXmlUtil
                        .getTextContent(elementField, "type"));
                if (field.getType() == null) {
                    throw new IllegalArgumentException(bundle
                            .getXml2javaclassErr002(processStructure.getName(),
                                    field.getName()));
                }

                field.setValue(BlancoXmlUtil.getTextContent(elementField,
                        "value"));
                if (field.getValue() == null) {
                    throw new IllegalArgumentException(bundle
                            .getXml2javaclassErr003(processStructure.getName(),
                                    field.getName()));
                }

                field.setDescription(BlancoXmlUtil.getTextContent(elementField,
                        "description"));

                processStructure.getListField().add(field);
            }
        }

        expandJavaSource(processStructure, directoryTarget);
    }

    /**
     * WꂽɁAJava\[XR[ho͂܂B
     * 
     * @param processStructure
     *            \
     * @param directoryTarget
     *            o͐tH_B
     */
    private void expandJavaSource(
            final BlancoConstantsStructure processStructure,
            final File directoryTarget) {
        final ClassExpander classExpander = new ClassExpander(new Type(
                processStructure.getPackage(), processStructure.getName())) {
            protected void expandClassStruct() {
                if (processStructure.getFileDescription() != null) {
                    String lines = BlancoJavaSourceUtil
                            .escapeStringAsJavaDoc(processStructure
                                    .getFileDescription());
                    addFileComment(lines);
                }

                if (processStructure.getDescription() != null) {
                    String lines = BlancoJavaSourceUtil
                            .escapeStringAsJavaDoc(processStructure
                                    .getDescription());
                    getJavaDoc().addLine(lines);
                }

                for (int indexField = 0; indexField < processStructure
                        .getListField().size(); indexField++) {
                    BlancoConstantsFieldStructure fieldLook = (BlancoConstantsFieldStructure) processStructure
                            .getListField().get(indexField);
                    final FieldExpander field = new FieldExpander(new Type(
                            fieldLook.getType()), fieldLook.getName()) {
                        protected Scope getScope() {
                            return Scope.PUBLIC;
                        }

                        protected boolean isFinal() {
                            return true;
                        }

                        protected boolean isStatic() {
                            return true;
                        }
                    };
                    if (fieldLook.getNo() != null) {
                        field.getJavaDoc().addLine(
                                bundle.getXml2javaclassFieldNo(fieldLook
                                        .getNo()));
                    }
                    if (fieldLook.getDescription() != null) {
                        field.getJavaDoc().addLine(fieldLook.getDescription());
                    }

                    // ^ɂăftHgl܂B
                    // ΉȂ^^ꂽꍇɂ͗O𔭐܂B
                    if (fieldLook.getType().equals("java.lang.String")) {
                        field.setDefaultStatement(new Statement("\""
                                + BlancoJavaSourceUtil
                                        .escapeStringAsJavaSource(fieldLook
                                                .getValue()) + "\""));
                    } else if (fieldLook.getType().equals("boolean")
                            || fieldLook.getType().equals("short")
                            || fieldLook.getType().equals("int")
                            || fieldLook.getType().equals("long")) {
                        field.setDefaultStatement(new Statement(fieldLook
                                .getValue()));
                    } else if (fieldLook.getType().equals("java.lang.Boolean")
                            || fieldLook.getType().equals("java.lang.Integer")
                            || fieldLook.getType().equals("java.lang.Long")) {
                        field.setDefaultStatement(new Statement("new "
                                + BlancoNameUtil.trimJavaPackage(fieldLook
                                        .getType()) + "("
                                + fieldLook.getValue() + ")"));
                    } else if (fieldLook.getType().equals("java.lang.Short")) {
                        field.setDefaultStatement(new Statement("new "
                                + BlancoNameUtil.trimJavaPackage(fieldLook
                                        .getType()) + "((short) "
                                + fieldLook.getValue() + ")"));
                    } else if (fieldLook.getType().equals(
                            "java.math.BigDecimal")) {
                        addImport(new Type("java.math.BigDecimal"));
                        field.setDefaultStatement(new Statement(
                                "new BigDecimal(\"" + fieldLook.getValue()
                                        + "\")"));
                    } else if (fieldLook.getType()
                            .equals("java.util.ArrayList")) {
                        // ArrayList̏ꍇɂ́A^ꂽ̂܂܍̗p܂B
                        addImport(new Type("java.util.ArrayList"));
                        field.setDefaultStatement(new Statement(fieldLook
                                .getValue()));
                    } else {
                        throw new IllegalArgumentException(bundle
                                .getXml2javaclassErr004(processStructure
                                        .getName(), fieldLook.getName(),
                                        fieldLook.getValue(), fieldLook
                                                .getType()));
                    }

                    addField(field);
                }
            }
        };

        ClassExpander.generateJavaSource(classExpander, directoryTarget);
    }
}
