/*
 * blanco Framework
 * Copyright (C) 2004-2010 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.log.custom;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;
import java.util.logging.Handler;
import java.util.logging.LogRecord;

import blanco.log.logging.BlancoLogLoggingConstants;
import blanco.log.resourcebundle.BlancoLogProgramMessage;

/**
 * java.util.logging ̃Oo͂̐ݒɂĎdo͌`ړĨJX^EOEnh[B
 * 
 * <UL>
 * <LI>
 * Java VXevpeB[ublanco.log.custom.BlancoLogHandlerConfDirv
 * ō\fBNg|CgĂ邱ƂOƂ܂B
 * <LI>\fBNgɁuBlancoLogHandler.propertiesvƂt@C݂邱ƂOƂ܂B
 * <LI>GlassFish Ȃǂ̃AvP[VET[o[ɑgݍŗp܂B(J_ł GlassFish 2.1.1 ł̂ݓmF{)
 * <LI>: ҂ɒʒmB̃NX̓ System.out Ȃǂ𗘗pƁAȌzĂ܂ꂪ܂B
 * </UL>
 * 
 * @author IGA Tosiki
 */
public class BlancoLogHandler extends Handler {
    /**
     * blancoLogbZ[WNXB
     */
    protected final BlancoLogProgramMessage fMsg = new BlancoLogProgramMessage();

    /**
     * \fBNg[킷 Java VXevpeB[B
     */
    public static final String ENV_CONF_DIR = "blanco.log.custom.BlancoLogHandlerConfDir";

    /**
     * OEt@C̍ۂɗpt`B
     */
    final SimpleDateFormat sdFormat = new SimpleDateFormat("yyyyMMdd");

    /**
     * OEtH[}b^[EIuWFNgB
     */
    final BlancoLogFormatter fLogFormatter = new BlancoLogFormatter();

    /**
     * Õ_CNgt@C(̂̈ꕔ)ɂẴ}bvB
     */
    final Map<String, String> fLogMap = new HashMap<String, String>();

    /**
     * Oo͐fBNgB
     */
    File fLogDir = null;

    /**
     * AēIuWFNgn錻ۂh~邽߂ɁAOIuWFNgL܂B IuWFNg̏ꍇɂ͏o͂}܂B
     */
    LogRecord fPrevRecord = null;

    /**
     * OtH[}b^[̃CX^XVK쐬܂B
     */
    public BlancoLogHandler() {
        // Java VXevpeB[ɂfBNgw肪KvB
        final String confDirName = System.getProperty(ENV_CONF_DIR);
        if (confDirName == null) {
            throw new IllegalArgumentException(fMsg.getMblgch01(ENV_CONF_DIR));
        }

        final File confDir = new File(confDirName);
        if (confDir.exists() == false) {
            throw new IllegalArgumentException(fMsg.getMblgch02(ENV_CONF_DIR,
                    confDir.getAbsolutePath()));
        }

        try {
            // vpeBEt@Cݒǂݍ݂܂B
            loadProperties(confDir);

            // w肳ꂽOt@CQ݂邩ǂO܂B΍쐬܂B
            ensureLogFilesExist();
        } catch (IOException ex) {
            throw new IllegalArgumentException(fMsg.getMblgch03(ex.toString()),
                    ex);
        }
    }

    @Override
    public void publish(final LogRecord rec) {
        // TODO Iȃt@Co͂̎邱ƁB

        if (rec != fPrevRecord) {
            fPrevRecord = rec;
            try {
                write(rec);
            } catch (IOException ex) {
                throw new IllegalArgumentException(fMsg.getMblgch09(ex
                        .toString()), ex);
            }
        } else {
            // ȂB
            // R͕sAxOo͈˗ꍇB̃Oo͂}B
        }
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() throws SecurityException {
    }

    /**
     * ^ꂽOER[ho͂܂B
     * 
     * @param rec
     * @throws IOException
     */
    void write(final LogRecord rec) throws IOException {
        // xɂAL킹O݁B
        if (BlancoLogLoggingConstants.SEVERE == rec.getLevel().intValue()) {
            // SEVERE x̂̂́Aʓr severe.log t@Cɏo͂܂B
            appendLogLine(fLogFormatter.format(rec), fLogDir, "severe");
        }

        // K[ɂ郍Ot@C̏B
        String logFileName = fLogMap.get(rec.getLoggerName());
        if (logFileName == null) {
            // dɓo^ĂȂ̂ŁA͖܂B
        } else {
            appendLogLine(fLogFormatter.format(rec), fLogDir, logFileName);
        }
    }

    /**
     * ̃t@C̃OEt@CɁAOEbZ[W 1 sǉ܂B
     * 
     * @param mesg
     * @param logDir
     * @param logFileName
     * @throws IOException
     */
    void appendLogLine(final String mesg, final File logDir,
            final String logFileName) throws IOException {
        BufferedWriter writer = null;
        for (int retry = 0; retry < 60; retry++) {
            try {
                final FileOutputStream outStream = new FileOutputStream(
                        new File(logDir, getLogFileName(logFileName)), true);
                outStream.getChannel().lock();
                writer = new BufferedWriter(new OutputStreamWriter(outStream));
                break;
            } catch (IOException ex) {
                // bN擾łB
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    // ܂B
                }
            }
        }

        if (writer == null) {
            throw new IOException(fMsg.getMblgch04(getLogFileName(logFileName)));
        }

        try {
            writer.write(mesg);
            writer.newLine();
        } finally {
            writer.close();
        }
    }

    /**
     * vpeBEt@Cݒǂݍ݂܂B
     * 
     * @param confDir
     * @throws IOException
     */
    void loadProperties(final File confDir) throws IOException {
        final Properties prop = new Properties();
        prop.load(new FileInputStream(new File(confDir,
                "BlancoLogHandler.properties")));

        // o͐惍OfBNgB
        fLogDir = new File(prop.getProperty("LogDir"));

        if (fLogDir.exists() == false) {
            // w̃OEfBNg[݂͑܂BOo͂߂܂B
            throw new IllegalArgumentException(fMsg.getMblgch05(fLogDir
                    .getAbsolutePath()));
        }
        if (fLogDir.isDirectory() == false) {
            throw new IllegalArgumentException(fMsg.getMblgch06(fLogDir
                    .getAbsolutePath()));
        }

        // vpeBt@C̋LqAOE}bsOɗp܂B
        final Set<Entry<Object, Object>> entries = prop.entrySet();
        for (Entry<Object, Object> entry : entries) {
            if ("LogDir".equals(entry.getKey()) == false) {
                fLogMap.put((String) entry.getKey(), (String) entry.getValue());
            }
        }
    }

    /**
     * w肳ꂽOt@CQ݂邩ǂO܂B
     * 
     * ΍쐬܂B
     * 
     * @throws IOException
     */
    void ensureLogFilesExist() throws IOException {
        // Ot@Cɂđ݂悤ɔOB
        ensureLogFileExist(fLogDir, "severe");

        for (Entry<String, String> entry : fLogMap.entrySet()) {
            ensureLogFileExist(fLogDir, (String) entry.getValue());
        }
    }

    /**
     * w肳ꂽOt@C݂邩ǂO܂B
     * 
     * ΍쐬܂B
     * 
     * @param logDir
     * @param logFileName
     * @throws IOException
     */
    void ensureLogFileExist(final File logDir, String logFileName)
            throws IOException {
        if (logFileName == null) {
            logFileName = "null";
        }

        final File file = new File(logDir, getLogFileName(logFileName));
        if (file.exists() == false) {
            try {
                if (file.createNewFile() == false) {
                    throw new IllegalArgumentException(fMsg.getMblgch07(file
                            .getAbsolutePath()));
                }
            } catch (IOException ex) {
                throw new IllegalArgumentException(fMsg.getMblgch08(file
                        .getAbsolutePath(), ex.toString()), ex);
            }
        }
    }

    /**
     * Õt@C肵܂B
     * 
     * @param logFileName
     *            {ƂȂt@CB
     * @return
     */
    String getLogFileName(String logFileName) {
        synchronized (sdFormat) {
            final Date now = new Date();
            return logFileName + ".log-" + sdFormat.format(now);
        }
    }
}