/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2011 Toshiki IGA
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2011 Toshiki IGA and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      Toshiki IGA - initial API and implementation
 *******************************************************************************/
package blanco.eclipseast2cg;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;

import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.IExtendedModifier;
import org.eclipse.jdt.core.dom.ImportDeclaration;
import org.eclipse.jdt.core.dom.Javadoc;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.PackageDeclaration;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.TagElement;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;

import blanco.cg.BlancoCgObjectFactory;
import blanco.cg.valueobject.BlancoCgClass;
import blanco.cg.valueobject.BlancoCgException;
import blanco.cg.valueobject.BlancoCgField;
import blanco.cg.valueobject.BlancoCgInterface;
import blanco.cg.valueobject.BlancoCgMethod;
import blanco.cg.valueobject.BlancoCgParameter;
import blanco.cg.valueobject.BlancoCgReturn;
import blanco.cg.valueobject.BlancoCgSourceFile;
import blanco.commons.util.BlancoNameUtil;
import blanco.eclipseast2cg.util.BlancoEclipseASTNameUtil;
import blanco.eclipseast2cg.util.BlancoEclipseASTUtil;

public class BlancoEclipseAST2CgVisitor extends ASTVisitor {
    static final boolean IS_DEBUG = false;

    final BlancoCgObjectFactory cgFactory = BlancoCgObjectFactory.getInstance();
    final BlancoCgSourceFile cgSource;

    final List<String> importList = new ArrayList<String>();

    final Stack<Object> objStack = new Stack<Object>();

    public BlancoEclipseAST2CgVisitor(final String sourcePackage) {
        cgSource = cgFactory.createSourceFile(sourcePackage, "Eclipse AST により解析しました。");
    }

    public BlancoCgSourceFile getCgSourceFile() {
        return cgSource;
    }

    public BlancoCgClass getCgClass() {
        for (int index = objStack.size() - 1; index >= 0; index--) {
            final Object lookup = objStack.get(index);
            if (lookup instanceof BlancoCgClass)
                return (BlancoCgClass) lookup;
        }
        return null;
    }

    public BlancoCgInterface getCgInterface() {
        for (int index = objStack.size() - 1; index >= 0; index--) {
            final Object lookup = objStack.get(index);
            if (lookup instanceof BlancoCgInterface)
                return (BlancoCgInterface) lookup;
        }
        return null;
    }

    public BlancoCgMethod getCgMethod() {
        for (int index = objStack.size() - 1; index >= 0; index--) {
            final Object lookup = objStack.get(index);
            if (lookup instanceof BlancoCgMethod)
                return (BlancoCgMethod) lookup;
        }
        return null;
    }

    // ////////////////////////////////////////////
    // visit メソッド群

    @Override
    public boolean visit(final PackageDeclaration node) {
        if (IS_DEBUG)
            System.out.println("visit(PackageDeclaration): " + node.getName().toString());

        cgSource.setPackage(node.getName().toString());
        return super.visit(node);
    }

    @Override
    public boolean visit(final ImportDeclaration node) {
        if (IS_DEBUG)
            System.out.println("visit(ImportDeclaration): " + node.getName());

        importList.add(node.getName().getFullyQualifiedName());

        return super.visit(node);
    }

    @Override
    public boolean visit(final TypeDeclaration node) {
        if (IS_DEBUG)
            System.out.println("visit(TypeDeclaration): " + node.getName());

        if (node.isInterface()) {
            final BlancoCgInterface cgInterface = cgFactory.createInterface(node.getName().toString(), null);
            cgSource.getInterfaceList().add(cgInterface);
            objStack.push(cgInterface);

            // FIXME JavaDoc 対応が未実施。

            parseAnnotation(node.modifiers(), cgInterface.getAnnotationList());
        } else {
            final BlancoCgClass cgClass = cgFactory.createClass(node.getName().toString(), null);
            cgSource.getClassList().add(cgClass);
            objStack.push(cgClass);

            // static クラスに未対応。
            cgClass.setFinal((node.getModifiers() & Modifier.FINAL) != 0);

            cgSource.setDescription("このクラスは '" + cgClass.getName() + "' の具象クラスとして blanco Framework によって自動生成されました。");

            // JavaDoc 展開
            final Javadoc javaDoc = node.getJavadoc();
            if (javaDoc != null) {
                @SuppressWarnings("unchecked")
                final List<TagElement> tagList = javaDoc.tags();
                for (TagElement tag : tagList) {
                    if (tag.getTagName() == null) {
                        // 通常のコメントについて、まずここで JavaDoc に展開します。
                        cgClass.getLangDoc().getDescriptionList().addAll(BlancoEclipseASTUtil.getText(tag));
                    }
                }
            }

            parseAnnotation(node.modifiers(), cgClass.getAnnotationList());
        }

        return super.visit(node);
    }

    @Override
    public void endVisit(final TypeDeclaration node) {
        if (IS_DEBUG)
            System.out.println("endVisit(TypeDeclaration): " + node.getName());

        objStack.pop();

        super.endVisit(node);
    }

    @Override
    public boolean visit(final MethodDeclaration node) {
        if (IS_DEBUG)
            System.out.println("visit(MethodDeclaration)");

        final BlancoCgMethod cgMethod = cgFactory.createMethod(node.getName().toString(), null);

        cgMethod.setStatic((node.getModifiers() & Modifier.STATIC) != 0);
        cgMethod.setFinal((node.getModifiers() & Modifier.FINAL) != 0);
        // デフォルトで「なし」をセットします。
        cgMethod.setAccess("");
        if ((node.getModifiers() & Modifier.PRIVATE) != 0) {
            cgMethod.setAccess("private");
        }
        if ((node.getModifiers() & Modifier.PROTECTED) != 0) {
            cgMethod.setAccess("protected");
        }
        if ((node.getModifiers() & Modifier.PUBLIC) != 0) {
            cgMethod.setAccess("public");
        }

        // JavaDoc 展開
        final Javadoc javaDoc = node.getJavadoc();
        if (javaDoc != null) {
            @SuppressWarnings("unchecked")
            final List<TagElement> tagList = javaDoc.tags();
            for (TagElement tag : tagList) {
                if (tag.getTagName() == null) {
                    // 通常のコメントについて、まずここで JavaDoc に展開します。
                    cgMethod.getLangDoc().getDescriptionList().addAll(BlancoEclipseASTUtil.getText(tag));
                }
            }
        }

        // クラスかインタフェースかの区別を実施。
        if (objStack.peek() instanceof BlancoCgClass) {
            final BlancoCgClass cgClass = getCgClass();
            cgClass.getMethodList().add(cgMethod);
        } else if (objStack.peek() instanceof BlancoCgInterface) {
            final BlancoCgInterface cgInterface = getCgInterface();
            cgInterface.getMethodList().add(cgMethod);
        } else {
            System.out.println("Method in not class or interface: " + cgMethod.getName());
        }

        objStack.push(cgMethod);

        parseAnnotation(node.modifiers(), cgMethod.getAnnotationList());

        for (Object obj : node.parameters()) {
            if (obj instanceof SingleVariableDeclaration) {
                final SingleVariableDeclaration varDec = (SingleVariableDeclaration) obj;
                final BlancoCgParameter cgParam = cgFactory.createParameter(
                        varDec.getName().toString(),
                        BlancoEclipseASTNameUtil.toFullyQualifiedClassName(varDec.getType().toString(),
                                cgSource.getPackage(), importList), null);
                cgMethod.getParameterList().add(cgParam);

                // パラメータのアノテーションを記憶します。
                parseAnnotation(varDec.modifiers(), cgParam.getAnnotationList());
            } else {
                throw new IllegalArgumentException("未対応:" + obj.getClass());
            }
        }

        final Type type = node.getReturnType2();
        if (type != null) {
            final BlancoCgReturn cgReturn = cgFactory.createReturn(BlancoEclipseASTNameUtil.toFullyQualifiedClassName(
                    type.toString(), cgSource.getPackage(), importList), null);
            cgMethod.setReturn(cgReturn);
        } else {
            // コンストラクタの場合、戻り値はありません。
        }

        for (Object obj : node.thrownExceptions()) {
            if (obj instanceof SimpleName) {
                final SimpleName name = (SimpleName) obj;

                final BlancoCgException cgException = cgFactory.createException(
                        BlancoEclipseASTNameUtil.toFullyQualifiedClassName(name.getFullyQualifiedName(),
                                cgSource.getPackage(), importList), null);
                cgMethod.getThrowList().add(cgException);
            } else if (obj instanceof QualifiedName) {
                final QualifiedName name = (QualifiedName) obj;

                final BlancoCgException cgException = cgFactory.createException(name.getFullyQualifiedName(), null);
                cgMethod.getThrowList().add(cgException);
            } else {
                throw new IllegalArgumentException("未対応:" + obj.getClass());
            }
        }

        if (javaDoc != null) {
            @SuppressWarnings("unchecked")
            final List<TagElement> tagList = javaDoc.tags();
            for (TagElement tag : tagList) {
                if (tag.getTagName() != null) {
                    // タグ名付きのコメントについて、展開します。
                    if ("@param".equals(tag.getTagName())) {
                        for (BlancoCgParameter lookupCgParameter : cgMethod.getParameterList()) {
                            if (lookupCgParameter.getName().equals(BlancoEclipseASTUtil.getSimpleNameIdentifier(tag))) {
                                lookupCgParameter.setDescription(BlancoEclipseASTUtil.list2String(BlancoEclipseASTUtil
                                        .getText(tag)));
                            }
                        }
                    } else if ("@return".equals(tag.getTagName())) {
                        cgMethod.getReturn().setDescription(
                                BlancoEclipseASTUtil.list2String(BlancoEclipseASTUtil.getText(tag)));
                    } else if ("@throws".equals(tag.getTagName())) {
                        for (BlancoCgException lookupException : cgMethod.getThrowList()) {
                            if (BlancoNameUtil.trimJavaPackage(lookupException.getType().getName()).equals(
                                    BlancoEclipseASTUtil.getSimpleNameIdentifier(tag))) {
                                lookupException.setDescription(BlancoEclipseASTUtil.list2String(BlancoEclipseASTUtil
                                        .getText(tag)));
                            }
                        }
                    } else {
                        // それ以外は無視します。
                    }
                }
            }
        }

        return super.visit(node);
    }

    public void endVisit(final MethodDeclaration node) {
        if (IS_DEBUG)
            System.out.println("endVisit(MethodDeclaration)");

        objStack.pop();

        super.endVisit(node);
    }

    @Override
    public boolean visit(final FieldDeclaration node) {
        if (IS_DEBUG)
            System.out.println("visit(FieldDeclaration)");

        final BlancoCgField cgField = cgFactory.createField(node.toString(), BlancoEclipseASTNameUtil
                .toFullyQualifiedClassName(node.getType().toString(), cgSource.getPackage(), importList), null);
        final BlancoCgClass cgClass = getCgClass();
        cgClass.getFieldList().add(cgField);

        cgField.setStatic((node.getModifiers() & Modifier.STATIC) != 0);
        cgField.setFinal((node.getModifiers() & Modifier.FINAL) != 0);
        // デフォルトで「なし」をセットします。
        cgField.setAccess("");
        if ((node.getModifiers() & Modifier.PRIVATE) != 0) {
            cgField.setAccess("private");
        }
        if ((node.getModifiers() & Modifier.PROTECTED) != 0) {
            cgField.setAccess("protected");
        }
        if ((node.getModifiers() & Modifier.PUBLIC) != 0) {
            cgField.setAccess("public");
        }

        for (Object obj : node.fragments()) {
            if (obj instanceof VariableDeclarationFragment) {
                VariableDeclarationFragment valFrag = (VariableDeclarationFragment) obj;
                // フィールド名をここで上書きします。
                cgField.setName(valFrag.getName().toString());

                if (valFrag.getInitializer() != null) {
                    // デフォルト値の読み込み。
                    cgField.setDefault(valFrag.getInitializer().toString());
                }
            } else {
                System.out.println("  フィールド分解: 処理しなかった型:" + obj.getClass().toString());
            }
        }

        // JavaDoc 展開
        final Javadoc javaDoc = node.getJavadoc();
        if (javaDoc != null) {
            @SuppressWarnings("unchecked")
            final List<TagElement> tagList = javaDoc.tags();
            for (TagElement tag : tagList) {
                if (tag.getTagName() == null) {
                    // 通常のコメントについて、まずここで JavaDoc に展開します。
                    cgField.getLangDoc().getDescriptionList().addAll(BlancoEclipseASTUtil.getText(tag));
                }
            }
        }

        parseAnnotation(node.modifiers(), cgField.getAnnotationList());

        return super.visit(node);
    }

    @Override
    public void endVisit(final FieldDeclaration node) {
        if (IS_DEBUG)
            System.out.println("endVisit(FieldDeclaration)");

        super.endVisit(node);
    }

    void parseAnnotation(final List<IExtendedModifier> modifiers, final List<String> result) {

        for (IExtendedModifier modifier : modifiers) {
            if (modifier.isAnnotation() == false)
                continue;

            String annotation = modifier.toString();

            annotation = annotation.substring(1);
            result.add(annotation);
        }
    }
}
