/*
 * Buggy Language (BL) GCC toy Frontend.
 *
 * Copyright (C) 2007, 2008 - WAKATSUKI toshihiro <alohakun _AT_ gmail.com>
 *                     2008 - SATO daisuke (QLeap)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/* Some code and ideas were taken from a hello-world gcc frontend
 * http://svn.gna.org/viewcvs/gsc/branches/hello-world/
 * the C frontend and gcc/treelang sample frontend.
 * */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "toplev.h"
#include "ggc.h"
#include "cgraph.h"
#include "tree-gimple.h"
#include "langhooks-def.h"
#include "langhooks.h"
#include "bl.h"

/* from gcc 4.3.0 treelang */
#include "debug.h"

static GTY(()) tree signed_and_unsigned_types[MAX_BITS_PER_WORD + 1][2];

static tree bl_type_for_size(unsigned precision, int unsignedp) {
  tree t;
  if(precision <= MAX_BITS_PER_WORD
      && signed_and_unsigned_types[precision][unsignedp] != 0)
    return signed_and_unsigned_types[precision][unsignedp];
  if(unsignedp)
    t = signed_and_unsigned_types[precision][1] = make_unsigned_type(precision);
  else
    t = signed_and_unsigned_types[precision][0] = make_signed_type(precision);
  return t;
}

static tree bl_type_for_mode(enum machine_mode mode, int unsignedp) {
  if(SCALAR_INT_MODE_P(mode))
    return bl_type_for_size(GET_MODE_BITSIZE(mode), unsignedp);
  else
    return NULL_TREE;
}

struct binding_level {
  tree names;
  tree blocks;
  tree stmts;
  struct binding_level *level_chain;
};

static struct binding_level *current_binding_level = NULL;
static struct binding_level *global_binding_level = NULL;
static struct binding_level clear_binding_level = {NULL, NULL, NULL, NULL };

static tree* getstmtlist(void) {
  return &current_binding_level->stmts;
}

/* need for expanding BIND_EXPR */
static void insert_block(tree block ATTRIBUTE_UNUSED) {
  puts("enter insert_block()");
  gcc_unreachable();
}

static int global_bindings_p(void) {
  return current_binding_level == global_binding_level ? true : false;
}

static void push_level(void) {
  struct binding_level *newlevel = xmalloc(sizeof(struct binding_level));
  *newlevel = clear_binding_level;
  newlevel->level_chain = current_binding_level;
  current_binding_level = newlevel;
  current_binding_level->stmts = alloc_stmt_list ();
}

static tree pushdecl(tree decl) {
  /* when auto (stack allocated) variable */
  if(TREE_CODE(decl) == VAR_DECL
      && !DECL_EXTERNAL(decl)
      && !TREE_STATIC(decl)
      && !TREE_PUBLIC(decl)) {
    TREE_CHAIN(decl) = BLOCK_VARS(current_binding_level->blocks);
    BLOCK_VARS(current_binding_level->blocks) = decl;
    tree code = build1(DECL_EXPR, void_type_node, decl);
    TREE_USED(code) = true;
    TREE_SIDE_EFFECTS(code) = true;
    append_to_statement_list(code, getstmtlist());
  } else
    append_to_statement_list(decl, getstmtlist());
  return decl;
}

static tree getdecls(void) {
  return current_binding_level->names;
}

#define MAX_LINE_SIZE 1024
static char current_line[MAX_LINE_SIZE];
static unsigned int current_lineno = 1;

static FILE *bl_input_file;

static bool bl_init(void) {

  bl_common_nodes_and_builtins ();

  if(!main_input_filename ||
      main_input_filename[0] == ' ' ||
      !main_input_filename[0]) {
    return false;
  }

  bl_input_file = fopen(main_input_filename, "r");

  if(!bl_input_file) {
    fprintf (stderr, "Unable to open input file %s\n", main_input_filename);
    exit (1);
  }

  if(!fgets(current_line, MAX_LINE_SIZE, bl_input_file)) {
    fprintf (stderr, "Input file %s is empty\n", main_input_filename);
    exit (1);
  }

  /* create toplevel */
  push_level();
  global_binding_level = current_binding_level;
  current_binding_level->level_chain = NULL;

  current_function_decl         = NULL_TREE;
  set_cfun(NULL); // gcc 4.3.0
  return true;
}

static void bl_finish(void) {
  fclose(bl_input_file);
}

#include <ctype.h>

static tree parse_integer_value(char* string, unsigned int length) {

  long long int val = 0;
  unsigned int i = 0;
  unsigned int signed_p = 0;
  int negative = 1;
  switch(string[0]) {
    case (unsigned char)'-':
      negative = -1;
      signed_p = i = 1;
      break;

    case (unsigned char)'+':
      signed_p = i = 1;
      break;

    default:
      break;
  }
  for (; i < length; i++)
    val = val * 10 + string[i] - (unsigned char)'0';
  val = val * negative;
  return build_int_cst_wide (signed_p == 1 ?
      integer_type_node : unsigned_type_node,
      val & 0xffffffff, (val >> 32) & 0xffffffff);
}

#define BUFSIZE 256

static tree next_tok(char **string) {
  static char buf[BUFSIZE];
  char c;
  int i = 0;

  while((c = **string) == ' ' || c == '\t' || c == '\r')
    (*string)++;

  if((c = **string) == '\n' || c == '\0')
    return NULL_TREE;

  if(**string == '(') {
    (*string)++;
    printf("[(] : id\n");
    return get_identifier("(");
  }

  if(**string == ')') {
    (*string)++;
    printf("[)] : id\n");
    return get_identifier(")");
  }
  
  if(**string == ',') {
    (*string)++;
    printf("[,] : id\n");
    return get_identifier(",");
  }

  if(**string == '"') {
    (*string)++;
    while(**string != '"') {
      buf[i++] = **string;
      (*string)++;
    }
    (*string)++;
    buf[i] = '\0';
    printf("\"%s\"\n", buf);
    return build_string_literal(strlen(buf) + 1, buf);
  } else {
    while((c = **string) != ' '
        && c != '\n'
        && c != '\t'
        && c != '\r'
        && c != '\0'
        && c != ','
        && c != '('
        && c != ')') {
      buf[i++] = **string;
      (*string)++;
    }
  }
  gcc_assert(i < BUFSIZE);
  buf[i] = '\0';
  printf("[%s]", buf);
  if((((c = buf[0]) == '+' || c == '-') && isdigit(buf[1]))
      || isdigit(c)) {
    printf(" : integer\n");
    return parse_integer_value(buf, strlen(buf));
  } else {
    printf(" : id\n");
    return get_identifier(buf);
  }
}

#undef BUFSIZE

static tree tok = NULL_TREE;
static tree prev_tok = NULL_TREE;

static tree next(void) {
  static char *last = current_line;

  if(prev_tok) {
    tok = prev_tok;
    prev_tok = NULL;
    return tok;
  }

  tok = next_tok(&last);

  if(!tok) {
    while(fgets(current_line, MAX_LINE_SIZE, bl_input_file)) {
      current_lineno++;
      if(current_line[0] == '\n')
        continue;
      last = current_line;
      printf("[EOL] (NULL_TREE)\n");
      return NULL_TREE;
    }
    printf("[EOF] (id_EOF)\n");
    return get_identifier("EOF");
  }
  return tok;
}

static void pushback(void) {
  prev_tok = tok;
}

static int check(const char *id) {
  tree t = next();
  pushback();
  if(get_identifier(id) == t)
    return true;
  else
    return false;
}

static int next_p(void) {
  tree t = next();
  pushback();
  if(!t || t == get_identifier("EOF"))
    return false;
  return true;
}

static void expect(const char *tokname) {
  tree t = next();
  if(get_identifier(tokname) != t) {
    if(TREE_CODE(t) == IDENTIFIER_NODE) {
      fprintf(stderr, "error : expected %s, but %s\n", tokname,
          IDENTIFIER_POINTER(t));
    } else
      fprintf(stderr, "error : expected %s, but invalid token\n", tokname);
    exit(1);
  }
}

static tree lookup_variable(tree name) {
  struct binding_level* cl = current_binding_level;
  do {
    tree names = cl->names;
    while(names) {
      if(name == DECL_NAME(names))
        return names;
      names = TREE_CHAIN(names);
    }
  } while((cl = cl->level_chain));
  fprintf(stderr, "warning : variable %s can't find in this scope.\n",
      IDENTIFIER_POINTER(name));
  return NULL_TREE;
}

  static tree eval_value(tree t) {
    if(TREE_CODE(t) == IDENTIFIER_NODE)
      return lookup_variable(t);
    return t;
  }

static enum tree_code get_operator(tree t) {
  if(get_identifier("+") == t)
    return PLUS_EXPR;
  else if(get_identifier("-") == t)
    return MINUS_EXPR;
  else if(get_identifier("*") == t)
    return MULT_EXPR;
  else if(get_identifier("/") == t)
    return TRUNC_DIV_EXPR;
  else if(get_identifier("%") == t)
    return TRUNC_MOD_EXPR;
  else {
    fprintf(stderr, "Invalid operator : %s\n",  IDENTIFIER_POINTER(t));
    gcc_unreachable();
  }
}

static void bl_finalize_function(tree fn_decl) {
  struct cgraph_node *cgn = cgraph_node(fn_decl);
  cgraph_finalize_function(fn_decl, true);
  if(cgn) {
    // finalize all nested functions recursively
    for(cgn = cgn->nested; cgn ; cgn = cgn->next_nested) {
      bl_finalize_function(cgn->decl);
    }
  }
}

/*
 *  Program ::= Def*
 *
 *  // global definition
 *  Def ::= id = Expr ('\n' | EOF)                 // variable definition
 *        | id = fun (argtype id)* -> rettype '\n' // function definition
 *                 Stmt*
 *               nuf ('\n' | EOF)
 *        | id :: argtype* -> rettype '\n'    // function prototype
 *
 *  Stmt ::= return Expr '\n'                      // return statement
 *         | id <- Expr '\n'                       // assignment statement
 *         | Def                                   // local definition
 *         | if Expr '\n'                          // if statement
 *              Stmt*
 *           else
 *              Stmt*
 *           if '\n'
 *
 *  Expr ::= Term + Term
 *         | Term - Term
 *         | Term
 *
 *  Term ::= Fact * Fact
 *         | Fact / Fact
 *         | Fact % Fact
 *         | Fact
 *
 *  Fact ::= id '(' {Expr} (,Expr)*  ')'           // function call
 *         | '(' Expr ')'
 *         | id                                    // variable
 *         | integer
 *         | ! Fact                                //  not
 */

static tree parse_expr(void);

static tree parse_fact(void) {
  tree T1 = next();

  // function call : id '(' {Expr} (, Expr)* ')'
  if(TREE_CODE(T1) == IDENTIFIER_NODE &&
      T1 != get_identifier("!") &&
      check("(")) {
    tree fn_decl = lookup_variable(T1);
    if(TREE_CODE(fn_decl) != FUNCTION_DECL) {
      fprintf(stderr, "error : %s is not function type node.\n",
          IDENTIFIER_POINTER(T1));
      exit(1);
    }
    tree args = NULL_TREE;
    next();
    if(!check(")")) {
      args = tree_cons(NULL_TREE, parse_expr(), NULL_TREE);
      while(check(",")) {
        next();
        args = tree_cons(NULL_TREE, parse_expr(), args);
      }
      expect(")");
    }
    args = nreverse(args);
    return build_function_call_expr(fn_decl, args);
  }

  if(T1 == get_identifier("!")) {
    tree exp = parse_fact();
    return build1(BIT_NOT_EXPR,TREE_TYPE(exp),exp);
  }

  if(T1 == get_identifier("(")) {
    tree t = parse_expr();
    expect(")");
    return t;
  }

  return eval_value(T1);
}

static tree parse_term(void) {
  tree T1 = parse_fact();

  while(next_p()) {
    if(!check("*")
        && !check("/")
        && !check("%")) {
      break;
    }
    tree op = next();
    tree T2 = parse_fact();
    T1 = fold_build2(get_operator(op), integer_type_node, T1,T2);
  }
  return T1;
}

static tree parse_expr(void) {
  tree T1 = parse_term();

  while(next_p()) {
    if(!check("+")
        && !check("-")) {
      break;
    }
    tree op = next();
    tree T2 = parse_term();
    T1 = fold_build2(get_operator(op), integer_type_node, T1,T2);
  }
  return T1;
}

static tree parse_stmt(void) {
  tree T1 = next();
  if(!T1
      || T1 == get_identifier("nuf")
      || T1 == get_identifier("else")
      || T1 == get_identifier("fi")
      || T1 == get_identifier("EOF"))
    return NULL_TREE;

  if(get_identifier("if") == T1) {
    fprintf(stderr, "enter if statement\n");
    tree exp = parse_expr();
    tree cond_exp = fold_build2(NE_EXPR, boolean_type_node, exp,
                                fold_build1(CONVERT_EXPR, TREE_TYPE(exp),
                                            integer_zero_node));

    // allocate new binding level and stmtlist for then stmts
    push_level();
    while(parse_stmt())
      ;
    next(); // discard EOL
    tree then_stmts = *getstmtlist();
    current_binding_level = current_binding_level->level_chain;

    // allocate new binding level and stmtlist for else stmts
    push_level();
    while(parse_stmt())
        ;
    next(); // discard EOL
    tree else_stmts = *getstmtlist();
    current_binding_level = current_binding_level->level_chain;

    tree cond = build3(COND_EXPR, void_type_node, cond_exp,
                       then_stmts, else_stmts);
    
    append_to_statement_list(cond, getstmtlist());
    
    fprintf(stderr, "leave if statement\n");
    
    TREE_CHAIN(cond) = current_binding_level->names;
    current_binding_level->names = cond;
    return cond;
  }

  if(get_identifier("return") == T1) {
    fprintf(stderr, "enter return\n");
    tree value = parse_expr();
    tree setret = fold_build2(MODIFY_EXPR, integer_type_node,
        DECL_RESULT(current_function_decl),
        fold_build1(CONVERT_EXPR, integer_type_node, value));
    TREE_SIDE_EFFECTS(setret) = true;
    TREE_USED(setret) = true;
    tree ret = fold_build1(RETURN_EXPR, void_type_node, setret);
    pushdecl(ret);
    fprintf(stderr, "leave return\n");
    return ret;
  }
  // DEFINITION EXPR
  if(TREE_CODE(T1) == IDENTIFIER_NODE && check("::")) {
    next();
    tree id_TYPE      = next();
    tree parm_types   = NULL_TREE;
    tree result_type  = NULL_TREE;

    fprintf(stderr, "enter parse fun protorype\n");
    while(id_TYPE != get_identifier("->")) {
      tree parm_type = NULL_TREE;
      if(id_TYPE == get_identifier("Int")) {
        parm_type = integer_type_node;
      }
      if(id_TYPE == get_identifier("String")) {
        parm_type = build_pointer_type(char_type_node);
      }
      parm_types = tree_cons(NULL_TREE, parm_type, parm_types);

      id_TYPE = next();
      fprintf(stderr, "type : %s\n", IDENTIFIER_POINTER(id_TYPE));   
    }
    tree id_RES = next();
    if(id_RES == get_identifier("Int")) {
      result_type = integer_type_node;
    }
    if(id_RES == get_identifier("Void")) {
      result_type = void_type_node;
    }
    next();
    tree fn_decl = build_fn_decl(IDENTIFIER_POINTER(T1),
        build_function_type(result_type, parm_types));
    DECL_EXTERNAL(fn_decl)   = true;    
    TREE_CHAIN(fn_decl) = current_binding_level->names;
    current_binding_level->names = fn_decl;
    fprintf(stderr, "leave parse fun protorype\n");

    rest_of_decl_compilation(fn_decl, /* TOPLEVEL */true, /* AT END */false);
    return fn_decl;
  }

  if(TREE_CODE(T1) == IDENTIFIER_NODE && check("=")) {
    next();

    // TYPED VARIABLE DEFINITION BY INITIALIZER (e.g. x = 123)
    if(!check("fun")) {
      fprintf(stderr, "enter init\n");
      tree value = parse_expr();
      tree var_decl = build_decl(VAR_DECL, T1, TREE_TYPE(value));
      DECL_EXTERNAL(var_decl) = false;
      if(global_bindings_p()) {
        TREE_STATIC(var_decl) = true;
        TREE_PUBLIC(var_decl) = true;
      } else {
        TREE_STATIC(var_decl) = false;
        TREE_PUBLIC(var_decl) = false;
      }
      TREE_USED(var_decl) = true;
      DECL_INITIAL(var_decl) = value;
      DECL_CONTEXT(var_decl) = current_function_decl;
      TREE_CHAIN(var_decl) = current_binding_level->names;
      current_binding_level->names = var_decl;
      if(global_bindings_p())
        rest_of_decl_compilation(var_decl, /* TOP LEVEL */true,
            /* AT END OF COMPILATION */false);
      else
        pushdecl(var_decl);
      fprintf(stderr, "leave init\n");
      return var_decl;
    }

    // FUNCTION DEFINITION
    if(check("fun")) {
      next();
      fprintf(stderr, "enter fun def\n");
      tree fn_decl     = build_fn_decl(IDENTIFIER_POINTER(T1), NULL_TREE);
      tree arg_types   = NULL_TREE;
      tree result_type = NULL_TREE;
      tree parms       = NULL_TREE;
      tree id_TYPE     = next();
      tree id_PARM     = next();
      tree id_RES      = NULL_TREE;
      tree block = build_block(NULL_TREE,  // variables
          NULL_TREE,  // subblocks
          NULL_TREE,  // supercontext
          NULL_TREE); // next same level block

      DECL_EXTERNAL(fn_decl)   = false;
      DECL_ARTIFICIAL(fn_decl) = false;
      DECL_CONTEXT(fn_decl) = current_function_decl;
      TREE_STATIC(fn_decl)  = true;
      TREE_PUBLIC(fn_decl)  = true;
      if(current_function_decl) {
        fprintf(stderr, "enter nested function def\n");
        TREE_STATIC(fn_decl)  = false;
        TREE_PUBLIC(fn_decl)  = false;
        BLOCK_SUPERCONTEXT(block) = current_function_decl;
        BLOCK_SUBBLOCKS(current_binding_level->blocks) = block;
      }
      DECL_INITIAL(fn_decl) = block;
      
      TREE_CHAIN(fn_decl) = current_binding_level->names;
      current_binding_level->names = fn_decl;

      // enter new scope (in function)
      push_level();
      tree prev_fn_decl         = current_function_decl;
      current_function_decl = fn_decl; 
      current_binding_level->blocks = block;

      while(id_TYPE != get_identifier("->")) {
        fprintf(stderr, "enter parse fun arg\n");
        tree parm_type = NULL_TREE;
        if(id_TYPE == get_identifier("Int")) {
          parm_type = integer_type_node;
        }
        tree parm = build_decl(PARM_DECL, id_PARM, parm_type);
        DECL_ARG_TYPE(parm) = parm_type;
        TREE_CHAIN(parm) = parms;
        parms = parm;      
        DECL_CONTEXT(parm)  = fn_decl;
        TREE_CHAIN(parm) = current_binding_level->names;
        current_binding_level->names = parm;
        arg_types = tree_cons(NULL_TREE, parm_type, arg_types);

        id_TYPE = next();
        id_PARM = next();

        fprintf(stderr, "type : %s, parm : %s\n", IDENTIFIER_POINTER(id_TYPE),
            IDENTIFIER_POINTER(id_PARM));   
      }
      gcc_assert(!next());
      id_RES = id_PARM;
      if(id_RES == get_identifier("Int")) {
        result_type = integer_type_node;
        fprintf(stderr, "parse result : %s\n",
            IDENTIFIER_POINTER(id_RES));
      }
      tree result_decl = build_decl(RESULT_DECL, NULL_TREE, result_type);
      DECL_CONTEXT(result_decl) = fn_decl;
      DECL_RESULT(fn_decl) = result_decl;

      DECL_ARGUMENTS(fn_decl) = parms;
      TREE_TYPE(fn_decl) = build_function_type(result_type, arg_types);

      //debug_tree(fn_decl);

      // parse stmts in fun ... nuf
      fprintf(stderr, "enter stmts in %s()\n", IDENTIFIER_POINTER(T1));
      while(parse_stmt())
        ;
      next(); // discard EOL or EOF
      fprintf(stderr, "leave stmts in %s()\n", IDENTIFIER_POINTER(T1));

      DECL_SAVED_TREE(fn_decl) = build3(BIND_EXPR, void_type_node,
          BLOCK_VARS(block), *(getstmtlist()), block);
      allocate_struct_function(fn_decl, false); // for gcc 4.3.0
      gimplify_function_tree(fn_decl);

      if(!prev_fn_decl) {
        fprintf(stderr, "enter finalize %s()\n", IDENTIFIER_POINTER(T1));
        bl_finalize_function(fn_decl);
        fprintf(stderr, "leave finalize %s()\n", IDENTIFIER_POINTER(T1));
      } else
        fprintf(stderr, "leave nested function def\n");
      // pop fnction decl
      current_function_decl         = prev_fn_decl; 
      current_binding_level         = current_binding_level->level_chain;

      fprintf(stderr, "leave fun def\n");
      return fn_decl;
    }
  }
  // ASSIGNMENT (e.g. x <- 123)
  if(TREE_CODE(T1) == IDENTIFIER_NODE &&
      check("<-")) {
    next();
    fprintf(stderr, "enter assign\n");
    tree value = parse_expr();
    tree assign = fold_build2(MODIFY_EXPR, integer_type_node, eval_value(T1),
        fold_build1(CONVERT_EXPR, integer_type_node, value));
    TREE_SIDE_EFFECTS(assign) = true;
    TREE_USED(assign) = true;
    pushdecl(assign);
    fprintf(stderr, "leave assign\n");
    return assign;
  }

  debug_tree(T1);
  gcc_unreachable();
}

static void bl_parse_file(int debug ATTRIBUTE_UNUSED) {

  while(parse_stmt())
    ;

  cgraph_finalize_compilation_unit();
  cgraph_optimize();  
}

/************************* from treelang **********************************/

struct lang_identifier GTY(()) {
  struct tree_identifier common;
};

/* Language-specific tree node information.  */

union lang_tree_node 
  GTY((desc ("TREE_CODE (&%h.generic) == IDENTIFIER_NODE")))
{
  union tree_node GTY ((tag ("0"), 
			desc ("tree_node_structure (&%h)"))) 
    generic;
  struct lang_identifier GTY ((tag ("1"))) identifier;
};

/* Language-specific type information.  */

struct lang_type GTY(())
{
  char junk; /* dummy field to ensure struct is not empty */
};

/* Language-specific declaration information.  */

struct lang_decl GTY(())
{
  char junk; /* dummy field to ensure struct is not empty */
};

struct language_function GTY(())
{
  char junk; /* dummy field to ensure struct is not empty */
};

static unsigned int bl_init_options (unsigned int argc ATTRIBUTE_UNUSED,
    const char **argv ATTRIBUTE_UNUSED) {
  return CL_bl;
}

static int bl_handle_option(size_t      scode,
    const char  *arg    ATTRIBUTE_UNUSED,
    int         value   ATTRIBUTE_UNUSED) {
  enum opt_code code = (enum opt_code) scode;

  if(code == OPT_bye) {
    return 1;
  }
  return 0;
}

tree convert(tree type ATTRIBUTE_UNUSED, tree expr ATTRIBUTE_UNUSED) {
  puts("enter convert()");
  gcc_unreachable();
}

static bool bl_mark_addressable(tree exp) {
  switch (TREE_CODE(exp)){
    case STRING_CST:
      break;
    case FUNCTION_DECL:
      TREE_ADDRESSABLE(exp) = true;
      break;
    case ARRAY_REF:
      return bl_mark_addressable(TREE_OPERAND(exp, 0));
      break;
    default:
      puts("enter bl_marK_adressable()");
      gcc_unreachable();
      break;
  }
  return true;
}


/*****************************************************************************/

#undef  LANG_HOOKS_NAME
#define LANG_HOOKS_NAME                      "bl"

#undef  LANG_HOOKS_INIT
#define LANG_HOOKS_INIT                      bl_init

#undef  LANG_HOOKS_INIT_OPTIONS
#define LANG_HOOKS_INIT_OPTIONS              bl_init_options

#undef LANG_HOOKS_FINISH
#define LANG_HOOKS_FINISH                    bl_finish

#undef  LANG_HOOKS_HANDLE_OPTION
#define LANG_HOOKS_HANDLE_OPTION             bl_handle_option

#undef  LANG_HOOKS_PARSE_FILE
#define LANG_HOOKS_PARSE_FILE                bl_parse_file

#undef  LANG_HOOKS_MARK_ADDRESSABLE
#define LANG_HOOKS_MARK_ADDRESSABLE          bl_mark_addressable

#undef  LANG_HOOKS_TYPE_FOR_MODE
#define LANG_HOOKS_TYPE_FOR_MODE             bl_type_for_mode

#undef  LANG_HOOKS_TYPE_FOR_SIZE
#define LANG_HOOKS_TYPE_FOR_SIZE             bl_type_for_size

#undef LANG_HOOKS_COMMON_ATTRIBUTE_TABLE
#define LANG_HOOKS_COMMON_ATTRIBUTE_TABLE    bl_attribute_table

#undef LANG_HOOKS_FORMAT_ATTRIBUTE_TABLE
#define LANG_HOOKS_FORMAT_ATTRIBUTE_TABLE    bl_format_attribute_table

#undef LANG_HOOKS_CALLGRAPH_EXPAND_FUNCTION
#define LANG_HOOKS_CALLGRAPH_EXPAND_FUNCTION tree_rest_of_compilation

const struct lang_hooks lang_hooks = LANG_HOOKS_INITIALIZER;

#define DEFTREECODE(SYM, NAME, TYPE, LENGTH) TYPE,
const enum tree_code_class tree_code_type[] = {
#include "tree.def"
  tcc_exceptional
};
#undef DEFTREECODE

#define DEFTREECODE(SYM, NAME, TYPE, LENGTH) LENGTH,
const unsigned char tree_code_length[] = {
#include "tree.def"
  0
};
#undef DEFTREECODE

#define DEFTREECODE(SYM, NAME, TYPE, LEN) NAME,
const char *const tree_code_name[] = {
#include "tree.def"
  "@@dummy"
};
#undef DEFTREECODE

#include "gt-bl-bl1.h"
#include "gtype-bl.h"
