require 'kconv'
require 'rexml/streamlistener'

module BloggerPost
  module Common
    class ExcerptListener
      include REXML::StreamListener

      attr_reader :excerpt

      def initialize
        @excerpt = ''
        @in_body = false
      end

      def tag_start(name, attrs)
        @in_body = true if name == 'body'
      end

      def text(src)
        @excerpt << src if @in_body
      end
    end

    def make_excerpt(html)
      listener = ExcerptListener.new
      REXML::Document.parse_stream(html, listener)
      listener.excerpt
    end

    def make_tempfile_in(src)
      tempfile = Tempfile.open('bloggerpost_entry')
      tempfile.write(src)
      tempfile.close
      tempfile
    end

    def make_tempfile_out
      tempfile = Tempfile.open('bloggerpost_html')
      tempfile.close
      tempfile
    end

    def get_proxy_info(config, use_ssl)
      if use_ssl
        proxy_host = config.http_proxy_host
        proxy_port = config.http_proxy_port
        proxy_user =  config.http_proxy_user
        proxy_passwd =  config.http_proxy_passwd
      else
        proxy_host = config.https_proxy_host
        proxy_port = config.https_proxy_port
        proxy_user =  config.https_proxy_user
        proxy_passwd =  config.https_proxy_passwd
      end
      [proxy_host, proxy_port, proxy_user, proxy_passwd]
    end

    def http_request(config, url, req)
      use_ssl = url.scheme == 'https'
      proxy_host = nil
      proxy_port = nil
      proxy_user =  nil
      proxy_passwd =  nil
      if config.use_proxy
        proxy_host, proxy_port, proxy_user, proxy_passwd = get_proxy_info(config, use_ssl)
      end
      http = Net::HTTP.new(url.host, url.port, proxy_host, proxy_port, proxy_user, proxy_passwd)
      if use_ssl
        http.use_ssl = true
        http.verify_mode = OpenSSL::SSL::VERIFY_PEER
        store = OpenSSL::X509::Store.new
        store.set_default_paths
        http.cert_store = store
      end
      http.start {|session| session.request(req) }
    end

    def http_get(config, url)
      if config.use_proxy
        target_url = url.to_s
      else
        target_url = url.request_uri
      end
      http_request(config, url, Net::HTTP::Get.new(target_url))
    end

    def delete_extname(filename)
      filename.gsub(/#{File.extname(filename)}$/, '')
    end

    def is_filename(filename)
      return false unless filename
      !(filename =~ /^http:/) && File.exist?(filename)
    end

    def get_first_tag_text(html, tag)
      html =~ /<#{tag}( .*?>|>)(.*?)<\/#{tag}>/im
      $2
    end

    def get_body_text(html)
      get_first_tag_text(html, 'body')
    end

    def get_h1_text(html)
      get_first_tag_text(html, 'h1')
    end

    def get_title_text(html)
      get_first_tag_text(html, 'title')
    end

    def delete_first_tag(html, tag)
      html.sub(/<#{tag}( .*?>|>).*?<\/#{tag}>/im, '')
    end

    def delete_first_h1(html)
      delete_first_tag(html, 'h1')
    end

    def delete_first_a(html)
      delete_first_tag(html, 'a')
    end

    def to_internal_encode(s)
      return s if Kconv::UTF8 == internal_encode
      s.kconv(Kconv::UTF8, internal_encode)
    end

    def to_external_encode(s)
      return s if Kconv::UTF8 == internal_encode
      s.kconv(internal_encode, Kconv::UTF8)
    end

    private

    def internal_encode
      encode = Kconv::UTF8
      case ENV['LANG']
      when 'ja_JP.eucJP'
        encode = Kconv::EUC
      end
      encode
    end
  end

  extend BloggerPost::Common
end
