package bodybuilder.builder.value;

/**
 * 拡張値クラス<br>
 * <code>#{...}</code>に囲まれた拡張値からオブジェクトを生成する。
 */
public abstract class ExtendedValue {

    /**
     * 値が拡張値かどうかを調べる。
     * 
     * @param value 文字列
     * @return 拡張値の場合はtrue
     */
    public static boolean isExtended(String value) {
        if (value == null) {
            // 文字列がnullの場合はfalse。
            return false;
        }

        // 値をトリム。
        value = value.trim();

        if (value == null || value.length() < 4) {
            // 文字列がnull、または3文字以下の場合はfalse。
            return false;
        } else if (!value.startsWith("#{") || !value.endsWith("}")) {
            // 文字列が#{...}に囲まれていない場合はfalse。
            return false;
        }

        // #{...}に囲まれた値を取得。
        value = format(value);

        // 値がマップされているかどうかを返す。
        return ExtendedValueMapping.hasValue(value);
    }

    /**
     * 拡張値からオブジェクトを取得する。
     * 
     * @param value 拡張値
     * @return オブジェクト
     */
    public static Object getValue(String value) {
        // 値をトリム。
        value = value.trim();

        // #{...}に囲まれた値を取得。
        value = format(value);

        // 値にマップされた拡張値オブジェクトを取得。
        ExtendedValue extVal = ExtendedValueMapping.getExtendedValue(value);

        // 拡張値オブジェクトを取得できなかった場合はnullを返す。
        if (extVal == null) {
            return null;
        }

        // 拡張値オブジェクトからオブジェクトを取得。
        return extVal.value(value);
    }

    /**
     * 拡張値からオブジェクトを取得する。
     * 
     * @param value 拡張値
     * @return オブジェクト
     */
    protected abstract Object value(String value);

    /**
     * 拡張値の前後から<code>#{...}</code>を取り除く。
     * 
     * @param value 拡張値
     * @return 前後から<code>#{...}</code>を取り除いた文字列
     */
    private static String format(String value) {
        // 拡張値の前後から#{...}を取り除く。
        return value.substring(2, value.length() - 1);
    }

    /**
     * 拡張書式のエスケープを解除する。
     * 
     * @param str 文字列
     * @return アンエスケープした文字列
     */
    public static String unescapeExtendedValue(String str) {
        // エスケープした拡張書式ではない場合はそのまま返す。
        if (!str.matches("^ *\\\\+#\\{.+\\} *$")) {
            return str;
        }

        // 文字列をバッファに移す。
        StringBuffer buf = new StringBuffer(str);

        // 「\」を削除。
        int idx = buf.indexOf("\\");
        buf.deleteCharAt(idx);

        // アンエスケープした文字列を返す。
        return buf.toString();
    }

}