package bodybuilder.test.database;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import bodybuilder.util.DbUnitUtils;
import bodybuilder.util.FileUtils;
import bodybuilder.util.jdom.JDOMUtils;

/**
 * データベースアサーション
 */
public class DatabaseAssertion {

    static {
        try {
            // JDBCドライバを登録。
            DbUnitUtils.registerDriver(Config.getDatabaseDriver());
        } catch (Throwable e) {
            e.printStackTrace();
            throw new BodyBuilderException("failed to initialize '"
                    + DatabaseAssertion.class.getName() + "'.", e);
        }
    }

    /**
     * ファイル名
     */
    private String file = null;

    /**
     * データセットのタイプ
     */
    private String type = null;

    /**
     * 検査しないカラムのマップ
     */
    private Map ignore = null;

    /**
     * コンストラクタ
     * 
     * @param element XML要素
     * @param baseDir ベースディレクトリ
     */
    public DatabaseAssertion(Element element, String baseDir) {
        // ファイル名を取得。
        file = JDOMUtils.getRequiredAttrValue(element, "file");

        // ファイル名がフルパスではない場合、ファイル名にベースディレクトリをくっつける。
        if (!FileUtils.isFullPath(file)) {
            file = baseDir + File.separator + file;
        }

        // データセットのタイプを取得。
        type = element.getAttributeValue("type");

        // データセットのタイプを取得できない場合は拡張子から判定。
        if (type == null) {
            Map dataSetMap = Config.getDataSetMap();
            type = (String) dataSetMap.get(FileUtils.getExt(file));
        }

        // 検査しないカラムのマップを生成。
        ignore = buildIgnore(element);
    }

    /**
     * データセットとデータベースが等しいことを表明する。
     */
    public void assertDataSetEquals() {
        // データベースのプロパティを取得。
        String driver = Config.getDatabaseDriver();
        String url = Config.getDatabaseUrl();
        String user = Config.getDatabaseUser();
        String password = Config.getDatabasePassword();

        // データベースを検査。
        DbUnitUtils utils = new DbUnitUtils(url, user, password);
        utils.assertDataSetEquals(file, type, ignore);
    }

    /**
     * ファイル名を取得する。
     * 
     * @return ファイル名
     */
    public String getFile() {
        return file;
    }

    /**
     * データセットのタイプを取得する。
     * 
     * @return データセットのタイプ。
     */
    public String getType() {
        return type;
    }

    /**
     * 検査しないカラムのマップを取得する。
     * 
     * @return 検査しないカラムのマップ
     */
    public Map getIgnore() {
        return ignore;
    }

    /**
     * 検査しないカラムのマップを生成する。
     * 
     * @param element XML要素
     * @return 検査しないカラムのマップ
     */
    private Map buildIgnore(Element element) {
        Map ignore = new HashMap();
        // <ignore>を取得。
        List children = element.getChildren("ignore");

        // テーブル名と検査しないカラムのマップを作成。
        // テーブル名、カラム名はすべて大文字に変換。
        for (int i = 0; i < children.size(); i++) {
            Element child = (Element) children.get(i);
            String table = JDOMUtils.getRequiredAttrValue(child, "table");
            String[] colmuns = JDOMUtils.getRequiredAttrValue(child, "columns")
                    .split(",");

            for (int j = 0; j < colmuns.length; j++) {
                colmuns[j] = colmuns[j].trim().toUpperCase();
            }

            ignore.put(table.toUpperCase(), colmuns);
        }

        // マップを返す。
        return ignore;
    }

}