package bodybuilder.test.dicon;

import org.jdom.Element;

import bodybuilder.test.TestCaseXML;
import bodybuilder.test.common.Catch;
import bodybuilder.test.common.Execute;
import bodybuilder.test.common.Return;
import bodybuilder.test.database.DatabaseAssertion;
import bodybuilder.test.database.DatabaseSetUp;
import bodybuilder.util.jdom.JDOMUtils;

/**
 * DIConテストケースXML
 */
public class DIConTestCaseXML extends TestCaseXML {

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * セッターインジェクション(Type 2)
     */
    public static final int SETTER_INJECTION = 2;

    /**
     * コンストラクタインジェクション(Type 3)
     */
    public static final int CONSTRUCTOR_INJECTION = 3;

    /////////////////////////////////////////////////////////////////
    // variable field

    /**
     * ターゲット
     */
    protected String target = null;

    /**
     * インジェクションタイプ
     */
    private int type = SETTER_INJECTION;

    /**
     * 実行するメソッド
     */
    protected Execute execute = null;

    /**
     * 戻り値
     */
    protected Return ret = null;

    /**
     * catch
     */
    protected Catch catch_ = null;

    /**
     * インジェクションオブジェクト
     */
    protected InjectedObjects injectedObjects = null;

    /**
     * データベースセットアップ
     */
    protected DatabaseSetUp databaseSetUp = null;

    /**
     * 期待するインジェクションオブジェクト
     */
    protected InjectedObjects expectedObjects = null;

    /**
     * データベースアサーション
     */
    protected DatabaseAssertion databaseAssertion = null;

    /////////////////////////////////////////////////////////////////
    // build method

    /**
     * テストケースXMLを構築する。
     */
    protected void build() {
        super.build();
        // ターゲットを読み込む。
        loadTarget(root);
        // 入力値を読み込む。
        Element input = JDOMUtils.getRequiredChild(root, "input");
        loadInput(input);
        // 期待値を読み込む。
        Element expected = JDOMUtils.getRequiredChild(root, "expected");
        loadExpect(expected);
    }

    /**
     * ターゲットを読み込む。
     * 
     * @param root テストケース要素
     */
    protected void loadTarget(Element root) {
        // ターゲットを取得。
        Element target = JDOMUtils.getRequiredChild(root, "target");
        this.target = target.getTextTrim();
        // インジェクションタイプを取得。
        String type = target.getAttributeValue("type");
        loadType(type);
    }

    /**
     * インジェクションタイプを読み込む。
     * 
     * @param type インジェクションタイプ。
     */
    private void loadType(String type) {
        if ("2".equals(type)) {
            this.type = SETTER_INJECTION;
        } else if ("3".equals(type)) {
            this.type = CONSTRUCTOR_INJECTION;
        }
    }

    /**
     * 入力値を読み込む。
     * 
     * @param input 入力値要素
     */
    private void loadInput(Element input) {
        // 実行するメソッドを取得。
        Element execElem = JDOMUtils.getRequiredChild(input, "execute");
        execute = new Execute(execElem);
        // インジェクションオブジェクトを取得。
        Element compElem = input.getChild("component");

        if (compElem != null) {
            injectedObjects = new InjectedObjects(compElem, type);
        }

        // データベースセットアップを取得。
        Element databaseElem = input.getChild("database");

        if (databaseElem != null) {
            databaseSetUp = new DatabaseSetUp(databaseElem, file.getParent());
        }
    }

    /**
     * 期待値を読み込む。
     * 
     * @param expected 期待値要素
     */
    protected void loadExpect(Element expected) {
        // 戻り値を取得。
        Element retElem = expected.getChild("return");

        if (retElem != null) {
            ret = new Return(retElem);
        }

        // catchを取得。
        Element catchElem = expected.getChild("catch");

        if (catchElem != null) {
            catch_ = new Catch(catchElem);
        }

        // 期待するインジェクションオブジェクトを取得。
        Element component = expected.getChild("component");

        if (component != null) {
            expectedObjects = new InjectedObjects(component, type);
        }

        // データベースアサーションを取得。
        Element databaseElem = expected.getChild("database");

        if (databaseElem != null) {
            databaseAssertion = new DatabaseAssertion(databaseElem, file
                    .getParent());
        }
    }

    /////////////////////////////////////////////////////////////////
    // property

    /**
     * ターゲットを取得する。
     * 
     * @return ターゲット
     */
    public String getTarget() {
        return target;
    }

    /**
     * インジェクションタイプを取得する。
     * 
     * @return インジェクションタイプ
     */
    public int getType() {
        return type;
    }

    /**
     * 実行するメソッドを取得する。
     * 
     * @return 実行するメソッド
     */
    public Execute getExecute() {
        return execute;
    }

    /**
     * インジェクションオブジェクトを取得する。
     * 
     * @return インジェクションオブジェクト
     */
    public InjectedObjects getInjectedObjects() {
        return injectedObjects;
    }

    /**
     * データベースセットアップを取得する。
     * 
     * @return データベースセットアップ
     */
    public DatabaseSetUp getDatabaseSetUp() {
        return databaseSetUp;
    }

    /**
     * 戻り値を取得する。
     * 
     * @return 戻り値
     */
    public Return getReturn() {
        return ret;
    }

    /**
     * catchを取得する。
     * 
     * @return catch
     */
    public Catch getCatch() {
        return catch_;
    }

    /**
     * 期待するインジェクションオブジェクトを取得する。
     * 
     * @return 期待するインジェクションオブジェクト
     */
    public InjectedObjects getExpectedObjects() {
        return expectedObjects;
    }

    /**
     * データベースアサーションを取得する。
     * 
     * @return データベースアサーション
     */
    public DatabaseAssertion getDatabaseAssertion() {
        return databaseAssertion;
    }

}