/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * http://www.gnu.org/copyleft/gpl.html
 */

#include "stdafx.h"
#include "BonDecoder.h"

#include "TsPacketParser.h"
#include "TsDescrambler.h"
#include "PacketSink.h"

//////////////////////////////////////////////////////////////
// CBonDecoder

CBonDecoder::CBonDecoder()
	: emmProcess_(false), serviceID_(0), patGeneration_(false), transportStreamID_(0),
	pPacketParser_(0), pDescrambler_(0), pPacketSink_(0)
{
}

HRESULT CBonDecoder::FinalConstruct()
{
	assert(pPacketParser_ == 0);
	assert(pDescrambler_ == 0);
	assert(pPacketSink_ == 0);

	winSCardPath_ = _bstr_t();
	emmProcess_ = false;
	serviceID_ = 0;
	patGeneration_ = false;
	transportStreamID_ = 0;

	pPacketParser_ = 0;
	pDescrambler_ = 0;
	pPacketSink_ = 0;

	return S_OK;
}

void CBonDecoder::FinalRelease()
{
	if (pPacketSink_ != 0)
	{
		pDescrambler_->SetOutputDecoder(0);
		delete pPacketSink_;
		pPacketSink_ = 0;
	}
	if (pDescrambler_ != 0)
	{
		pPacketParser_->SetOutputDecoder(0);
		delete pDescrambler_;
		pDescrambler_ = 0;
	}
	if (pPacketParser_ != 0)
	{
		delete pPacketParser_;
		pPacketParser_ = 0;
	}

	winSCardPath_ = _bstr_t();
	emmProcess_ = false;
	serviceID_ = 0;
	patGeneration_ = false;
	transportStreamID_ = 0;
}

STDMETHODIMP CBonDecoder::InterfaceSupportsErrorInfo(REFIID riid)
{
	static const IID* const arr[] = 
	{
		&IID_IBonDecoder
	};

	for (int i=0; i < sizeof(arr) / sizeof(arr[0]); i++)
	{
		if (InlineIsEqualGUID(*arr[i],riid))
			return S_OK;
	}
	return S_FALSE;
}

///////////////////////////////////////////////////

HRESULT CBonDecoder::get_WinSCardPath(BSTR *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	*pVal = winSCardPath_.copy();
	return S_OK;
}

HRESULT CBonDecoder::put_WinSCardPath(BSTR newVal)
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ != 0)
	{
		return HRESULT_FROM_WIN32(ERROR_INVALID_STATE);
	}

	winSCardPath_ = newVal;
	return S_OK;
}

HRESULT CBonDecoder::get_EMMProcess(VARIANT_BOOL *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	*pVal = emmProcess_ ? VARIANT_TRUE : VARIANT_FALSE;
	return S_OK;
}

HRESULT CBonDecoder::put_EMMProcess(VARIANT_BOOL newVal)
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	emmProcess_ = newVal == VARIANT_TRUE;
	if (pDescrambler_ != 0)
	{
		pDescrambler_->EnableEmmProcess(emmProcess_);
	}

	return S_OK;
}

HRESULT CBonDecoder::get_ServiceID(USHORT *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	*pVal = serviceID_;
	return S_OK;
}

HRESULT CBonDecoder::put_ServiceID(USHORT newVal)
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	serviceID_ = newVal;
	if (pDescrambler_ != 0)
	{
		pDescrambler_->SetTargetServiceID(serviceID_);
	}

	return S_OK;
}

HRESULT CBonDecoder::get_PATGeneration(VARIANT_BOOL *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	*pVal = patGeneration_ ? VARIANT_TRUE : VARIANT_FALSE;
	return S_OK;
}

HRESULT CBonDecoder::put_PATGeneration(VARIANT_BOOL newVal)
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	patGeneration_ = newVal == VARIANT_TRUE;
	if (pPacketParser_ != 0)
	{
		pPacketParser_->EnablePATGeneration(patGeneration_);
	}

	return S_OK;
}

HRESULT CBonDecoder::get_TransportStreamID(USHORT *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	*pVal = transportStreamID_;
	return S_OK;
}

HRESULT CBonDecoder::put_TransportStreamID(USHORT newVal)
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	transportStreamID_ = newVal;
	if (pPacketParser_ != 0)
	{
		pPacketParser_->SetTransportStreamID(transportStreamID_);
	}

	return S_OK;
}

HRESULT CBonDecoder::get_Running(VARIANT_BOOL *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	*pVal =  (pDescrambler_ != 0) ? VARIANT_TRUE : VARIANT_FALSE;
	return S_OK;
}

HRESULT CBonDecoder::get_UsingInstruction(UsingInstructions *pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = UsingInstruction_Normal;
	}
	else
	{
		if (pDescrambler_->IsSSSE3Available() == true)
		{
			*pVal = UsingInstruction_SSSE3;
		}
		else if (pDescrambler_->IsSSE2Available() == true)
		{
			*pVal = UsingInstruction_SSE2;
		}
		else
		{
			*pVal = UsingInstruction_Normal;
		}
	}

	return S_OK;
}

HRESULT CBonDecoder::get_CardID(ULONGLONG* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = 0;
	}
	else
	{
		UCHAR rawid[8];
		if (pDescrambler_->GetBcasCardID(rawid) == false)
		{
			*pVal = 0;
		}
		else
		{
			ULONGLONG id = 0;
			for (int i = 0; i < 6; i++)
			{
				id <<= 8;
				id |= rawid[i];
			}
			*pVal = id;
		}
	}

	return S_OK;
}

HRESULT CBonDecoder::get_CardType(CardTypes* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = CardType_Invalid;
	}
	else
	{
		CBcasCard::BcasCardInfo info;
		if (pDescrambler_->GetBcasCardInfo(&info) == false)
		{
			*pVal = CardType_Invalid;
		}
		else
		{
			*pVal = static_cast<CardTypes>(info.CardType);
		}
	}

	return S_OK;
}

HRESULT CBonDecoder::get_CardManufacturerID(UCHAR* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = 0;
	}
	else
	{
		*pVal = static_cast<UCHAR>(pDescrambler_->GetBcasCardManufacturerID());
	}

	return S_OK;
}

HRESULT CBonDecoder::get_CardVersion(UCHAR* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = 0;
	}
	else
	{
		*pVal = static_cast<UCHAR>(pDescrambler_->GetBcasCardVersion());
	}

	return S_OK;
}

HRESULT CBonDecoder::get_CardReaderName(BSTR* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = 0;
	}
	else
	{
		*pVal = _bstr_t(pDescrambler_->GetCardReaderName()).Detach();
	}

	return S_OK;
}

HRESULT CBonDecoder::GetEcmPIDByServiceID(USHORT serviceID, USHORT* pVal)
{
	if (pVal == 0)
	{
		return E_POINTER;
	}

	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		 *pVal = 0;
	}
	else
	{
		*pVal = pDescrambler_->GetEcmPIDByServiceID(serviceID);
	}

	return S_OK;
}

HRESULT CBonDecoder::Start()
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pPacketSink_ != 0)
	{
		pDescrambler_->SetOutputDecoder(0);
		delete pPacketSink_;
	}
	if (pDescrambler_ != 0)
	{
		pDescrambler_->CloseBcasCard();
		pPacketParser_->SetOutputDecoder(0);
		delete pDescrambler_;
	}
	if (pPacketParser_ != 0)
	{
		delete pPacketParser_;
	}

	pPacketParser_ = new CTsPacketParser();
	pDescrambler_ = new CTsDescrambler();
	pPacketSink_ = new PacketSink();

	pPacketParser_->SetOutputDecoder(pDescrambler_);
	pDescrambler_->SetOutputDecoder(pPacketSink_);

	pPacketParser_->EnablePATGeneration(patGeneration_);
	pPacketParser_->SetTransportStreamID(transportStreamID_);

	if (winSCardPath_.length() >= 1)
	{
		if (pDescrambler_->OpenBcasCard(CCardReader::READER_SCARD_DYNAMIC, winSCardPath_) == false)
		{
			return E_FAIL;
		}
	}
	else
	{
		if (pDescrambler_->OpenBcasCard(CCardReader::READER_SCARD_DYNAMIC) == false)
		{
			return E_FAIL;
		}
	}

	if (pDescrambler_->IsSSSE3Available() == true)
	{
		pDescrambler_->SetInstruction(CTsDescrambler::INSTRUCTION_SSSE3);
	}
	else if (pDescrambler_->IsSSE2Available() == true)
	{
		pDescrambler_->SetInstruction(CTsDescrambler::INSTRUCTION_SSE2);
	}
	else
	{
		pDescrambler_->SetInstruction(CTsDescrambler::INSTRUCTION_NORMAL);
	}

	pDescrambler_->EnableEmmProcess(emmProcess_);
	pDescrambler_->SetTargetServiceID(serviceID_);
	pDescrambler_->EnableDescramble(true);

	return S_OK;
}

HRESULT CBonDecoder::Decode(VARIANT sourceData, ULONG maxResultSize, VARIANT *pRetVal)
{
	if (pRetVal == 0)
	{
		return E_POINTER;
	}

	if (V_VT(&sourceData) != (VT_ARRAY | VT_UI1))
	{
		return E_INVALIDARG;
	}

	SAFEARRAY *pSourceData = V_ARRAY(&sourceData);
	if (pSourceData == 0)
	{
		return E_INVALIDARG;
	}

	SAFEARRAY *pResultData = 0;
	const HRESULT hr = Decode(pSourceData, maxResultSize, &pResultData);
	if (FAILED(hr))
	{
		return hr;
	}

	V_VT(pRetVal) = VT_ARRAY | VT_UI1;
	V_ARRAY(pRetVal) = pResultData;

	return S_OK;
}

HRESULT STDMETHODCALLTYPE CBonDecoder::Decode(SAFEARRAY *pSourceData, ULONG maxResultSize, SAFEARRAY **ppRetVal)
{
	if (pSourceData == 0)
	{
		return E_POINTER;
	}
	if (ppRetVal == 0)
	{
		return E_POINTER;
	}

	*ppRetVal = 0;

	if (::SafeArrayGetDim(pSourceData) != 1)
	{
		return E_INVALIDARG;
	}

	LONG ubound, lbound;
	::SafeArrayGetUBound(pSourceData, 0, &ubound);
	::SafeArrayGetLBound(pSourceData, 0, &lbound);
	LONG sourceSize = ubound - lbound + 1;

	UCHAR *pRawSourceData = 0;
	::SafeArrayAccessData(pSourceData, reinterpret_cast<void **>(&pRawSourceData));

	SAFEARRAY *psa = ::SafeArrayCreateVector(VT_UI1, 0, maxResultSize);
	if (psa == 0)
	{
		return HRESULT_FROM_WIN32(::GetLastError());
	}

	UCHAR *pRawResultData = 0;
	::SafeArrayAccessData(psa, reinterpret_cast<void **>(&pRawResultData));

	ULONG resultSize = 0;
	const HRESULT hr = DecodeDirect(sourceSize, pRawSourceData, maxResultSize, &resultSize, pRawResultData);
	if (FAILED(hr))
	{
		::SafeArrayUnaccessData(psa);
		::SafeArrayDestroy(psa);
		::SafeArrayUnaccessData(pSourceData);
		return hr;
	}

	::SafeArrayUnaccessData(psa);
	::SafeArrayUnaccessData(pSourceData);

	*ppRetVal = psa;

	return S_OK;
}

HRESULT STDMETHODCALLTYPE CBonDecoder::DecodeDirect(
	ULONG sourceSize, UCHAR *pSourceData,
	ULONG maxResultSize, ULONG *pResultSize, UCHAR *pResultData)
{
	if (pSourceData == 0)
	{
		return E_POINTER;
	}
	if (pResultData == 0)
	{
		return E_POINTER;
	}
	if (pResultSize == 0)
	{
		return E_POINTER;
	}
	if (pResultData == 0)
	{
		return E_POINTER;
	}

	*pResultSize = 0;

	ATL::CComCritSecLock<_CritSec> lock(cs_);
	
	if (pDescrambler_ == 0)
	{
		return HRESULT_FROM_WIN32(ERROR_INVALID_STATE);
	}

	assert(pPacketParser_ != 0);
	assert(pPacketSink_ != 0);

	CMediaData data(pSourceData, sourceSize);
	pPacketParser_->InputMedia(&data);

	*pResultSize = pPacketSink_->Dequeue(pResultData, maxResultSize);

	return S_OK;
}

HRESULT CBonDecoder::Finish(ULONG maxResultSize, VARIANT *pRetVal)
{
	if (pRetVal == 0)
	{
		return E_POINTER;
	}

	SAFEARRAY *pResultData = 0;
	const HRESULT hr = Finish(maxResultSize, &pResultData);
	if (FAILED(hr))
	{
		return hr;
	}

	V_VT(pRetVal) = VT_ARRAY | VT_UI1;
	V_ARRAY(pRetVal) = pResultData;

	return S_OK;
}

HRESULT CBonDecoder::Finish(ULONG maxResultSize, SAFEARRAY **ppRetVal)
{
	if (ppRetVal == 0)
	{
		return E_POINTER;
	}

	*ppRetVal = 0;

	SAFEARRAY *psa = ::SafeArrayCreateVector(VT_UI1, 0, maxResultSize);
	if (psa == 0)
	{
		return HRESULT_FROM_WIN32(::GetLastError());
	}

	UCHAR *pRawResultData = 0;
	::SafeArrayAccessData(psa, reinterpret_cast<void **>(&pRawResultData));

	ULONG resultSize = 0;
	const HRESULT hr = FinishDirect(maxResultSize, &resultSize, pRawResultData);
	if (FAILED(hr))
	{
		::SafeArrayUnaccessData(psa);
		::SafeArrayDestroy(psa);
		return hr;
	}

	::SafeArrayUnaccessData(psa);

	*ppRetVal = psa;

	return S_OK;
}

HRESULT CBonDecoder::FinishDirect(ULONG maxResultSize, ULONG *pResultSize, UCHAR *pResultData)
{
	if (pResultSize == 0)
	{
		return E_POINTER;
	}
	if (pResultData == 0)
	{
		return E_POINTER;
	}

	*pResultSize = 0;

	ATL::CComCritSecLock<_CritSec> lock(cs_);
		
	if (pDescrambler_ == 0)
	{
		return HRESULT_FROM_WIN32(ERROR_INVALID_STATE);
	}

	assert(pPacketParser_ != 0);
	assert(pPacketSink_ != 0);

	*pResultSize = pPacketSink_->Dequeue(pResultData, maxResultSize);

	return pPacketSink_->Empty() ? S_OK : S_FALSE;
}

HRESULT CBonDecoder::Flush()
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);

	if (pDescrambler_ == 0)
	{
		return HRESULT_FROM_WIN32(ERROR_INVALID_STATE);
	}

	assert(pPacketParser_ != 0);

	pPacketParser_->ResetGraph();

	return S_OK;
}

HRESULT CBonDecoder::Reset()
{
	ATL::CComCritSecLock<_CritSec> lock(cs_);
	
	patGeneration_ = false;
	transportStreamID_ = 0;
	emmProcess_ = false;
	serviceID_ = 0;

	if (pDescrambler_ != 0)
	{
		assert(pPacketParser_ != 0);

		pPacketParser_->ResetGraph();

		pPacketParser_->EnablePATGeneration(patGeneration_);
		pPacketParser_->SetTransportStreamID(transportStreamID_);
		pDescrambler_->EnableEmmProcess(emmProcess_);
		pDescrambler_->SetTargetServiceID(serviceID_);
	}

	return S_OK;
}
