/* -*-tab-width:8;indent-tabs-mode:t;c-file-style:"bsd"-*- */
/*
 * rndis: USB Remote ndis driver for Windows Mobile
 *
 * Copyright (c) 2007,2008,2009 Hashi,Hiroaki <hashiz@meridiani.jp>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE. KEEP THE LOVE.
 *
 * $Id: if_rndis.c 352M 2007-08-31 01:30:42Z (ローカル) $
 */

#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/lock.h>
#include <sys/mbuf.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/lockmgr.h>
#include <sys/sockio.h>
#include <sys/endian.h>
#include <sys/mutex.h>
#include <sys/condvar.h>
#include <sys/sysctl.h>
#include <sys/sx.h>
#include <sys/taskqueue.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <net/if_media.h>
#include <net/ethernet.h>
#include <net/if_types.h>

#include <net/bpf.h>

#include <netinet/in.h>
#include <netinet/if_ether.h>

#include <sys/bus.h>
#include <machine/bus.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include <dev/usb/usbdi_util.h>
#include "usbdevs.h"

#define USB_DEBUG_VAR rndis_debug
#include <dev/usb/usb_debug.h>
#include <dev/usb/usb_process.h>
#include <dev/usb/net/usb_ethernet.h>
#include <dev/usb/usb_cdc.h>

#include <compat/ndis/cfg_var.h>
#include <compat/ndis/resource_var.h>
#include <compat/ndis/pe_var.h>
#include <compat/ndis/ntoskrnl_var.h>
#include <compat/ndis/ndis_var.h>

#include <sys/sysctl.h>

#include "if_rndisreg.h"

/*
 * sysctl
 */

SYSCTL_DECL(_hw_usb);

SYSCTL_NODE(_hw_usb, OID_AUTO, rndis, CTLFLAG_RW, 0, "USB rndis");

#ifdef USB_DEBUG
static int rndis_debug = 0;

SYSCTL_INT(_hw_usb_rndis, OID_AUTO, debug, CTLFLAG_RW, &rndis_debug, 0, "rndis debug level");
#endif

char rndis_version_string[] = "rndis version " RNDIS_VERSION;

/*
 * Supported devices are #defined below. 
 * These would normally be added into src/sys/dev/usb/usbdevs.
 */

#ifndef USB_PRODUCT_SHARP_WZERO3ADES_RNDIS
#define USB_PRODUCT_SHARP_WZERO3ADES_RNDIS	0x91ad
#endif

#ifndef UICLASS_MISC
#define UICLASS_MISC		0xef
#endif
#ifndef UIPROTO_CDC_VENDOR
#define UIPROTO_CDC_VENDOR	0xff
#endif

typedef u_char etheraddr_t[ETHER_ADDR_LEN] ;

#define RNDIS_IOBUF_SIZE   16384

#define RNDIS_CONFIG_NO	   1

/* New supported products go in here */

static const struct usb_device_id rndis_devs[] = {
	{USB_VPI(USB_VENDOR_SHARP, USB_PRODUCT_SHARP_WZERO3ADES_RNDIS, 0)},
};

/* attach/detach */
static device_probe_t rndis_probe;
static device_attach_t rndis_attach;
static device_detach_t rndis_detach;

/* usb */
static usb_callback_t rndis_bulk_read_callback;
static usb_callback_t rndis_bulk_write_callback;

/* ethernet */
static uether_fn_t rndis_attach_post;
static uether_fn_t rndis_init;
static uether_fn_t rndis_stop;
static uether_fn_t rndis_start;
static uether_fn_t rndis_tick;
static uether_fn_t rndis_setmulti;
static uether_fn_t rndis_setpromisc;

/* device control */
static int rndis_init_device(struct rndis_softc *);
//static int rndis_reset_device(struct rndis_softc *);
//static int rndis_halt_device(struct rndis_softc *);
static int rndis_get_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int *cnt);
static int rndis_set_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int cnt);
static int rndis_get_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int *listcnt);
static int rndis_set_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int listcnt);

/* ifnet */
static int  rndis_ifmedia_change(struct ifnet *);
static void rndis_ifmedia_status(struct ifnet *, struct ifmediareq *);

/* rndis */
static int  rndis_send_message(struct rndis_softc *, struct rndis_msg_snd *, int);
static int  rndis_recv_message(struct rndis_softc *, struct rndis_msg_rcv *, int, u_int32_t, u_int32_t);
static int  rndis_msg_query(struct rndis_softc *, u_int32_t, u_char *, int *);
static int  rndis_msg_set(struct rndis_softc *, u_int32_t, u_char *, int);

static const char *rndis_status_str(u_int32_t);

#define REQID (htole32((++sc->sc_req_id)?(sc->sc_req_id):(++sc->sc_req_id)))

#ifdef RNDIS_DEBUG
#define RNDIS_DEBUG_DUMP(b,l) dump_data((u_char *)(b), (int)(l))
static void
dump_data(u_char *datap, int datalen)
{
	int bcnt;
	for ( bcnt = 0 ; bcnt < datalen ; bcnt++ ) {
		if ( bcnt % 8 == 0) {
			printf("    %04x : ", bcnt);
		}
		printf("%02x", datap[bcnt]) ;
		if ( bcnt%8 == 7 ) {
			printf("\n") ;
		}
		else {
			printf(" ") ;
		}
	}
	if ( bcnt%8 > 0 ) {
		printf("\n") ;
	}
}
#else
#define RNDIS_DEBUG_DUMP(b,l)
#endif

static device_method_t rndis_methods[] = {
        /* Device interface */
        DEVMETHOD(device_probe,         rndis_probe),
        DEVMETHOD(device_attach,        rndis_attach),
        DEVMETHOD(device_detach,        rndis_detach),

        /* bus interface */
        DEVMETHOD(bus_print_child,      bus_generic_print_child),
        DEVMETHOD(bus_driver_added,     bus_generic_driver_added),

        { 0, 0 }
};

static driver_t rndis_driver = {
        .name = "rndis",
        .methods = rndis_methods,
        .size = sizeof(struct rndis_softc)
};

static devclass_t rndis_devclass;

DRIVER_MODULE(rndis, uhub, rndis_driver, rndis_devclass, NULL, 0);
MODULE_VERSION(rndis, 1);
MODULE_DEPEND(rndis, uether, 1, 1, 1);
MODULE_DEPEND(rndis, usb, 1, 1, 1);
MODULE_DEPEND(rndis, ether, 1, 1, 1);

static const struct usb_ether_methods rndis_ue_methods = {
        .ue_attach_post = rndis_attach_post,
        .ue_start = rndis_start,
        .ue_init = rndis_init,
        .ue_stop = rndis_stop,
        .ue_tick = rndis_tick,
        .ue_setmulti = rndis_setmulti,
        .ue_setpromisc = rndis_setpromisc,
        .ue_mii_upd = rndis_ifmedia_change,
        .ue_mii_sts = rndis_ifmedia_status,
};

static const struct usb_config rndis_config[RNDIS_N_TRANSFER] = {
	[RNDIS_BULK_DT_WR] = {
		.type = UE_BULK,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_OUT,
		.bufsize = 0,
		.flags = {.pipe_bof = 1,.force_short_xfer = 1,},
		.callback = rndis_bulk_write_callback,
		.timeout = 10000,
	},
	[RNDIS_BULK_DT_RD] = {
		.type = UE_BULK,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_IN,
		.bufsize = RNDIS_IOBUF_SIZE,
		.flags = {.pipe_bof = 1,.short_xfer_ok = 1,},
		.callback = rndis_bulk_read_callback,
	},
};

#define rndis_lookup(v, p) ((const struct rndis_type *)usb_lookup(rndis_devs, v, p))

static int
rndis_probe(device_t self)
{
	struct usb_attach_arg *uaa = device_get_ivars(self);
	
	DPRINTF("%s: %s: enter\n", device_get_nameunit(self), __func__);

	if (uaa->usb_mode != USB_MODE_HOST)
		return (ENXIO);
	if (uaa->info.bConfigIndex != RNDIS_CONFIG_INDEX)
		return (ENXIO);
	if (uaa->info.bIfaceIndex != RNDIS_IFACE_INDEX)
		return (ENXIO);

	return (usbd_lookup_id_by_uaa(rndis_devs, sizeof(rndis_devs), uaa));
}

/* device attach */
static int
rndis_attach(device_t dev)
{
	struct usb_attach_arg *uaa = device_get_ivars(dev);
	struct rndis_softc *sc = device_get_softc(dev);
	struct usb_ether *ue = &sc->sc_ue;
	int error;
	usb_error_t err;

	const char *devname;

	etheraddr_t eaddr;
	int         eaddrcnt;

	DPRINTF("%s: %s: enter, sc=%p\n",
		device_get_nameunit(dev), __func__, sc);

	sc->sc_flags = USB_GET_DRIVER_INFO(uaa);

	device_set_usb_desc(dev);

	devname = device_get_nameunit(dev);

	mtx_init(&sc->sc_mtx, devname, MTX_NETWORK_LOCK,
		 MTX_DEF | MTX_RECURSE);

	sc->sc_max_rx_size = RNDIS_IOBUF_SIZE;

	/* search interface */
	int ctrl_iface_index = -1;
	int data_iface_index = -1;
	
	for ( uint8_t ifaceIndex = uaa->info.bIfaceIndex; ifaceIndex < uaa->info.bIfaceNum; ifaceIndex++ ) {
		struct usb_interface *iface;
		struct usb_interface_descriptor *id;

		iface = usbd_get_iface(uaa->device, ifaceIndex);
		if (iface == NULL)
			break;
		id = usbd_get_interface_descriptor(iface);
		if (id == NULL)
			continue;

		DPRINTF("%s: iface class=0x%x, subclass=0x%x, protocol=0x%x\n",
			devname,
			id->bInterfaceClass,
			id->bInterfaceSubClass,
			id->bInterfaceProtocol);

		usbd_set_parent_iface(uaa->device, ifaceIndex, uaa->info.bIfaceIndex);

		if ( id->bInterfaceClass == UICLASS_CDC &&
		     id->bInterfaceSubClass == UISUBCLASS_ABSTRACT_CONTROL_MODEL &&
		     id->bInterfaceProtocol == UIPROTO_CDC_VENDOR ) {
			DPRINTF("%s: found CDC communication interface\n",
				devname);
			ctrl_iface_index = ifaceIndex;
		}
		else if ( id->bInterfaceClass == UICLASS_MISC &&
			  id->bInterfaceSubClass == 1 &&
			  id->bInterfaceProtocol == 1 ) {
			DPRINTF("%s: found ActiveSync interface\n",
				devname);
			ctrl_iface_index = ifaceIndex;
		}
		else if ( id->bInterfaceClass == UICLASS_CDC_DATA &&
			  id->bInterfaceSubClass == UISUBCLASS_DATA ) {
			DPRINTF("%s: found CDC Data interface\n",
				devname);
			data_iface_index = ifaceIndex;

			error = usbd_transfer_setup(uaa->device, &ifaceIndex,
						    sc->sc_xfer, rndis_config, RNDIS_N_TRANSFER,
						    sc, &sc->sc_mtx);
			if (error) {
				device_printf(dev, "allocating USB transfers failed\n");
				goto attachfail;
			}
		}
		else {
			printf("%s: found unexpected interface: class=0x%x,subclass=0x%x,protocol=0x%x\n",
			       devname,
			       id->bInterfaceClass,
			       id->bInterfaceSubClass,
			       id->bInterfaceProtocol);
			continue;
		}

	}

	if ( ctrl_iface_index < 0 ) {
		device_printf(dev, "Could not find control interface\n");
		goto attachfail;
	}
	sc->sc_ctrl_iface_index = ctrl_iface_index;

	if ( data_iface_index < 0 ) {
		device_printf(dev, "Could not find data interface\n");
		goto attachfail;
	}
	sc->sc_data_iface_index = data_iface_index;

	sc->sc_flags = USB_GET_DRIVER_INFO(uaa);

	/* initialize device */
	err = rndis_init_device(sc);
	if (err) {
		printf("%s: device initialize failed\n", devname);
		goto attachfail;
	}

	/* Get Ethernet Address */
	eaddrcnt = 1 ;
	err = rndis_get_etheraddr(sc, OID_802_3_PERMANENT_ADDRESS, &eaddr, &eaddrcnt);
	if (err) {
		printf("%s: read MAC address failed\n", devname);
		goto attachfail;
	}

	/* Print Ethernet Address */
	printf("%s: Ethernet address %s\n", devname, ether_sprintf(eaddr));

	/* Initialize Ethernet Interface Infomation */
	ue->ue_sc = sc;
	ue->ue_dev = dev;
	ue->ue_mtx = &sc->sc_mtx;
	ue->ue_methods = &rndis_ue_methods;

	error = uether_ifattach(ue);
	if (error) {
		device_printf(dev, "could not attach interface\n");
		goto attachfail;
	}

	return 0;

attachfail:
	DPRINTF("%s: %s: attach failed\n", devname, __func__);
	usbd_transfer_unsetup(sc->sc_xfer, RNDIS_N_TRANSFER);
	uether_ifdetach(ue);
	mtx_destroy(&sc->sc_mtx);
	return ENXIO;
}

/* device detach */
static int
rndis_detach(device_t dev)
{
	struct rndis_softc *sc;
	struct usb_ether *ue;

	sc = device_get_softc(dev);

	DPRINTF("%s: %s: sc=%p\n",
		device_get_nameunit(dev), __func__, sc);

	ue = &sc->sc_ue;

	usbd_transfer_unsetup(sc->sc_xfer, RNDIS_N_TRANSFER);
	uether_ifdetach(ue);
	mtx_destroy(&sc->sc_mtx);

	return 0;
}

static void
rndis_init(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);
	struct ifnet	   *ifp = uether_getifp(ue);
	etheraddr_t *eaddr;

	DPRINTF("%s: %s: enter\n",
		device_get_nameunit(sc->sc_dev), __func__);

	/* Cancel pending I/O and free all TX/RX buffers */
	rndis_stop(ue);

#if __FreeBSD_version >= 700000
	eaddr = (etheraddr_t *)IF_LLADDR(ifp);
#else
	eaddr = (etheraddr_t *)IFP2ENADDR(ifp);
#endif

	if (rndis_set_etheraddr(sc, OID_802_3_CURRENT_ADDRESS, eaddr, 1))
		printf("%s: set MAC addr failed\n", device_get_nameunit(sc->sc_dev));

	/* Initialize packet filter */
	sc->sc_ndis_filter = NDIS_PACKET_TYPE_DIRECTED;
	if (ifp->if_flags & IFF_BROADCAST)
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_BROADCAST;
	if (ifp->if_flags & IFF_PROMISC)
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_PROMISCUOUS;
	if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER, &sc->sc_ndis_filter, 1)) {
		printf("%s: set filter failed\n",
		       device_get_nameunit(sc->sc_dev));
	}

	/* Set lookahead */
	u_int32_t val = ifp->if_mtu ;
	if (rndis_set_uint32(sc, OID_GEN_CURRENT_LOOKAHEAD, &val, 1))
		printf("%s: set lookahead failed\n", device_get_nameunit(sc->sc_dev));

	/* Load the multicast filter */
	rndis_setmulti(ue);

	ifp->if_drv_flags |= IFF_DRV_RUNNING;
	ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;

	return ;
}

/* Stop the adapter and free any mbufs allocated to the RX and TX lists. */
static void
rndis_stop(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);
	struct ifnet      *ifp = uether_getifp(ue);

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);
	sc->sc_flags &= ~RNDIS_FLAG_LINK;

	/* Stop transfers */
	for ( int i = 0; i < RNDIS_N_TRANSFER; i++) {
		usbd_transfer_stop(sc->sc_xfer[i]);
	}

	return;
}


static int
rndis_init_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int fact;
	int err = 0;
	
	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	msg = malloc(RNDIS_MSG_INITIALIZE_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_INITIALIZE;
	msg->h.length = htole32(RNDIS_MSG_INITIALIZE_SIZE);
	msg->s.initialize.req_id    = REQID;
	msg->s.initialize.ver_major = htole32(1);
	msg->s.initialize.ver_minor = htole32(0);
	msg->s.initialize.max_xfer_size = htole32(sc->sc_max_rx_size);

	err = rndis_send_message(sc, msg, RNDIS_MSG_INITIALIZE_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}

	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.initialize.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	if (res->r.initialize_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: device initialize failed %s\n",
		       device_get_nameunit(sc->sc_dev), rndis_status_str(res->r.initialize_cmplt.status));
		err = EIO;
		goto done;
	}
	if (res->r.initialize_cmplt.medium != RNDIS_MEDIUM_802_3) {
		printf("%s: unsupported rndis media 0x%x\n",
		       device_get_nameunit(sc->sc_dev), le32toh(res->r.initialize_cmplt.medium));
		err = EIO;
		goto done;
	}
	sc->sc_max_tx_size      = le32toh(res->r.initialize_cmplt.max_xfer_size);
	sc->sc_max_pkts_per_msg = le32toh(res->r.initialize_cmplt.max_pkts_per_msg);
	fact = le32toh(res->r.initialize_cmplt.pkt_align_factor);
	if (fact > 3) {
		printf("%s: unsupported packet align factor %d\n",
		       device_get_nameunit(sc->sc_dev), fact);
		err = EIO;
		goto done;
	}
	sc->sc_pkt_align_bytes = (1 << fact);

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;

}

#if 0
static int
rndis_reset_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int err = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	msg = malloc(RNDIS_MSG_RESET_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_RESET;
	msg->h.length = htole32(RNDIS_MSG_RESET_SIZE);

	err = rndis_send_message(sc, msg, RNDIS_MSG_RESET_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}

	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, 0);
	if (err) {
		printf("%s: rndis_recv_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	if (res->r.reset_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: device reset failed %s\n",
		       device_get_nameunit(sc->sc_dev), rndis_status_str(res->r.reset_cmplt.status));
		err = EIO;
		goto done;
	}
	if (le32toh(res->r.reset_cmplt.addressing_reset)) {
		rndis_setmulti(&sc->sc_ue);
	}

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}
#endif

#if 0
static int
rndis_halt_device(struct rndis_softc *sc)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int err = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	if (sc->sc_dying)
		return 0;

	msg = malloc(RNDIS_MSG_RESET_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_HALT;
	msg->h.length = htole32(RNDIS_MSG_HALT_SIZE);
	msg->s.halt.req_id = REQID;

	err = rndis_send_message(sc, msg, RNDIS_MSG_HALT_SIZE);
	if (err) {
		printf("%s: rndis_send_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	/* this message has no completion message */

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}
#endif

static int
rndis_msg_query(struct rndis_softc *sc, u_int32_t oid, u_char *datap, int *datalen)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	u_char *infp;
	int msglen = 0;
	int reslen = 0;
	int inflen = 0;
	int infoff = sizeof(msg->s.query);
	int err = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	/* dummy? */
	switch (oid) {
	case OID_GEN_SUPPORTED_LIST:
		inflen = 0x00;
		break;
	case OID_802_3_MAXIMUM_LIST_SIZE:
	case OID_GEN_CURRENT_LOOKAHEAD:
	case OID_GEN_CURRENT_PACKET_FILTER:
	case OID_GEN_LINK_SPEED:
	case OID_GEN_MAC_OPTIONS:
	case OID_GEN_MAXIMUM_FRAME_SIZE:
	case OID_GEN_MAXIMUM_LOOKAHEAD:
	case OID_GEN_MAXIMUM_SEND_PACKETS:
	case OID_GEN_VENDOR_DRIVER_VERSION:
		inflen = 0x04;
		break;
	case OID_802_3_CURRENT_ADDRESS:
	case OID_802_3_PERMANENT_ADDRESS:
		inflen = 0x06;
		break;
	default:
		inflen = 0;
		break;
	}

	msglen = RNDIS_MSG_QUERY_SIZE + inflen;

	msg = malloc(msglen, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_QUERY;
	msg->h.length = htole32(msglen);
	msg->s.query.req_id    = REQID;
	msg->s.query.oid       = htole32(oid);
	msg->s.query.inf_buf_length = htole32(inflen);
	msg->s.query.inf_buf_offset = htole32(infoff);

	err = rndis_send_message(sc, msg, msglen);
	if (err) {
		printf("%s: rndis_send_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.query.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	if (res->r.query_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: query message failed %s\n",
		       device_get_nameunit(sc->sc_dev), rndis_status_str(res->r.reset_cmplt.status));
		err = EIO;
		goto done;
	}
	reslen = le32toh(res->h.length) - sizeof(res->h);
	inflen = le32toh(res->r.query_cmplt.inf_buf_length);
	infoff = le32toh(res->r.query_cmplt.inf_buf_offset);
	if (reslen < infoff + inflen) {
		printf("%s: query completion size mismatch reslen(%d) < infoff(%d) + inflen(%d)\n",
		       device_get_nameunit(sc->sc_dev), reslen, infoff, inflen) ;
		err = EIO;
		goto done;
	}
	if (inflen > *datalen) {
		printf("%s: data buffer size mismatch need=%d, present=%d\n",
		       device_get_nameunit(sc->sc_dev), inflen, *datalen);
		err = EIO;
		goto done;
	}
	infp = (u_char *)&res->r + infoff;
	bcopy(infp, datap, inflen);
	*datalen = inflen;

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}

static int
rndis_msg_set(struct rndis_softc *sc, u_int32_t oid, u_char *datap, int datalen)
{
	struct rndis_msg_snd *msg;
	struct rndis_msg_rcv *res;
	int msglen = 0;
	u_char *infp;
	int err = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	msglen = RNDIS_MSG_SET_SIZE + datalen;

	msg = malloc(msglen, M_USBDEV, M_WAITOK | M_ZERO);
	res = malloc(RNDIS_MSG_BUFFER_SIZE, M_USBDEV, M_WAITOK | M_ZERO);

	msg->h.type   = RNDIS_MSG_SET;
	msg->h.length = htole32(msglen);
 	msg->s.set.req_id    = REQID;
	msg->s.set.oid       = htole32(oid);
	msg->s.set.inf_buf_length = htole32(datalen);
	msg->s.set.inf_buf_offset = htole32(sizeof(msg->s.set));

	infp = (u_char *)&msg->s + sizeof(msg->s.set);
	bcopy(datap, infp, datalen);

	err = rndis_send_message(sc, msg, msglen);
	if (err) {
		printf("%s: rndis_send_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	err = rndis_recv_message(sc, res, RNDIS_MSG_BUFFER_SIZE,
				 msg->h.type | RNDIS_MSG_COMPLETION, msg->s.set.req_id);
	if (err) {
		printf("%s: rndis_recv_message failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	if (res->r.set_cmplt.status != RNDIS_STATUS_SUCCESS) {
		printf("%s: set message failed: %s\n",
		       device_get_nameunit(sc->sc_dev),
		       rndis_status_str(res->r.set_cmplt.status));
		err = EIO;
		goto done;
	}

 done:
        free(msg, M_USBDEV);
        free(res, M_USBDEV);

	return err;
}

static int
rndis_get_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int *cnt)
{
	int err = 0;
	int len = 0;
	int i;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	len = sizeof(*val) * (*cnt);
	err = rndis_msg_query(sc, oid, (u_char *)val, &len);
	if (err) {
		printf("%s: rndis_msg_query failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}
	len = len/sizeof(*val);
	for (i = 0; i < len; i++)
		*(val + 1) = le32toh(*(val + 1));
	*cnt = len;
 done:
	return err;
}

static int
rndis_set_uint32(struct rndis_softc *sc, u_int32_t oid, u_int32_t *val, int cnt)
{
	int err = 0;
	int i = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	for (i = 0; i < cnt; i++)
		*(val + i) = htole32(*(val + i));

	err = rndis_msg_set(sc, oid, (u_char *)val, sizeof(*val)*cnt);
	if (err) {
		printf("%s: rndis_msg_set failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}

done:
	return err;
}

static int
rndis_get_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int *cnt)
{
	int err = 0;
	int len = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	len = ETHER_ADDR_LEN * (*cnt);
	err = rndis_msg_query(sc, oid, (u_char *)eaddr, &len);
	if (err) {
		printf("%s: rndis_msg_query failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}

 done:
	return err;
}

static int
rndis_set_etheraddr(struct rndis_softc *sc, u_int32_t oid, etheraddr_t *eaddr, int cnt)
{
	int err = 0;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	err = rndis_msg_set(sc, oid, (u_char *)eaddr, sizeof(*eaddr)*cnt);
	if (err) {
		printf("%s: rndis_msg_set failed\n", device_get_nameunit(sc->sc_dev));
		goto done;
	}

done:
	return err;
}

static int
rndis_send_message(struct rndis_softc *sc, struct rndis_msg_snd *msg, int msglen)
{
	usb_error_t uerr;
	int error = 0;
	usb_device_request_t req;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	msg->h.length = htole32(msglen);

	req.bmRequestType = UT_WRITE_CLASS_INTERFACE;
	req.bRequest = UCDC_SEND_ENCAPSULATED_COMMAND;
	USETW(req.wValue, 0x0000);
	USETW(req.wIndex, sc->sc_ctrl_iface_index);
	USETW(req.wLength, msglen);

	DPRINTF("%s: write request send\n", device_get_nameunit(sc->sc_dev));

	RNDIS_DEBUG_DUMP(msg, msglen);

	uerr = uether_do_request(&sc->sc_ue,
				 &req,
				 msg,
				 1000);
	if (uerr) {
		device_printf(sc->sc_dev, "uether_do_request failed: %s\n",
			      usbd_errstr(uerr));
		error = EIO;
		goto done;
	}
	DPRINTF("%s: write request done\n", device_get_nameunit(sc->sc_dev));

done:
	return error;
}

static int
rndis_recv_message(struct rndis_softc *sc, struct rndis_msg_rcv *res, int buflen,
		   u_int32_t msg_type, u_int32_t msg_req_id)
{
	int i;
	usb_device_request_t req;
	int reslen;
	int total_len;
	int error = 0;
	usb_error_t uerr;
	int flag;
	u_int32_t req_id = 0, status = 0;
	struct rndis_msg_snd *msg;
	uint16_t actlen;

#define RNDIS_MSG_HAS_NONE   0
#define RNDIS_MSG_HAS_REQID  1
#define RNDIS_MSG_HAS_STATUS 2
#define RNDIS_MSG_HAS_CMPLT  4
#define RNDIS_MSG_HAS_ALL    7

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	for ( i = 0; i < 10; i++ ) {
		req.bmRequestType = UT_READ_CLASS_INTERFACE;
		req.bRequest = UCDC_GET_ENCAPSULATED_RESPONSE;
		USETW(req.wValue, 0x0000);
		USETW(req.wIndex, sc->sc_ctrl_iface_index);
		USETW(req.wLength, buflen);

		DPRINTF("%s: read request send\n", device_get_nameunit(sc->sc_dev));

		total_len = 0;

		uerr = usbd_do_request_flags(sc->sc_ue.ue_udev,
					     &sc->sc_mtx,
					     &req,
					     res,
					     USB_SHORT_XFER_OK,
					     &actlen,
					     USB_DEFAULT_TIMEOUT);
		if (uerr == USB_ERR_TIMEOUT) {
			device_printf(sc->sc_dev,
				      "usbd_do_request_flags timeout, retry\n");
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if (uerr) {
			device_printf(sc->sc_dev,
				      "usbd_do_request_flags failed: %s\n",
				      usbd_errstr(uerr));
			error = EIO;
			goto done;
		}
		total_len = actlen;
		DPRINTF("%s: read request done, actlen = %d, req.wLength=%d\n",
			device_get_nameunit(sc->sc_dev), total_len, UGETW(req.wLength));

		if (total_len < RNDIS_MSG_HEADER_SIZE) {
			device_printf(sc->sc_dev,
				      "too short message(no header size) require %d, get %d\n",
				      RNDIS_MSG_HEADER_SIZE, total_len);
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}

		DPRINTF("%s: type=0x%x\n",
			device_get_nameunit(sc->sc_dev), htole32(res->h.type));

		reslen = le32toh(res->h.length);
		if (total_len < reslen) {
			device_printf(sc->sc_dev,
				      "size mismatch total_len(%d) < message len(%d)\n",
				      total_len, reslen);
			error = EIO;
			goto done;
		}

		RNDIS_DEBUG_DUMP(res, total_len);

		flag = RNDIS_MSG_HAS_NONE;
		switch (res->h.type) {
		case RNDIS_MSG_INITIALIZE_CMPLT:
			if (reslen < RNDIS_MSG_INITIALIZE_CMPLT_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			req_id = res->r.initialize_cmplt.req_id;
			status = res->r.initialize_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_QUERY_CMPLT:
			if (reslen < RNDIS_MSG_QUERY_CMPLT_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			req_id = res->r.query_cmplt.req_id;
			status = res->r.query_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_SET_CMPLT:
			if (reslen < RNDIS_MSG_SET_CMPLT_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			req_id = res->r.set_cmplt.req_id;
			status = res->r.set_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		case RNDIS_MSG_RESET_CMPLT:
			if (reslen < RNDIS_MSG_RESET_CMPLT_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			status = res->r.reset_cmplt.status;
			flag = RNDIS_MSG_HAS_STATUS|RNDIS_MSG_HAS_CMPLT;
			break;
		case RNDIS_MSG_INDICATE_STATUS:
			if (reslen < RNDIS_MSG_INDICATE_STATUS_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			status = res->r.indicate_status.status;
			flag = RNDIS_MSG_HAS_STATUS;
			break;
		case RNDIS_MSG_KEEPALIVE:
			if (reslen < RNDIS_MSG_KEEPALIVE_SIZE) {
				printf("%s: %s: too short\n",
				       device_get_nameunit(sc->sc_dev), __func__);
				break;
				
			}
			msg = malloc(RNDIS_MSG_KEEPALIVE_SIZE, M_USBDEV, M_WAITOK | M_ZERO);
			msg->h.type   = RNDIS_MSG_KEEPALIVE_CMPLT;
			msg->h.length = htole32(RNDIS_MSG_KEEPALIVE_CMPLT_SIZE);
			msg->s.keepalive_cmplt.req_id = res->r.keepalive.req_id;
			msg->s.keepalive_cmplt.status = RNDIS_STATUS_SUCCESS;
			rndis_send_message(sc, msg, RNDIS_MSG_KEEPALIVE_CMPLT_SIZE);
			free(msg, M_USBDEV);
			break;
		case RNDIS_MSG_KEEPALIVE_CMPLT:
			if (reslen < RNDIS_MSG_KEEPALIVE_CMPLT_SIZE) {
				DPRINTF("%s: %s too short message\n",
					device_get_nameunit(sc->sc_dev), __func__);
				goto done;
			}
			req_id = res->r.keepalive_cmplt.req_id;
			status = res->r.keepalive_cmplt.status;
			flag = RNDIS_MSG_HAS_ALL;
			break;
		default:
			printf("%s: %s: unknown/unexpected message recv %x\n",
			       device_get_nameunit(sc->sc_dev), __func__, le32toh(res->h.type));
			break;
		}

		if (res->h.type != msg_type) {
			DPRINTF("%s: %s: message type mismatch, retry\n",
				device_get_nameunit(sc->sc_dev), __func__);
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if ((flag & RNDIS_MSG_HAS_REQID) &&
		    (req_id != msg_req_id)) {
			DPRINTF("%s: %s: request id mismatch, retry\n",
				device_get_nameunit(sc->sc_dev), __func__);
			tsleep(sc, PWAIT, "rndis", 2);
			continue;
		}
		if ((flag & RNDIS_MSG_HAS_STATUS) &&
		    (status != RNDIS_STATUS_SUCCESS)) {
			DPRINTF("%s: %s: status is not a success\n",
				device_get_nameunit(sc->sc_dev), __func__);
		}
		/* read success */
		error = 0;
		goto done;
	}
	/* retry timeout */
	DPRINTF("%s: %s retry time out\n",
		device_get_nameunit(sc->sc_dev), __func__);
	error = EIO;
done:
	return error;
}

static void
rndis_bulk_read_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
	struct rndis_softc *sc = usbd_xfer_softc(xfer);
	struct ifnet *ifp;
	int  total_len, msglen, framelen;
	struct rndis_msg_packet msg;
	struct usb_page_cache *pc;
	struct usb_ether *ue;
	int  framestart;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	sc = usbd_xfer_softc(xfer);
	ue = &sc->sc_ue;

	usbd_xfer_status(xfer, &total_len, NULL, NULL, NULL);
	
	ifp = uether_getifp(&sc->sc_ue);

	switch (USB_GET_STATE(xfer)) {
	case USB_ST_TRANSFERRED:
		break;

	case USB_ST_SETUP:
		goto tr_setup;

	default:
		DPRINTF("%s: %s: bulk read error, %s\n", device_get_nameunit(sc->sc_dev), __func__,
			usbd_errstr(uerr));

		if (uerr != USB_ERR_CANCELLED) {
			/* try to clear stall first */
			usbd_xfer_set_stall(xfer);
			goto tr_setup;
		}
		return;
	}

	DPRINTF("%s: %s: xfersize=%u\n", device_get_nameunit(sc->sc_dev), __func__,
		total_len);

	pc = usbd_xfer_get_frame(xfer, 0);

	for ( int offset = 0; offset < total_len; offset += msglen ) {
		msglen = total_len;
		DPRINTF("%s: %s: msg offset=0x%04x\n",
			device_get_nameunit(sc->sc_dev), __func__,
			offset);
		if ((total_len - offset) < sizeof(msg.h)) {
			/* tooooo short */
			printf("%s: usb packet length(%d) < rndis message header size(%d)\n",
			       device_get_nameunit(sc->sc_dev), total_len, sizeof(msg.h));
			goto tr_setup;
		}
		usbd_copy_out(pc, offset, &msg.h, sizeof(msg.h));
		if (le32toh(msg.h.length) == 0) {
			DPRINTF("%s: %s: recieved zero size message\n",
				device_get_nameunit(sc->sc_dev), __func__);
			continue;
		}
		msglen = sizeof(msg.h);
		if (msg.h.type != RNDIS_MSG_PACKET) {
			/* unknown message */
			printf("%s: unknown rndis message type(0x%x)\n",
			       device_get_nameunit(sc->sc_dev), le32toh(msg.h.type));
			goto tr_setup;
		}
		msglen = le32toh(msg.h.length);
		if (msglen < RNDIS_MSG_PACKET_SIZE) {
			/* too short */
			printf("%s: message length(%d) < rndis packet messege size(%d)\n",
			       device_get_nameunit(sc->sc_dev), msglen, RNDIS_MSG_PACKET_SIZE);
			continue;
		}
		if ((total_len - offset) < msglen) {
			/* what? */
			DPRINTF("%s: %s: usb packet length(%d) < message length(%d)\n",
				device_get_nameunit(sc->sc_dev), __func__, total_len, msglen);
			goto tr_setup;
		}
		usbd_copy_out(pc, offset + sizeof(msg.h), &msg.d, sizeof(msg.d));

		framelen = le32toh(msg.d.length);

		framestart = offset + sizeof(msg.h) + le32toh(msg.d.offset);

		if (framelen < sizeof(struct ether_header)) {
			printf("%s: frame length(%d) < ether header(%d)",
			       device_get_nameunit(sc->sc_dev), framelen, sizeof(struct ether_header));
			ifp->if_ierrors++;
			continue;
		}

		/* copy */
		uether_rxbuf(ue, pc, framestart, framelen);
		DPRINTF("%s: %s: recieved %d\n", device_get_nameunit(sc->sc_dev),
			__func__, framelen);
	}

tr_setup:
	usbd_xfer_set_frame_len(xfer, 0, usbd_xfer_max_len(xfer));
	usbd_transfer_submit(xfer);
	uether_rxflush(ue);
	return;
}

static void
rndis_bulk_write_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
	struct rndis_softc *sc = usbd_xfer_softc(xfer);
	struct ifnet *ifp;
	struct usb_page_cache *pc;
	struct mbuf *m;
	struct rndis_msg_packet msg;
	int actlen;
	int total_len;
	int align = sc->sc_pkt_align_bytes;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	ifp = uether_getifp(&sc->sc_ue);

	usbd_xfer_status(xfer, &actlen, NULL, NULL, NULL);
	pc = usbd_xfer_get_frame(xfer, 0);

	switch (USB_GET_STATE(xfer)) {
	case USB_ST_TRANSFERRED:
		ifp->if_opackets++;
		/* fall through */
	case USB_ST_SETUP:
	tr_setup:
		if ((sc->sc_flags & RNDIS_FLAG_LINK) == 0 ) {
			return;
		}
		IFQ_DRV_DEQUEUE(&ifp->if_snd, m);
		if (m == NULL) {
			return;
		}
		if (m->m_pkthdr.len > MCLBYTES) {
			m->m_pkthdr.len = MCLBYTES;
		}
		actlen = sizeof(msg) + m->m_pkthdr.len;
		total_len = (actlen + align - 1)/align * align;

		bzero(&msg, sizeof(msg));
		msg.h.type   = RNDIS_MSG_PACKET;
		msg.h.length = htole32(total_len);
		msg.d.offset = htole32(sizeof(msg.d));
		msg.d.length = htole32(m->m_pkthdr.len);

		usbd_copy_in(pc, 0, &msg, sizeof(msg));
		usbd_m_copy_in(pc, sizeof(msg), m, 0, m->m_pkthdr.len);
		if (total_len != actlen) {
			usbd_frame_zero(pc, actlen, total_len - actlen);
		}
		BPF_MTAP(ifp, m);
		m_freem(m);
		usbd_xfer_set_frame_len(xfer, 0, total_len);
		usbd_transfer_submit(xfer);
		return;
	default: /* error */
		DPRINTF("%s: %s: transfer error, %s\n",
			device_get_nameunit(sc->sc_dev), __func__,
			usbd_errstr(uerr));
		ifp->if_oerrors++;

		if (uerr != USB_ERR_CANCELLED) {
			/* try to clear stall first */
			usbd_xfer_set_stall(xfer);
			goto tr_setup;
		}
		return;
	}
}

static void
rndis_start(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	/* start USB transfer */
	for ( int i = 0; i < RNDIS_N_TRANSFER; i++) {
		usbd_transfer_start(sc->sc_xfer[i]);
	}
	return;
}

static void
rndis_setpromisc(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);
	struct ifnet *ifp = uether_getifp(ue);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	if (ifp->if_flags & IFF_PROMISC) {
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_PROMISCUOUS;
	}
	else {
		sc->sc_ndis_filter &= ~NDIS_PACKET_TYPE_PROMISCUOUS;
	}
	if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER,
			     &sc->sc_ndis_filter, 1)) {
		device_printf(sc->sc_dev,
			      "set filter failed\n");
	}
}

static void
rndis_setmulti(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);
	struct ifnet *ifp = uether_getifp(ue);
	struct ifmultiaddr *ifma;
	u_int mclistmax;
	int cnt;
	int mclistlen;
	int error = 0;
	u_char (*mclist)[ETHER_ADDR_LEN] = NULL;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	if (ifp->if_flags & IFF_ALLMULTI || ifp->if_flags & IFF_PROMISC) {
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		goto done;
	}

	if (TAILQ_EMPTY(&ifp->if_multiaddrs))
		return;

	cnt = 1;
	if (rndis_get_uint32(sc, OID_802_3_MAXIMUM_LIST_SIZE, &mclistmax, &cnt)) {
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		device_printf(sc->sc_dev,
			      "get malticast list max failed\n");
		error = EIO;
		goto done;
	}

	mclist = malloc(sizeof(mclist[0]) * mclistmax, M_USBDEV, M_WAITOK | M_ZERO);

	sc->sc_ndis_filter |= NDIS_PACKET_TYPE_MULTICAST;

	mclistlen = 0;
	IF_ADDR_LOCK(ifp);
	TAILQ_FOREACH(ifma, &ifp->if_multiaddrs, ifma_link) {
		if (ifma->ifma_addr->sa_family != AF_LINK)
			continue;
		if (mclistlen >= mclistmax) {
			IF_ADDR_UNLOCK(ifp);
			sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
			sc->sc_ndis_filter &= ~NDIS_PACKET_TYPE_MULTICAST;
			goto done;
		}
		bcopy(LLADDR((struct sockaddr_dl *)ifma->ifma_addr),
		      mclist[mclistlen], sizeof(mclist[mclistlen]));
		mclistlen++;
	}
	IF_ADDR_UNLOCK(ifp);

	error = rndis_set_etheraddr(sc, OID_802_3_MULTICAST_LIST, mclist, mclistlen);
	if (error) {
		device_printf(sc->sc_dev,
			      "set mclist failed: %d\n", error);
		sc->sc_ndis_filter |= NDIS_PACKET_TYPE_ALL_MULTICAST;
		sc->sc_ndis_filter &= ~NDIS_PACKET_TYPE_MULTICAST;
	}

done:
	if (mclist)
		free(mclist, M_USBDEV);

	if (rndis_set_uint32(sc, OID_GEN_CURRENT_PACKET_FILTER, &sc->sc_ndis_filter, 1)) {
		device_printf(sc->sc_dev,
			      "set filter failed\n");
		return;
	}

	return;

}

/* rndis status to string */
static const char *
rndis_status_str(u_int32_t rndis_status)
{
	switch (rndis_status) {
	case RNDIS_STATUS_BUFFER_OVERFLOW:
		return "buffer overflow";
	case RNDIS_STATUS_FAILURE:
		return "failure";
	case RNDIS_STATUS_INVALID_DATA:
		return "invalid data";
	case RNDIS_STATUS_MEDIA_CONNECT:
		return "media connect";
	case RNDIS_STATUS_MEDIA_DISCONNECT:
		return "media disconnect";
	case RNDIS_STATUS_NOT_SUPPORTED:
		return "not supported";
	case RNDIS_STATUS_PENDING:
		return "pending";
	case RNDIS_STATUS_RESOURCES:
		return "resources";
	case RNDIS_STATUS_SUCCESS:
		return "success";
	default:
		return "unknown";
	}
}

static int
rndis_ifmedia_change(struct ifnet *ifp)
{
	struct rndis_softc *sc = (struct rndis_softc*)ifp->if_softc;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	rndis_init(&sc->sc_ue);

	return 0;
}

static void
rndis_ifmedia_status(struct ifnet * const ifp, struct ifmediareq *ifmr)
{
	struct rndis_softc *sc = (struct rndis_softc*)ifp->if_softc;
	
	u_int32_t		media_info;
	ndis_media_state	linkstate = nmc_connected;
	int			error, cnt;

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	ifmr->ifm_status = IFM_AVALID ;
	ifmr->ifm_active = IFM_ETHER ;

#if 0
	cnt = 1;
	error = rndis_get_uint32(sc, OID_GEN_MEDIA_CONNECT_STATUS,
				 (u_int32_t *)&linkstate, &cnt);
#endif

	cnt = 1;
	error = rndis_get_uint32(sc, OID_GEN_LINK_SPEED,
				 &media_info, &cnt);
	if (linkstate == nmc_connected)
		ifmr->ifm_status |= IFM_ACTIVE;

	switch(media_info) {
	case 100000:
		ifmr->ifm_active |= IFM_10_T;
		break;
	case 1000000:
		ifmr->ifm_active |= IFM_100_TX;
		break;
	case 10000000:
		ifmr->ifm_active |= IFM_1000_T;
		break;
	default:
		DPRINTF("%s: unknown speed: %d\n", device_get_nameunit(sc->sc_dev), media_info);
		break;
	}

	return;
}

static void
rndis_attach_post(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);

	DPRINTF("%s: %s: enter\n", device_get_nameunit(sc->sc_dev), __func__);

	/* reset */
	rndis_init(ue);
}

static void
rndis_tick(struct usb_ether *ue)
{
	struct rndis_softc *sc = uether_getsc(ue);
	struct mii_data *mii = uether_getmii(ue);

	RNDIS_LOCK_ASSERT(sc, MA_OWNED);

	mii_tick(mii);
	if ((sc->sc_flags & RNDIS_FLAG_LINK) == 0
	    && mii->mii_media_status & IFM_ACTIVE &&
	    IFM_SUBTYPE(mii->mii_media_active) != IFM_NONE) {
		sc->sc_flags |= RNDIS_FLAG_LINK;
		rndis_start(ue);
	}
}
