/* -*- mode: C; -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestFixture.h,v 1.9 2003/10/05 11:29:59 tsutsumi Exp $
 */

/**
 * @file
 * TestFixture module.
 */
#ifndef CCUNITTESTFIXTURE_H
#define CCUNITTESTFIXTURE_H

#include <ccunit/CCUnitConfig.h>
#include <ccunit/CCUnitList.h>
#include <ccunit/CCUnitTest.h>
#include <ccunit/CCUnitTestCase.h>
#include <ccunit/CCUnitTestResult.h>

/**
 * @ingroup CCUnitTest
 * @defgroup CCUnitTestFixture TestFixture
 * A test fixture defines the fixture to run multiple tests.
 *
 * @{
 */

/**
 * Wraps a test case with setUp and tearDown methods.
 *
 * A TestCase is used to provide a common environment for a set
 * of test cases.
 *
 * To define a test case, do the following:
 * - the case is defined by static variables
 * - initialize the case state by setUp function
 * - clean-up after a test by tearDown function
 *
 * Each test runs in its own case so there
 * can be no side effects among test runs.
 * Here is an example:
 * 
 * @code
 * static int value1, value2;
 *
 * void setUp_MathTest ()
 * {
 *   value1 = 2;
 *   value2 = 3;
 * }
 *
 * ...
 *
 * CCUnitTestFixture* MathTest_newTestFixture ()
 * {
 *   return ccunit_newTestFixture ("MathTest", setUp_MathTest, NULL);
 * }
 * @endcode
 *
 * For each test implement a function which interacts with the
 * case. Verify the expected results with assertions specified by
 * calling CCUNIT_ASSERT on the expression you want to test:
 * 
 * @code
 * void testAdd ()
 * {
 *   int result = value1 + value2;
 *   CCUNIT_ASSERT (result == 5);
 * }
 *
 * ...
 *
 * void MathTest_newTestCase_testAdd ()
 * {
 *   return ccunit_newTestCase ("addTest", "add test", addTest);
 * }
 * @endcode
 * 
 * The tests to be run can be collected into a TestSuite. 
 * 
 * @code
 * CCUintTestSuite* MathTest_suite ()
 * {
 *   CCUnitTestSuite* suite = ccunit_newTestSuite ("MathTest");
 *   CCUnitTestFixture* fixture = MathTest_newTestFixture ();
 *   ccunit_addTestFixture (suite, fixture);
 *   ccunit_addTestCase (fixture, MathTest_newTestCase_testAdd ());
 *   ccunit_addTestCase (fixture, MathTest_newTestCase_testDivZero ())
 *   return suite;
 * }
 * @endcode
 * 
 * Once the functions are defined you can run them. To do this, use a
 * TestRunner.
 *
 * @code
 *   CCUnitTestRunner *runner = ccunit_newTestRunner (stdout);
 *   CCUnitTestSuite *suite = MathTest_suite ();
 *   runner->run (runner, suite);
 * @endcode
 *
 * A command line tool have been created for convenience. It is located
 * in src/tools/ccunit_makeSuite.c.
 *
 * @see CCUnitTestResult, CCUnitTestCase, CCUnitTestSuite, CCUnitMakeSuite,
 *
 * @ingroup WritingTestFixture
 */
typedef struct CCUnitTestFixture
{
  CCUnitTest test;				/**< super class */
  const char* name;				/**< test fixture name */
  CCUnitTestFunc* setUp;			/**< setUp function */
  CCUnitTestFunc* tearDown;			/**< tearDown function */
  CCUnitList testCases;				/**< test cases */
} CCUnitTestFixture;

/**
 * TestFixture definition structure
 */
typedef struct CCUnitTestFixtureDfn
{
  CCUnitTestDfn test;				/**< super class */
  const char* name;				/**< test fixture name */
  CCUnitTestFunc setUp;				/**< setUp function */
  CCUnitTestFunc tearDown;			/**< tearDown function */
  CCUnitTestFunc* testCases;			/**< test cases */
} CCUnitTestFixtureDfn;

/**
 * create new test fixture.
 *
 * @param name test fixture name.
 * @param setUp test fixture setUp function.
 * @param tearDown test fixture tearDown function.
 * @return new test fixture.
 * @ingroup WritingTestFixture
 */
extern CCUnitTestFixture* ccunit_newTestFixture (const char* name,
						 CCUnitTestFunc* setUp,
						 CCUnitTestFunc* tearDown);

/**
 * add test case to test fixture.
 *
 * @param fixture test fixture.
 * @param testCase test case
 * @ingroup WritingTestFixture
 */
extern inline void ccunit_addTestCase (CCUnitTestFixture* fixture,
				       CCUnitTestCase* testCase);

/**
 * add new test case to test fixture.
 *
 * @param fixture test fixture.
 * @param name test case name.
 * @param desc test case description.
 * @param runTest run test function.
 * @return new test case
 * @ingroup WritingTestFixture
 */
extern CCUnitTestCase* ccunit_addNewTestCase (CCUnitTestFixture* fixture,
					      const char* name,
					      const char* desc,
					      void (*runTest)());

/**
 * run test cases and collect its results.
 * @param f test fixture.
 * @return test result.
 * @ingroup ExecutingTest
 */
extern inline struct CCUnitTestResult* ccunit_runTestFixture (CCUnitTestFixture* f);

/** @} */
#endif
