/* -*- mode: C; -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestResult.h,v 1.6 2003/09/28 13:47:30 tsutsumi Exp $
 */
/** @file
 * TestResult module.
 */
#ifndef CCUNITTESTRESULT_H
#define CCUNITTESTRESULT_H

#include <ccunit/CCUnitConfig.h>
#include <ccunit/CCUnitList.h>
#include <ccunit/CCUnitTestFailure.h>
#include <ccunit/CCUnitTestListener.h>

/**
 * @ingroup ModuleHierarchy
 * @defgroup CCUnitTestResult TestResult
 * Collects the results of a test case module.
 *
 * This module use a CCUnitTestListener to be informed of the ongoing
 * tests.
 *
 * @see CCUnitTest, CCUnitTestListener.
 *
 * @{
 */

/**
 * Collects the results of a test case.
 *
 * This module use a CCUnitTestListener to be informed of the ongoing
 * tests.
 *
 * @see CCUnitTest, CCUnitTestListener.
 * @ingroup TrackingTestExecution
 */
typedef struct CCUnitTestResult
{
  CCUnitList failures;				/**< failure objects */
  CCUnitList listeners;				/**< testCase objects  */
  size_t runCount;				/**< number of run test */
  bool shouldStop;				/**< test run should stop flag */
} CCUnitTestResult;

/**
 * Construct TestResult
 * @ingroup TrackingTestExecution
 */
extern CCUnitTestResult* ccunit_newTestResult ();

/**
 * Destruct TestResult
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern void ccunit_deleteTestResult (CCUnitTestResult* result);

/**
 * Adds a failure to the list of failures.
 * The passed in failed assertion caused the failure.
 * @param result result object.
 * @param failure failure.
 * @ingroup TrackingTestExecution
 */
extern void ccunit_addFailure (CCUnitTestResult* result,
			       struct CCUnitTestFailure* failure);

/**
 * Registers a TestRunner as TestListener.
 * @param result result object.
 * @param listener adding listener.
 * @ingroup TrackingTestExecution
 */
extern inline void ccunit_addResultListener (CCUnitTestResult* result,
					     CCUnitTestListener* listener);

/**
 * Returns an Iterator for the failures.
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern inline CCUnitListIterator* ccunit_failures(CCUnitTestResult* result);

/**
 * Gets the number of detected failures.
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern inline size_t ccunit_failureCount (CCUnitTestResult* result);

/**
 * Returns whether the entire test was successful or not.
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern inline bool ccunit_wasSuccessful (CCUnitTestResult* result);

/**
 * Marks that the test run should stop.
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern inline void ccunit_stopTest (CCUnitTestResult* result);

/**
 * Gets the number of run tests.
 * @param result result object.
 * @ingroup TrackingTestExecution
 */
extern inline size_t ccunit_runCount (CCUnitTestResult* result);

/** @} */

#endif
