/* -*- mode: C; -*- */
/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestFunc.h,v 1.2 2010/08/20 09:36:01 tsutsumi Exp $
 */

/** @file
 * TestCase function module.
 */

#ifndef CCUNITTESTFUNC_H
#define CCUNITTESTFUNC_H

#include <ccunit/CCUnitConfig.h>

/**
 * @ingroup CCUnitTestCase
 * @defgroup CCUnitTestFunc TestFunc
 *
 * A single test object.
 *
 * This class is used to implement a simple test func.
 *
 * @{
 */

/**
 * A single test function object.
 *
 * For each test implement a function which interacts with the
 * case. Verify the expected results with assertions specified by
 * calling CCUNIT_ASSERT on the expression you want to test:
 * 
 * @code
 * void testAdd ()
 * {
 *   int result = value1 + value2;
 *   CCUNIT_ASSERT (result == 5);
 * }
 *
 * ...
 * int main ()
 * {
 *   CCUnitTestResult* r;
 *   CCUnitTestCase* c = ccunit_newTestCase ("math test");
 *   CCUnitTestFunc* f = ccunit_newTestFunc ("testAdd", "add test", testAdd);
 *   ccunit_addTestFunc (c, f);
 *   r = ccunit_runTestCase (c);
 *   return 0;
 * }
 * @endcode
 *
 * @see CCUnitTestCase, CCUnitTestSuite, CCUnitMakeSuite
 * 
 * @ingroup WritingTestCase
 */
typedef struct CCUnitTestFunc
{
  const char* name;				/**< test func name */
  const char* desc;				/**< test description */
  void (*runTest) ();				/**< run test function */
} CCUnitTestFunc;

/**
 * Create new test function.
 * @param name function name.
 * @param desc function description.
 * @param runTest run test function.
 * @return new test function object.
 * @ingroup WritingTestCase
 */
extern CCUnitTestFunc* ccunit_newTestFunc (const char* name,
					   const char* desc,
					   void (*runTest)());
#define CCUNIT_NEWTESTFUNC(FUNC) ccunit_newTestFunc (#FUNC, NULL, FUNC)

/**
 * Delete test function.
 * @param f deleting function.
 * @ingroup WritingTestCase
 */
extern void ccunit_deleteTestFunc (CCUnitTestFunc* f);

/**
 * Delete test function.
 * @param f deleting function.
 * @ingroup WritingTestCase
 */
extern inline void ccunit_deleteTestFunc (CCUnitTestFunc* f);

/** @} */

#endif	/* !CCUNITTESTFUNC_H */
