#!/usr/bin/perl -w

# $Id: clamav-update.pl,v 1.5 2006/11/12 12:58:14 okamura Exp $

=head1 NAME

clamav-update.pl - auto update clamav

=head1 SYNOPSIS

 clamav-update.pl [options]
 clamav-update.pl --help|-h
 clamav-update.pl --version|-V

=head1 DESCRIPTION

B<clamav-update.pl> update clamav for ClamXav user.

=cut

use strict;
use utf8;
no encoding;
#-------------------------------------------------------------------------------
# モジュール
#-------------------------------------------------------------------------------
# システム
use Getopt::Long	qw(:config no_ignore_case);
use Pod::Usage;
use Sys::Syslog	qw(:DEFAULT setlogsock);
use Carp;

# プロジェクト


#-------------------------------------------------------------------------------
# グローバル変数宣言
#-------------------------------------------------------------------------------
use vars qw(
	$Version
	$ConfigFile
	%ErrorCode
	$Setting
	$SyslogOpened
	$Force
);


#-------------------------------------------------------------------------------
# サブルーチン
#-------------------------------------------------------------------------------
sub Log;

sub _Syslog {
	my	($priority, $format, @args) = @_;

	syslog $priority, $format, @args;
}

sub _Fdlog {
	my	($priority, $format, @args) = @_;
	my	$now;

	return 0	if (_IsMasked($priority));
	$format = '%s '.$format."\n";
	$now = localtime;
	if (ref($Setting->{logging}->{setlogsock}) eq 'GLOB') { 
		printf {$Setting->{logging}->{setlogsock}} $format, $now, @args;
	}
	else {
		printf STDERR $format, $now, @args;
	}
}

sub _IsMasked {
	my	($priority) = @_;
	my	$numpri = undef;
	my	$numfac = undef;

	foreach (split(/\W+/, $priority, 2)) {
		my	$num = Sys::Syslog::xlate($_);

		if (/^kern$/ or $num < 0) {
			croak "invalid level/facility: $_";
		}
		elsif ($num <= &Sys::Syslog::LOG_PRIMASK) {
			croak "too many levels given: $_"	if defined($numpri);
			$numpri = $num;
			return 1	unless (
				Sys::Syslog::LOG_MASK($numpri) & $Sys::Syslog::maskpri);
		}
		else {
			croak "too many facilities given: $_" if defined($numfac);
			$numfac = $num;
		}
	}

	return 0;
}

sub SetEnv {
	foreach my $varname (keys %{$Setting->{environment}}) {
		$ENV{$varname} = $Setting->{environment}->{$varname};
		Log 'debug', 'setenv %s: %s', $varname, $ENV{$varname};
	}
}

sub Prepare {
	PrepareLogging();
	if (-f $ConfigFile) {
		LoadConfig($ConfigFile);
		SetEnv();
		PrepareLogging();
	}
	else {
		SetEnv();
	}
	Log 'info', 'Start clamav-update.pl --config %s', $ConfigFile;
}

sub Setlogmask {
	my	($level) = @_;
	my	$new;
	my	$old;

	$new = Sys::Syslog::LOG_UPTO(Sys::Syslog::xlate($level));
	$old = setlogmask $new;
	Log 'debug', 'setlogmask %s from %s', $new, $old;
	return $old;
}

sub PrepareLogging {
	if ($SyslogOpened) {
		closelog();
	}

	# setlogmask
	if (defined $Setting->{logging}->{setlogmask}) {
		Setlogmask $Setting->{logging}->{setlogmask};
	}

	# setlogsock
	if (ref($Setting->{logging}->{setlogsock}) eq 'GLOB') {
		*Log = *_Fdlog;
	}
	else {
		if (ref($Setting->{logging}->{setlogsock}) eq 'ARRAY') {
			unless (setlogsock @{$Setting->{logging}->{setlogsock}}) {
				Log 'crit', 'setlogsock failed: [%s]',
					"@{$Setting->{logging}->{setlogsock}}";
			}
		}
		elsif (ref($Setting->{logging}->{setlogsock}) eq '') {
			unless (setlogsock $Setting->{logging}->{setlogsock}) {
				Log 'crit', 'setlogsock failed: %s',
					$Setting->{logging}->{setlogsock};
			}
		}
		else {
			Log 'err', 'unknown logsock type';
			return undef;
		}

		*Log = *_Syslog;
		openlog $Setting->{logging}->{openlog}->{ident},
			$Setting->{logging}->{openlog}->{logopt},
			$Setting->{logging}->{openlog}->{facility};
		$SyslogOpened = 1;
	}
}

sub LoadConfig {
	my	($path) = @_;

	Log 'debug', 'Loading: %s', $path;
	eval {require $path};
	if ($@) {
		Log 'crit', 'Can\'t load the configration file %s: %s', $path, $@;
		exit $ErrorCode{loadConfig};
	}
}

sub Version {
	my	%spec = @_;
	my	($command, $regexp) = @_;
	my	$version;

	Log 'notice', 'execute: %s', $spec{command};
	$version = `$spec{command}`;
	if ($? == -1) {
		Log 'alert', 'fail to execute';
		return '';
	}
	elsif ($? & 127) {
		if ($? & 128) {
			Log 'crit', 'child died with coredump: %d', ($? & 127);
		}
		else {
			Log 'crit', 'child died withoout coredump: %d', ($? & 127);
		}
		return '';
	}
	elsif ($?) {
		Log 'err', 'child exited: %d', $? >> 8;
		return '';
	}

	unless ($version =~ m/$spec{regexp}/) {
		Log 'debug', 'regexp: %s', $spec{regexp};
		Log 'alert', 'unexpected respons: %s', $version;
		return '';
	}

	return $1;
}

sub LatestVersion {
	my	$version = Version(%{$Setting->{version}->{latest}});

	if ($version) {
		Log 'info', 'latest ClamAV version is %s', $version; 
	}
	return $version;
}

sub CurrentVersion {
	return '0'	if ($Force);
	my	$version = Version(%{$Setting->{version}->{current}});

	if ($version) {
		Log 'info', 'current ClamAV version is %s', $version; 
	}
	return $version;
}

sub	DoCommand {
	my	@command = @_;

	Log 'notice', 'execute: %s', "@command";
	system(@command);
	if ($? == -1) {
		Log 'crit', 'fail to execute';
		return 0;
	}
	elsif ($? & 127) {
		if ($? & 128) {
			Log 'crit', 'child died with coredump: %d', ($? & 127);
		}
		else {
			Log 'crit', 'child died without coredump: %d', ($? & 127);
		}
		return 0;
	}
	elsif ($?) {
		Log 'err', 'child exited: %d', $? >> 8;
		return 0;
	}

	return 1;
}

sub Download {
	my	($version) = @_;
	my	$fname = sprintf $Setting->{download}->{name}.'%s',
		$version, $Setting->{download}->{ext};
	my	@command = (
		'curl', '--silent', '-o',
		sprintf('%s/%s', $Setting->{download}->{dst}, $fname),
		sprintf('%s/%s', $Setting->{download}->{src}, $fname)
	);

	return DoCommand(@command);
}

sub Clean {
	my	($version) = @_;
	my	$dname;
	my	$fname;

	$dname = sprintf $Setting->{download}->{name}, $version;
	$fname = $dname.$Setting->{download}->{ext};
	unless (chdir $Setting->{download}->{dst}) {
		Log 'crit', 'can\'t change directory: %s', $Setting->{download}->{dst};
		return;
	}
	if (-f $fname) {
		if ($Setting->{doClean}->{archive}) {
			unless (unlink $fname) {
				Log 'err', 'can\'t delete: %s', $fname;
			}
		}
	}
	else {
		Log 'notice', 'the file doesn\'t exists: %s', $fname;
	}
	if (-d $dname) {
		if ($Setting->{doClean}->{dir}) {
			unless (DoCommand('rm', '-rf', $dname)) {
				Log 'err', 'can\'t delete: %s', $dname;
			}
		}
		elsif ($Setting->{doClean}->{objects}) {
			if (chdir $dname) {
				unless (DoCommand(qw(make clean))) {
					Log 'err', 'can\'t delete: %s', $dname;
				}
			}
			else {
				Log 'err', 'can\'t change directory: %s', $dname;
			}
		}
	}
	else {
		Log 'notice', 'the directory doesn\'t exists: %s', $dname;
	}
}


#-------------------------------------------------------------------------------
# グローバル変数初期値
#-------------------------------------------------------------------------------
$Version = 'clamav-update.pl v1.0.1';

$ConfigFile = "/usr/local/clamXav/etc/clamav-update.conf";

%ErrorCode = (
	arguments	=> 1,
	prepare		=> 2,
	loadConfig	=> 3,
	download	=> 4,
	extract		=> 5,
	build		=> 6,
	install		=> 7,
	clean		=> 8,
);

$Setting = {
	logging	=> {
		setlogsock	=> \*STDERR,
		openlog	=> {
			ident		=> 'clamav-update',
			logopt		=> [qw(cons)],
			facility	=> 'local6',
		},
		setlogmask	=> 'warning',
	},
	environment	=> {
		PATH	=> '/bin:/sbin:/usr/bin:/usr/sbin:/usr/local/clamXav/bin',
	},
	version	=> {
		latest	=> {
			command	=> 'host -t txt current.cvd.clamav.net',
			regexp	=> qr/"(\d+(?:\.\d+)*):\d+:\d+:\d+:[^"]*"\s*$/,
		},
		current	=> {
			command	=> 'clamav-config --version',
			regexp	=> qr/^(\d+(?:\.\d+)*)\s*$/,
		},
	},
	download	=> {
		src	=> 'http://jaist.dl.sourceforge.net/sourceforge/clamav',
		dst	=> '/tmp',
		name	=> 'clamav-%s',
		ext		=> '.tar.gz',
	},
	build	=> [
		[qw(./configure --prefix=/usr/local/clamXav)],
		[qw(make)],
	],
	install	=> [
		[qw(make install)],
		[qw(chown -R root:admin /usr/local/clamXav)],
		[qw(chmod 664 /usr/local/clamXav/etc/freshclam.conf)],
		[qw(chown clamav /usr/local/clamXav/bin/freshclam)],
		[qw(chmod u+s /usr/local/clamXav/bin/freshclam)],
		[qw(touch /usr/local/clamXav/share/clamav/freshclam.log)],
		[qw(chmod 664 /usr/local/clamXav/share/clamav/freshclam.log)],
		[qw(chown -R clamav:clamav /usr/local/clamXav/share/clamav)],
		[qw(chmod -R ug+w /usr/local/clamXav/share/clamav)],
	],
	daemon	=> [
#		[qw(chown root /usr/local/clamXav/bin/freshclam)],
#		[qw(chmod u-s /usr/local/clamXav/bin/freshclam)],
#		[qw(/Library/StartupItems/FreshClamDaemon/FreshClamDaemon restart)],
#		[qw(/Library/StartupItems/ClamAntiVirusDaemon/ClamAntiVirusDaemon restart)],
	],
	doClean	=> {
		archive	=> 1,
		objects	=> 0,
		dir		=> 1,
	},
};

$SyslogOpened = 0;
*Log = *_Fdlog;

$Force = 0;


#-------------------------------------------------------------------------------
# 引数解釈
#-------------------------------------------------------------------------------

=head1 OPTIONS

=over 4

=item -h|--help

Display help.

=item -V|--version

Display version.

=item -c|--config I<filepath>

Set the configuration file to I<filepath>.
DEFAULT: /usr/local/clamXav/etc/clamav-update.conf

=item -f|--force

Force install latest version.

=back

=cut

GetOptions(
	'h|help'		=> sub {
		PrepareLogging();
		pod2usage(-verbose => 2, -exitval => 0)
	},
	'V|version'		=> sub {
		PrepareLogging();
		print <<EOF;
$Version
Copyright (C) 2006 OKAMURA Yuji, All rights reserved.
This is free software; see the source for copying conditions. There is NO 
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
EOF
		exit 0;
	},
	'c|config=s'	=> \$ConfigFile,
	'f|force'		=> \$Force,
) or pod2usage($ErrorCode{arguments});


#-------------------------------------------------------------------------------
# メイン
#-------------------------------------------------------------------------------
# 初期化
Prepare();
if ($> != 0) {
	Log 'alert', 'effective user is not root: %d', $>;
	exit $ErrorCode{prepare};
}

# アップデート
{
	my	$latestVersion;

	if (CurrentVersion() lt ($latestVersion = LatestVersion())) {
		my	$dname = sprintf $Setting->{download}->{name}, $latestVersion;
		my	$fname = $dname.$Setting->{download}->{ext};

		# clean
		{
			my	$doClean = $Setting->{doClean};

			$Setting->{doClean} = {
				archive	=> 0,
				objects	=> 0,
				dir		=> 1,
			};
			Clean($latestVersion);
			$Setting->{doClean} = $doClean;
		}

		# download
		unless (chdir $Setting->{download}->{dst}) {
			Log 'crit', 'can\'t change directory: %s',
				$Setting->{download}->{dst};
			exit $ErrorCode{download};
		}
		unless (Download($latestVersion)) {
			Clean($latestVersion);
			exit $ErrorCode{download};
		}

		# extract
		unless (DoCommand('tar', 'xfz', $fname)) {
			Clean($latestVersion);
			exit $ErrorCode{extract};
		}
		unless (chdir $dname) {
			Log 'emerg', 'can\'t change directory: %s', $dname;
			exit $ErrorCode{extract};
		}

		# build
		foreach (@{$Setting->{build}}) {
			unless (DoCommand(@{$_})) {
				Clean($latestVersion);
				exit $ErrorCode{build};
			}
		}

		# install
		foreach (@{$Setting->{install}}) {
			unless (DoCommand(@{$_})) {
				Clean($latestVersion);
				exit $ErrorCode{install};
			}
		}
		foreach (@{$Setting->{daemon}}) {
			unless (DoCommand(@{$_})) {
				Clean($latestVersion);
				exit $ErrorCode{install};
			}
		}

		# clean
		unless (Clean($latestVersion)) {
			exit $ErrorCode{clean};
		}
	}
}

END {
	Log 'info', 'Finish clamav-update.pl';
	closelog()	if ($SyslogOpened);
}

#===============================================================================

=head1 CHANGE LOG

=over 4

=item 1.0.1

 * Improve output contents of log.
 * Fix miss spelling in clamav-update.conf.

=item 1.0

Initial version.

=back

=head1 AUTHOR

OKAMURA Yuji E<lt>okamura@users.sourceforge.jpE<gt>

=cut

__END__;
