#!/usr/local/bin/ruby
#
# $Id: $
#
# LICENSE: GPL
#   USAGE: ./codediff.rb <repository path> <output path> <URL without slash at last>
#
require "codediff/utility.rb"
require "codediff/diff2html.rb"
require "codediff/diff2rss20.rb"

module CodeDiff

  class FileList

    include CodeDiff::HtmlConf

    @baseURL
    @command

    def initialize(baseURL, command)
      @baseURL = baseURL
      @command = command
    end

    def printHtml(ostream)
      ostream << <<HEADER
<html>
  <head><link rel=\"stylesheet\" type=\"text/css\" href=\"codediff.css\" />
  </head><body><h4 class=\"title\">Changed Files</h4><ul>
HEADER
      `#{@command}`.split("\n").each do |line|
        if line.match(/^([ACDGMPU]+)\s+(.*)$/) then
          ostream.puts "<li><a href=\"#{@baseURL}##{$2}\" target=\"diff\"><span class=\"#{$1}\">#{$2}</span></a></li>"
        end
      end
      ostream.puts "</ul><p class=\"timestamp\">Created on #{Time.now}</p></body></html>"
    end
  end

  class CodeDiffApp

    include CodeDiff::HtmlConf
    include CodeDiff::RcsConf

    @relPath
    @baseURL
    @listFileName
    @diffFileName

    def initialize(repoPath, relPath, baseURL, listFileName, diffFileName)
      Dir.chdir(repoPath)
      @relPath = relPath
      @baseURL = baseURL
      @listFileName = listFileName
      @diffFileName = diffFileName
    end

    def run
      prev = revision()

      File.open("#{@relPath}/#{@listFileName}", "w") do |f|
        fileList = CodeDiff::FileList.new(@diffFileName, cmdUpdate())
        fileList.printHtml(f)
      end

      title = titleString(prev, revision())

      File.open("#{@relPath}/#{@diffFileName}", "w") do |f|
        fileDiff = CodeDiff::Diff2Html.new(f)
        fileDiff.parse(`#{cmdFullDiff(prev)}`.split("\n"))
      end

      File.open("#{@relPath}/diff.rss", "w") do |f|
        rssWriter = CodeDiff::Diff2Rss20.new(f, "#{@baseURL}/#{@diffFileName}")
        rssWriter.parse(`#{cmdDiff(prev)}`)
      end

      File.open("#{@relPath}/index.html", "w") do |f|
        f.print <<"ROOTHTML"
<html>
  <head>
    <title>CodeDiff Result: #{@relPath}</title>
    <link rel="alternate" type="application/rss+xml" title="CodeDiff RSS" href="#{@baseURL}/diff.rss">
  </head>
  <frameset cols=\"30%,*\">
    <frameset rows=\"35%,*\">
      <frame src=\"date.html\" name=\"date\" marginwidth=\"0\"/>
      <frame src=\"about:blank\" name=\"list\" marginwidth=\"0\"/>
    </frameset>
    <frame src=\"about:blank\" name=\"diff\" marginwidth=\"0\"/>
  </frameset>
</html>
ROOTHTML
      end

      begin
        dateHtml = File.open("#{@relPath}/date.html", "r")
      rescue
        File.open("#{@relPath}/date.html", "w") do |f|
          f.puts "<html><head><link rel=\"stylesheet\" type=\"text/css\" href=\"codediff.css\" /></head><body><ul>\n</ul></body></html>"
        end
      end

      lines = nil
      File.open("#{@relPath}/date.html", "r") do |f|
        lines = f.readlines
        lines[1, 0] = ["<li><a href=\"#{@listFileName}\" target=\"list\">#{title}</a></li>\n"]
      end

      File.open("#{@relPath}/date.html", "w") do |f|
        lines.each do |line|
          f.write(line)
        end
      end

      begin
        css = File.open("#{@relPath}/codediff.css", "r")
      rescue
        File.open("#{@relPath}/codediff.css", "w") do |f|
          printCss(f)
        end
      end
    end
  end
end

if $0 == __FILE__ then
  RepoPath     = ARGV[0]
  RelPath      = ARGV[1]
  BaseURL      = ARGV[2]
  TimeStamp    = Time.now
  ListFileName = "#{TimeStamp.strftime("%Y%m%d_%H%M%S_%a")}_list.html"
  DiffFileName = "#{TimeStamp.strftime("%Y%m%d_%H%M%S_%a")}_diff.html"
  app = CodeDiff::CodeDiffApp.new(RepoPath, RelPath, BaseURL, ListFileName, DiffFileName)
  app.run
end
