#!/usr/local/bin/ruby
#
# $Id: $
#
# LICENSE: GPL
# EXAMPLE: % diff -u  | ./diff2html.rb > output.html
#          % svn diff | ./diff2html.rb > output.html
#          % cvs diff --unified=10 --ignore-all-space | ./diff2html.rb > output.html
#
require "codediff/diffparser.rb"
require "codediff/utility.rb"
require "codediff/config.rb"

module CodeDiff
  class Diff2Html < DiffParser

    include CodeDiff::XmlTool
    include CodeDiff::HtmlConf

    @ostream
    @fileCount
    @diffCount
    @nextBlock

    def initialize(os)
      @ostream = os
      @fileCount = 0
      @diffCount = 0
      @nextBlock = false
    end

    #
    # Override
    #

    def startParse
      @ostream.puts "<html>\n<head><title>diff2html result</title><style>"
      printCss(@ostream)
      @ostream.puts "</style></head>\n<body>\n<table cellspacing=\"0\" cellpadding=\"0\">"
    end

    def endParse
      @ostream.puts "</table>\n</body>\n</html>"
    end

    def startDocument(source, target)
      source = sanitize(source)
      target = sanitize(target)
      @ostream.puts "<tr><th class=\"title\"><a name=\"file_#{@fileCount}\"/><a name=\"#{source}\"/><a href=\"#file_#{@fileCount+1}\">#{source} vs #{target}</a></th></tr>"
      @fileCount = @fileCount + 1
    end

    def endDocument
    end

    def startBlock(sourceBase, sourceLength, targetBase, targetLength)
      @ostream.puts "<tr><td class=\"omit\">(From line #{sourceBase} to #{sourceBase.to_i+sourceLength.to_i})</td></tr>"
      @nextBlock = true
    end

    def sourceLine(line)
      line = sanitize(line)
      if @nextBlock then
        @diffCount = @diffCount + 1
        @nextBlock = false
        @ostream.puts "<tr><td class=\"del\"><a name=\"diff_#{@diffCount}\"/><a href=\"#diff_#{@diffCount+1}\">#{line}</a></td></tr>"
      else
        @ostream.puts "<tr><td class=\"del\"><a href=\"#diff_#{@diffCount+1}\">#{line}</a></td></tr>"
      end
    end

    def targetLine(line)
      line = sanitize(line)
      if @nextBlock then
        @diffCount = @diffCount + 1
        @nextBlock = false
        @ostream.puts "<tr><td class=\"add\"><a name=\"diff_#{@diffCount}\"/><a href=\"#diff_#{@diffCount+1}\">#{line}</a></td></tr>"
      else
        @ostream.puts "<tr><td class=\"add\"><a href=\"#diff_#{@diffCount+1}\">#{line}</a></td></tr>"
      end
    end

    def commonLine(line)
      line = sanitize(line)
      @ostream.puts "<tr><td class=\"same\">#{line}</td></tr>"
      @nextBlock = true
    end

  end
end

if $0 == __FILE__ then
  app = CodeDiff::Diff2Html.new(STDOUT)
  app.parse(STDIN)
end
