/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.model;

import java.math.*;
import java.sql.*;
import java.util.*;
import java.util.Date;
import java.util.logging.*;
import org.compiere.util.*;

/**
 *	Billing Callouts	
 *	
 *  @author Jirimuto
 *  @version $Id: CalloutBilling.java,v 1.1 2010/04/09 09:14:45 jrmt Exp $
 */
public class CalloutBilling extends CalloutEngine
{
	/**
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String invoice (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_Invoice_ID = (Integer)value;
		if (C_Invoice_ID == null || C_Invoice_ID.intValue() == 0)
			return "";
		
		boolean IsSOTrx = ctx.getContext(WindowNo, "IsSOTrx").equals("Y");
		MInvoice invoice = MInvoice.get(ctx, C_Invoice_ID);
		if ( IsSOTrx != invoice.isSOTrx() )
			return "";
		int billingDocType = 0;
		if( IsSOTrx ){
			MDocType[] docTypes = MDocType.getOfDocBaseType(ctx, "BRI");
			billingDocType = docTypes[0].getC_DocType_ID();
		} else {
			MDocType[] docTypes = MDocType.getOfDocBaseType(ctx, "BPI");
			billingDocType = docTypes[0].getC_DocType_ID();
		}
		
		mTab.setValue("C_DocTypeTarget_ID", new Integer(billingDocType));
		mTab.setValue("DateInvoiced", invoice.getDateInvoiced());
		
		if( invoice.getC_Activity_ID() != 0 )
			mTab.setValue("C_Activity_ID", new Integer(invoice.getC_Activity_ID()));
		if( invoice.getC_Campaign_ID() != 0 )
			mTab.setValue("C_Campaign_ID", new Integer(invoice.getC_Campaign_ID()));
		
		if( invoice.getC_Project_ID() != 0 )
			mTab.setValue("C_Project_ID", new Integer(invoice.getC_Project_ID()));
		if( invoice.getC_Order_ID() != 0 )
			mTab.setValue("C_Order_ID", new Integer(invoice.getC_Order_ID()));
		if( invoice.getC_Charge_ID() != 0 )
			mTab.setValue("C_Charge_ID", new Integer(invoice.getC_Charge_ID()));
		if( invoice.getC_ConversionType_ID() != 0 )
			mTab.setValue("C_ConversionType_ID", new Integer(invoice.getC_ConversionType_ID()));
		if( invoice.getC_CashLine_ID() != 0 )
			mTab.setValue("C_CashLine_ID", new Integer(invoice.getC_CashLine_ID()));
		mTab.setValue("DateOrdered", invoice.getDateOrdered());
		mTab.setValue("Description", invoice.getDescription());
		mTab.setValue("POReference", invoice.getPOReference());
		if( invoice.getSalesRep_ID() != 0 )
			mTab.setValue("SalesRep_ID", new Integer(invoice.getSalesRep_ID()));
		if( invoice.getUser1_ID() != 0 )
			mTab.setValue("User1_ID", new Integer(invoice.getUser1_ID()));
		if( invoice.getUser2_ID() != 0 )
			mTab.setValue("User2_ID", new Integer(invoice.getUser2_ID()));
		
		mTab.setValue("C_BPartner_ID", new Integer(invoice.getC_BPartner_ID()));
		mTab.setValue("C_BPartner_Location_ID", new Integer(invoice.getC_BPartner_Location_ID()));
		mTab.setValue("AD_User_ID", new Integer(invoice.getAD_User_ID()));
		
		mTab.setValue("M_PriceList_ID", new Integer(invoice.getM_PriceList_ID()));
		mTab.setValue("C_Currency_ID", new Integer(invoice.getC_Currency_ID()));
		mTab.setValue("C_PaymentTerm_ID", new Integer(invoice.getC_PaymentTerm_ID()));
		mTab.setValue("PaymentRule", invoice.getPaymentRule());
			
		return "";
	}	//	invocie

	/**
	 *	Billing Header - DocType.
	 *		- PaymentRule
	 *		- temporary Document
	 *  Context:
	 *  	- DocSubTypeSO
	 *		- HasCharges
	 *	- (re-sets Business Partner info of required)
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String docType (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_DocType_ID = (Integer)value;
		if (C_DocType_ID == null || C_DocType_ID.intValue() == 0)
			return "";

		String sql = "SELECT d.HasCharges,'N',d.IsDocNoControlled,"
			+ "s.CurrentNext, d.DocBaseType "
			+ "FROM C_DocType d "
			+ "LEFT OUTER JOIN AD_Sequence s ON (d.DocNoSequence_ID=s.AD_Sequence_ID) "
			+ "WHERE C_DocType_ID=?";		//	1
		try
		{
			PreparedStatement pstmt = DB.prepareStatement(sql, null);
			pstmt.setInt(1, C_DocType_ID.intValue());
			ResultSet rs = pstmt.executeQuery();
			if (rs.next())
			{
				//	Charges - Set Context
				ctx.setContext(WindowNo, "HasCharges", rs.getString(1));
				//	DocumentNo
				if (rs.getString(3).equals("Y"))
					mTab.setValue("DocumentNo", "<" + rs.getString(4) + ">");
				//  DocBaseType - Set Context
				String s = rs.getString(5);
				ctx.setContext(WindowNo, "DocBaseType", s);
				//  AP Check & AR Credit Memo
				if (s.startsWith("BP"))
					mTab.setValue("PaymentRule", "S");    //  Check
				else if (s.endsWith("C"))
					mTab.setValue("PaymentRule", "P");    //  OnCredit
			}
			rs.close();
			pstmt.close();
		}
		catch (SQLException e)
		{
			log.log(Level.SEVERE, sql, e);
			return e.getLocalizedMessage();
		}
		return "";
	}	//	docType

	/**
	 *	Billing Header- BPartner.
	 *		- M_PriceList_ID (+ Context)
	 *		- C_BPartner_Location_ID
	 *		- AD_User_ID
	 *		- POReference
	 *		- SO_Description
	 *		- IsDiscountPrinted
	 *		- PaymentRule
	 *		- C_PaymentTerm_ID
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String bPartner (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_BPartner_ID = (Integer)value;
		if (C_BPartner_ID == null || C_BPartner_ID.intValue() == 0)
			return "";

		String sql = "SELECT p.AD_Language,p.C_PaymentTerm_ID,"
			+ " COALESCE(p.M_PriceList_ID,g.M_PriceList_ID) AS M_PriceList_ID, p.PaymentRule,p.POReference,"
			+ " p.SO_Description,p.IsDiscountPrinted,"
			+ " p.SO_CreditLimit, p.SO_CreditLimit-p.SO_CreditUsed AS CreditAvailable,"
			+ " l.C_BPartner_Location_ID,c.AD_User_ID,"
			+ " COALESCE(p.PO_PriceList_ID,g.PO_PriceList_ID) AS PO_PriceList_ID, p.PaymentRulePO,p.PO_PaymentTerm_ID " 
			+ "FROM C_BPartner p"
			+ " INNER JOIN C_BP_Group g ON (p.C_BP_Group_ID=g.C_BP_Group_ID)"			
			+ " LEFT OUTER JOIN C_BPartner_Location l ON (p.C_BPartner_ID=l.C_BPartner_ID AND l.IsBillTo='Y' AND l.IsActive='Y')"
			+ " LEFT OUTER JOIN AD_User c ON (p.C_BPartner_ID=c.C_BPartner_ID) "
			+ "WHERE p.C_BPartner_ID=? AND p.IsActive='Y'";		//	#1

		boolean IsSOTrx = ctx.getContext(WindowNo, "IsSOTrx").equals("Y");
		try
		{
			PreparedStatement pstmt = DB.prepareStatement(sql, null);
			pstmt.setInt(1, C_BPartner_ID.intValue());
			ResultSet rs = pstmt.executeQuery();
			//
			if (rs.next())
			{
				//	PriceList & IsTaxIncluded & Currency
				Integer ii = new Integer(rs.getInt(IsSOTrx ? "M_PriceList_ID" : "PO_PriceList_ID"));
				if (!rs.wasNull())
					mTab.setValue("M_PriceList_ID", ii);
				else
				{	//	get default PriceList
					int i = ctx.getContextAsInt("#M_PriceList_ID");
					if (i != 0)
						mTab.setValue("M_PriceList_ID", new Integer(i));
				}

				//	PaymentRule
				String s = rs.getString(IsSOTrx ? "PaymentRule" : "PaymentRulePO");
				if (s != null && s.length() != 0)
				{
					if (ctx.getContext(WindowNo, "DocBaseType").endsWith("C"))	//	Credits are Payment Term
						s = "P";
					else if (IsSOTrx && (s.equals("S") || s.equals("U")))	//	No Check/Transfer for SO_Trx
						s = "P";											//  Payment Term
					mTab.setValue("PaymentRule", s);
				}
				//  Payment Term
				ii = new Integer(rs.getInt(IsSOTrx ? "C_PaymentTerm_ID" : "PO_PaymentTerm_ID"));
				if (!rs.wasNull())
					mTab.setValue("C_PaymentTerm_ID", ii);

				//	Location
				int locID = rs.getInt("C_BPartner_Location_ID");
				//	overwritten by InfoBP selection - works only if InfoWindow
				//	was used otherwise creates error (uses last value, may belong to differnt BP)
				if (C_BPartner_ID.toString().equals(ctx.getContext(Env.WINDOW_INFO, Env.TAB_INFO, "C_BPartner_ID")))
				{
					String loc = ctx.getContext(Env.WINDOW_INFO, Env.TAB_INFO, "C_BPartner_Location_ID");
					if (loc.length() > 0)
						locID = Integer.parseInt(loc);
				}
				if (locID == 0)
					mTab.setValue("C_BPartner_Location_ID", null);
				else
					mTab.setValue("C_BPartner_Location_ID", new Integer(locID));

				//	Contact - overwritten by InfoBP selection
				int contID = rs.getInt("AD_User_ID");
				if (C_BPartner_ID.toString().equals(ctx.getContext(Env.WINDOW_INFO, Env.TAB_INFO, "C_BPartner_ID")))
				{
					String cont = ctx.getContext(Env.WINDOW_INFO, Env.TAB_INFO, "AD_User_ID");
					if (cont.length() > 0)
						contID = Integer.parseInt(cont);
				}
				if (contID == 0)
					mTab.setValue("AD_User_ID", null);
				else
					mTab.setValue("AD_User_ID", new Integer(contID));

				//	CreditAvailable
				if (IsSOTrx)
				{
					double CreditLimit = rs.getDouble("SO_CreditLimit");
					if (CreditLimit != 0)
					{
						double CreditAvailable = rs.getDouble("CreditAvailable");
						if (!rs.wasNull() && CreditAvailable < 0)
							mTab.fireDataStatusEEvent("CreditLimitOver",
								DisplayType.getNumberFormat(DisplayType.Amount).format(CreditAvailable),
								false);
					}
				}
				
				//	PO Reference
				s = rs.getString("POReference");
				if (s != null && s.length() != 0)
					mTab.setValue("POReference", s);
				else
					mTab.setValue("POReference", null);
				//	SO Description
				s = rs.getString("SO_Description");
				if (s != null && s.trim().length() != 0)
					mTab.setValue("Description", s);
				//	IsDiscountPrinted
				s = rs.getString("IsDiscountPrinted");
				if (s != null && s.length() != 0)
					mTab.setValue("IsDiscountPrinted", s);
				else
					mTab.setValue("IsDiscountPrinted", "N");
			}
			rs.close();
			pstmt.close();
			
			if (IsSOTrx)
			{
				Timestamp billingDate = (Timestamp)mTab.getValue("DateBilling");
				MBilling lastBilling = MBilling.getLastBilling(ctx, C_BPartner_ID, billingDate, null);
				if( lastBilling != null ){
					BigDecimal lastInvoicedAmt = lastBilling.getTotalInvoiceAmt();
					mTab.setValue("LastInvoicedAmt", lastBilling.getTotalInvoiceAmt());
				
					BigDecimal lastPaid = MPayment.getSumOfBPartner(ctx, C_BPartner_ID, lastBilling.getDateBilling(), billingDate, null);
					mTab.setValue("LastPaidAmt", lastPaid);
					BigDecimal adjustAmt = (BigDecimal)mTab.getValue("AdjustAmt");
					if( adjustAmt == null ){
						adjustAmt=BigDecimal.ZERO;
						mTab.setValue("AdjustAmt", adjustAmt);
					}
					BigDecimal lastTansferedAmt = lastInvoicedAmt.subtract(lastPaid).add(adjustAmt);
					mTab.setValue("LastTransferedAmt", lastTansferedAmt);
				
					mTab.setValue("IsAdjusted", "N");
				}

			}			
		}
		catch (SQLException e)
		{
			log.log(Level.SEVERE, "bPartner", e);
			return e.getLocalizedMessage();
		}

		return "";
	}	//	bPartner

	/**
	 *	Set Payment Term.
	 *	Payment Term has changed 
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String paymentTerm (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_PaymentTerm_ID = (Integer)value;
		int C_Billing_ID = ctx.getContextAsInt(WindowNo, "C_Billing_ID");
		if (C_PaymentTerm_ID == null || C_PaymentTerm_ID.intValue() == 0
			|| C_Billing_ID == 0)	//	not saved yet
			return "";
		//
		MPaymentTerm pt = new MPaymentTerm (ctx, C_PaymentTerm_ID.intValue(), null);
		if (pt.get_ID() == 0)
			return "PaymentTerm not found";
				
		return "";
	}	//	paymentTerm

	
	/**
	 *	Billing - Transfered Amount.
	 *		- called from QtyBilling, PriceActual
	 *		- calculates LineNetAmt
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String transferedAmt (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		if (isCalloutActive() || value == null)
			return "";
		setCalloutActive(true);

		BigDecimal LastInvoicedAmt, LastPaidAmt, AdjustAmt, LastTransferedAmt, RevenueNetAmt, TaxAmt, TotalInvoiceAmt;
		//	get values
		LastInvoicedAmt = (BigDecimal)mTab.getValue("LastInvoicedAmt");
		LastPaidAmt = (BigDecimal)mTab.getValue("LastPaidAmt");
		AdjustAmt = (BigDecimal)mTab.getValue("AdjustAmt");
		if( LastInvoicedAmt == null ){
			LastInvoicedAmt = BigDecimal.ZERO;
			mTab.setValue("LastInvoicedAmt", LastInvoicedAmt);
		}
		if( LastPaidAmt == null ){
			LastPaidAmt = BigDecimal.ZERO;
			mTab.setValue("LastPaidAmt", LastPaidAmt);
		}
		if( AdjustAmt == null ){
			AdjustAmt = BigDecimal.ZERO;
			mTab.setValue("AdjustAmt", AdjustAmt);
		}
		
		LastTransferedAmt = LastInvoicedAmt.subtract(LastPaidAmt).add(AdjustAmt);
		mTab.setValue("LastTransferedAmt", LastTransferedAmt);
		
		RevenueNetAmt = (BigDecimal)mTab.getValue("RevenueNetAmt");
		TaxAmt = (BigDecimal)mTab.getValue("TaxAmt");
		if( RevenueNetAmt == null ){
			RevenueNetAmt = BigDecimal.ZERO;
			mTab.setValue("RevenueNetAmt", RevenueNetAmt);
		}
		if( TaxAmt == null ){
			TaxAmt = BigDecimal.ZERO;
			mTab.setValue("TaxAmt", TaxAmt);
		}
		TotalInvoiceAmt = LastTransferedAmt.add(RevenueNetAmt).add(TaxAmt);
		mTab.setValue("TotalInvoiceAmt", TotalInvoiceAmt);
		
		setCalloutActive(false);
		
		return "";
	}	//	transferedAmt

	
	
	/**************************************************************************
	 *	Billing Line - Product.
	 *		- reset C_Charge_ID / M_AttributeSetInstance_ID
	 *		- PriceList, PriceStd, PriceLimit, C_Currency_ID, EnforcePriceLimit
	 *		- UOM
	 *	Calls Tax
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String product (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer M_Product_ID = (Integer)value;
		if (M_Product_ID == null || M_Product_ID.intValue() == 0)
			return "";
		setCalloutActive(true);
		mTab.setValue("C_Charge_ID", null);
		
		//	Set Attribute
		if (ctx.getContextAsInt(Env.WINDOW_INFO, Env.TAB_INFO, "M_Product_ID") == M_Product_ID.intValue()
			&& ctx.getContextAsInt(Env.WINDOW_INFO, Env.TAB_INFO, "M_AttributeSetInstance_ID") != 0)
			mTab.setValue("M_AttributeSetInstance_ID", new Integer(ctx.getContextAsInt(Env.WINDOW_INFO, Env.TAB_INFO, "M_AttributeSetInstance_ID")));
		else
			mTab.setValue("M_AttributeSetInstance_ID", null);

		/*****	Price Calculation see also qty	****/
		boolean IsSOTrx = ctx.getContext(WindowNo, "IsSOTrx").equals("Y");
		int C_BPartner_ID = ctx.getContextAsInt(WindowNo, WindowNo, "C_BPartner_ID");
		BigDecimal Qty = (BigDecimal)mTab.getValue("QtyBilling");
		MProductPricing pp = new MProductPricing (M_Product_ID.intValue(), C_BPartner_ID, Qty, IsSOTrx);
		//
		int M_PriceList_ID = ctx.getContextAsInt(WindowNo, "M_PriceList_ID");
		pp.setM_PriceList_ID(M_PriceList_ID);
		int M_PriceList_Version_ID = ctx.getContextAsInt(WindowNo, "M_PriceList_Version_ID");
		pp.setM_PriceList_Version_ID(M_PriceList_Version_ID);
		Timestamp date = ctx.getContextAsDate(WindowNo, "DateBilling");
		pp.setPriceDate(date);
		//		
		mTab.setValue("PriceList", pp.getPriceList());
		mTab.setValue("PriceLimit", pp.getPriceLimit());
		mTab.setValue("PriceActual", pp.getPriceStd());
		mTab.setValue("PriceEntered", pp.getPriceStd());
		mTab.setValue("C_Currency_ID", new Integer(pp.getC_Currency_ID()));
	//	mTab.setValue("Discount", pp.getDiscount());
		mTab.setValue("C_UOM_ID", new Integer(pp.getC_UOM_ID()));
		ctx.setContext(WindowNo, "EnforcePriceLimit", pp.isEnforcePriceLimit() ? "Y" : "N");
		ctx.setContext(WindowNo, "DiscountSchema", pp.isDiscountSchema() ? "Y" : "N");
		//
		setCalloutActive(false);
		return tax (ctx, WindowNo, mTab, mField, value);
	}	//	product

	/**
	 *	Billing Line - Charge.
	 * 		- updates PriceActual from Charge
	 * 		- sets PriceLimit, PriceList to zero
	 * 	Calles tax
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String charge (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_Charge_ID = (Integer)value;
		if (C_Charge_ID == null || C_Charge_ID.intValue() == 0)
			return "";
			
		//	No Product defined
		if (mTab.getValue("M_Product_ID") != null)
		{
			mTab.setValue("C_Charge_ID", null);
			return "ChargeExclusively";
		}
		mTab.setValue("M_AttributeSetInstance_ID", null);
		mTab.setValue("S_ResourceAssignment_ID", null);
		mTab.setValue("C_UOM_ID", new Integer(100));	//	EA

		ctx.setContext(WindowNo, "DiscountSchema", "N");
		String sql = "SELECT ChargeAmt FROM C_Charge WHERE C_Charge_ID=?";
		try
		{
			PreparedStatement pstmt = DB.prepareStatement(sql, null);
			pstmt.setInt(1, C_Charge_ID.intValue());
			ResultSet rs = pstmt.executeQuery();
			if (rs.next())
			{
				mTab.setValue ("PriceEntered", rs.getBigDecimal (1));
				mTab.setValue ("PriceActual", rs.getBigDecimal (1));
				mTab.setValue ("PriceLimit", Env.ZERO);
				mTab.setValue ("PriceList", Env.ZERO);
				mTab.setValue ("Discount", Env.ZERO);
			}
			rs.close();
			pstmt.close();
		}
		catch (SQLException e)
		{
			log.log(Level.SEVERE, sql + e);
			return e.getLocalizedMessage();
		}
		//
		return tax (ctx, WindowNo, mTab, mField, value);
	}	//	charge


	/**
	 *	Billing Line - Tax.
	 *		- basis: Product, Charge, BPartner Location
	 *		- sets C_Tax_ID
	 *  Calles Amount
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String tax (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		String column = mField.getColumnName();
		if (value == null)
			return "";

		//	Check Product
		int M_Product_ID = 0;
		if (column.equals("M_Product_ID"))
			M_Product_ID = ((Integer)value).intValue();
		else
			M_Product_ID = ctx.getContextAsInt(WindowNo, "M_Product_ID");
		int C_Charge_ID = 0;
		if (column.equals("C_Charge_ID"))
			C_Charge_ID = ((Integer)value).intValue();
		else
			C_Charge_ID = ctx.getContextAsInt(WindowNo, "C_Charge_ID");
		log.fine("Product=" + M_Product_ID + ", C_Charge_ID=" + C_Charge_ID);
		if (M_Product_ID == 0 && C_Charge_ID == 0)
			return amt (ctx, WindowNo, mTab, mField, value);	//

		//	Check Partner Location
		int shipC_BPartner_Location_ID = ctx.getContextAsInt(WindowNo, "C_BPartner_Location_ID");
		if (shipC_BPartner_Location_ID == 0)
			return amt (ctx, WindowNo, mTab, mField, value);	//
		log.fine("Ship BP_Location=" + shipC_BPartner_Location_ID);
		int billC_BPartner_Location_ID = shipC_BPartner_Location_ID;
		log.fine("Bill BP_Location=" + billC_BPartner_Location_ID);

		//	Dates
		Timestamp billDate = ctx.getContextAsDate(WindowNo, "DateBilling");
		log.fine("Bill Date=" + billDate);
		Timestamp shipDate = billDate;
		log.fine("Ship Date=" + shipDate);

		int AD_Org_ID = ctx.getContextAsInt(WindowNo, "AD_Org_ID");
		log.fine("Org=" + AD_Org_ID);

		int M_Warehouse_ID = ctx.getContextAsInt("#M_Warehouse_ID");
		log.fine("Warehouse=" + M_Warehouse_ID);

		//
		int C_Tax_ID = Tax.get(ctx, M_Product_ID, C_Charge_ID, billDate, shipDate,
			AD_Org_ID, M_Warehouse_ID, billC_BPartner_Location_ID, shipC_BPartner_Location_ID,
			ctx.getContext(WindowNo, "IsSOTrx").equals("Y"));
		log.info("Tax ID=" + C_Tax_ID);
		//
		if (C_Tax_ID == 0)
			mTab.fireDataStatusEEvent(CLogger.retrieveError());
		else
			mTab.setValue("C_Tax_ID", new Integer(C_Tax_ID));
		//
		return amt (ctx, WindowNo, mTab, mField, value);
	}	//	tax


	/**
	 *	Billing - Amount.
	 *		- called from QtyBilling, PriceActual
	 *		- calculates LineNetAmt
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String amt (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		if (isCalloutActive() || value == null)
			return "";
		setCalloutActive(true);

	//	log.log(Level.WARNING,"amt - init");
		int C_UOM_To_ID = ctx.getContextAsInt(WindowNo, "C_UOM_ID");
		int M_Product_ID = ctx.getContextAsInt(WindowNo, "M_Product_ID");
		int M_PriceList_ID = ctx.getContextAsInt(WindowNo, "M_PriceList_ID");
		int StdPrecision = MPriceList.getPricePrecision(ctx, M_PriceList_ID);
		BigDecimal QtyEntered, QtyBilling, PriceEntered, PriceActual, PriceLimit, Discount, PriceList;
		//	get values
		QtyEntered = (BigDecimal)mTab.getValue("QtyEntered");
		QtyBilling = (BigDecimal)mTab.getValue("QtyBilling");
		log.fine("QtyEntered=" + QtyEntered + ", Billing=" + QtyBilling + ", UOM=" + C_UOM_To_ID);
		//
		PriceEntered = (BigDecimal)mTab.getValue("PriceEntered");
		PriceActual = (BigDecimal)mTab.getValue("PriceActual");
	//	Discount = (BigDecimal)mTab.getValue("Discount");
		PriceLimit = (BigDecimal)mTab.getValue("PriceLimit");
		PriceList = (BigDecimal)mTab.getValue("PriceList");
		log.fine("PriceList=" + PriceList + ", Limit=" + PriceLimit + ", Precision=" + StdPrecision);
		log.fine("PriceEntered=" + PriceEntered + ", Actual=" + PriceActual);// + ", Discount=" + Discount);

		//	Qty changed - recalc price
		if ((mField.getColumnName().equals("QtyBilling") 
			|| mField.getColumnName().equals("QtyEntered")
			|| mField.getColumnName().equals("M_Product_ID")) 
			&& !"N".equals(ctx.getContext(WindowNo, "DiscountSchema")))
		{
			int C_BPartner_ID = ctx.getContextAsInt(WindowNo, "C_BPartner_ID");
			if (mField.getColumnName().equals("QtyEntered"))
				QtyBilling = MUOMConversion.convertProductTo (ctx, M_Product_ID, 
					C_UOM_To_ID, QtyEntered);
			if (QtyBilling == null)
				QtyBilling = QtyEntered;
			boolean IsSOTrx = ctx.getContext(WindowNo, "IsSOTrx").equals("Y");
			MProductPricing pp = new MProductPricing (M_Product_ID, C_BPartner_ID, QtyBilling, IsSOTrx);
			pp.setM_PriceList_ID(M_PriceList_ID);
			int M_PriceList_Version_ID = ctx.getContextAsInt(WindowNo, "M_PriceList_Version_ID");
			pp.setM_PriceList_Version_ID(M_PriceList_Version_ID);
			Timestamp date = ctx.getContextAsDate(WindowNo, "DateBilling");
			pp.setPriceDate(date);
			//
			PriceEntered = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, pp.getPriceStd());
			if (PriceEntered == null)
				PriceEntered = pp.getPriceStd();
			//
			log.fine("amt - QtyChanged -> PriceActual=" + pp.getPriceStd() 
				+ ", PriceEntered=" + PriceEntered + ", Discount=" + pp.getDiscount());
			PriceActual = pp.getPriceStd();
			mTab.setValue("PriceActual", PriceActual);
		//	mTab.setValue("Discount", pp.getDiscount());
			mTab.setValue("PriceEntered", PriceEntered);
			ctx.setContext(WindowNo, "DiscountSchema", pp.isDiscountSchema() ? "Y" : "N");
		}
		else if (mField.getColumnName().equals("PriceActual"))
		{
			PriceActual = (BigDecimal)value;
			PriceEntered = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, PriceActual);
			if (PriceEntered == null)
				PriceEntered = PriceActual;
			//
			log.fine("amt - PriceActual=" + PriceActual 
				+ " -> PriceEntered=" + PriceEntered);
			mTab.setValue("PriceEntered", PriceEntered);
		}
		else if (mField.getColumnName().equals("PriceEntered"))
		{
			PriceEntered = (BigDecimal)value;
			PriceActual = MUOMConversion.convertProductTo (ctx, M_Product_ID, 
				C_UOM_To_ID, PriceEntered);
			if (PriceActual == null)
				PriceActual = PriceEntered;
			//
			log.fine("amt - PriceEntered=" + PriceEntered 
				+ " -> PriceActual=" + PriceActual);
			mTab.setValue("PriceActual", PriceActual);
		}
		
		/**  Discount entered - Calculate Actual/Entered
		if (mField.getColumnName().equals("Discount"))
		{
			PriceActual = new BigDecimal ((100.0 - Discount.doubleValue()) / 100.0 * PriceList.doubleValue());
			if (PriceActual.scale() > StdPrecision)
				PriceActual = PriceActual.setScale(StdPrecision, BigDecimal.ROUND_HALF_UP);
			PriceEntered = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, PriceActual);
			if (PriceEntered == null)
				PriceEntered = PriceActual;
			mTab.setValue("PriceActual", PriceActual);
			mTab.setValue("PriceEntered", PriceEntered);
		}
		//	calculate Discount
		else
		{
			if (PriceList.intValue() == 0)
				Discount = Env.ZERO;
			else
				Discount = new BigDecimal ((PriceList.doubleValue() - PriceActual.doubleValue()) / PriceList.doubleValue() * 100.0);
			if (Discount.scale() > 2)
				Discount = Discount.setScale(2, BigDecimal.ROUND_HALF_UP);
			mTab.setValue("Discount", Discount);
		}
		log.fine("amt = PriceEntered=" + PriceEntered + ", Actual" + PriceActual + ", Discount=" + Discount);
		/* */

		//	Check PriceLimit
		String epl = ctx.getContext(WindowNo, "EnforcePriceLimit");
		boolean enforce = ctx.isSOTrx(WindowNo) && epl != null && epl.equals("Y");
		if (enforce && MRole.getDefault().isOverwritePriceLimit())
			enforce = false;
		//	Check Price Limit?
		if (enforce && PriceLimit.doubleValue() != 0.0
		  && PriceActual.compareTo(PriceLimit) < 0)
		{
			PriceActual = PriceLimit;
			PriceEntered = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, PriceLimit);
			if (PriceEntered == null)
				PriceEntered = PriceLimit;
			log.fine("amt =(under) PriceEntered=" + PriceEntered + ", Actual" + PriceLimit);
			mTab.setValue ("PriceActual", PriceLimit);
			mTab.setValue ("PriceEntered", PriceEntered);
			mTab.fireDataStatusEEvent ("UnderLimitPrice", "", false);
			//	Repeat Discount calc
			if (PriceList.intValue() != 0)
			{
				Discount = new BigDecimal ((PriceList.doubleValue () - PriceActual.doubleValue ()) / PriceList.doubleValue () * 100.0);
				if (Discount.scale () > 2)
					Discount = Discount.setScale (2, BigDecimal.ROUND_HALF_UP);
			//	mTab.setValue ("Discount", Discount);
			}
		}

		//	Line Net Amt
		BigDecimal LineNetAmt = QtyBilling.multiply(PriceActual);
		if (LineNetAmt.scale() > StdPrecision)
			LineNetAmt = LineNetAmt.setScale(StdPrecision, BigDecimal.ROUND_HALF_UP);
		log.info("amt = LineNetAmt=" + LineNetAmt);
		mTab.setValue("LineNetAmt", LineNetAmt);

		//	Calculate Tax Amount for PO
		boolean IsSOTrx = "Y".equals(ctx.getContext(WindowNo, "IsSOTrx"));
		if (!IsSOTrx)
		{
			BigDecimal TaxAmt = Env.ZERO;
			if (mField.getColumnName().equals("TaxAmt"))
			{
				TaxAmt = (BigDecimal)mTab.getValue("TaxAmt");
			}
			else
			{
				Integer taxID = (Integer)mTab.getValue("C_Tax_ID");
				// Jirimuto modified for Tax calculation Adjustment for Planex Spec. 2009/03/16
				Integer partnerID = (Integer)mTab.getValue("C_BPartner_ID");
				if (taxID != null)
				{
					int C_Tax_ID = taxID.intValue();
					MTax tax = new MTax (ctx, C_Tax_ID, null);
					// Jirimuto modified for Tax calculation Adjustment for Planex Spec. 2009/03/16
					int roundingmode = BigDecimal.ROUND_HALF_UP;
					if( partnerID.intValue()!=0 ){
						MBPartner partner = MBPartner.get(ctx, partnerID);
						roundingmode = partner.getTaxRoundModeAsInt();
					}
					TaxAmt = tax.calculateTax(LineNetAmt, isTaxIncluded(ctx, WindowNo), StdPrecision, roundingmode);
					mTab.setValue("TaxAmt", TaxAmt);
				}
			}
			//	Add it up
			mTab.setValue("LineTotalAmt", LineNetAmt.add(TaxAmt));
		}

		setCalloutActive(false);
		return "";
	}	//	amt

	/**
	 * 	Is Tax Included
	 *	@param WindowNo window no
	 *	@return tax included (default: false)
	 */
	private boolean isTaxIncluded (Ctx ctx, int WindowNo)
	{
		String ss = ctx.getContext(WindowNo, "IsTaxIncluded");
		//	Not Set Yet
		if (ss.length() == 0)
		{
			int M_PriceList_ID = ctx.getContextAsInt(WindowNo, "M_PriceList_ID");
			if (M_PriceList_ID == 0)
				return false;
			ss = DB.getSQLValueString(null,
				"SELECT IsTaxIncluded FROM M_PriceList WHERE M_PriceList_ID=?", 
				M_PriceList_ID);
			if (ss == null)
				ss = "N";
			ctx.setContext(WindowNo, "IsTaxIncluded", ss);
		}
		return "Y".equals(ss);
	}	//	isTaxIncluded

	/**
	 *	Billing Line - Quantity.
	 *		- called from C_UOM_ID, QtyEntered, QtyBilling
	 *		- enforces qty UOM relationship
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String qty (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		if (isCalloutActive() || value == null)
			return "";
		setCalloutActive(true);

		int M_Product_ID = ctx.getContextAsInt(WindowNo, "M_Product_ID");
	//	log.log(Level.WARNING,"qty - init - M_Product_ID=" + M_Product_ID);
		BigDecimal QtyBilling, QtyEntered, PriceActual, PriceEntered;
		
		//	No Product
		if (M_Product_ID == 0)
		{
			QtyEntered = (BigDecimal)mTab.getValue("QtyEntered");
			mTab.setValue("QtyBilling", QtyEntered);
		}
		//	UOM Changed - convert from Entered -> Product
		else if (mField.getColumnName().equals("C_UOM_ID"))
		{
			int C_UOM_To_ID = ((Integer)value).intValue();
			QtyEntered = (BigDecimal)mTab.getValue("QtyEntered");
			BigDecimal QtyEntered1 = QtyEntered.setScale(MUOM.getPrecision(ctx, C_UOM_To_ID), BigDecimal.ROUND_HALF_UP);
			if (QtyEntered.compareTo(QtyEntered1) != 0)
			{
				log.fine("Corrected QtyEntered Scale UOM=" + C_UOM_To_ID 
					+ "; QtyEntered=" + QtyEntered + "->" + QtyEntered1);  
				QtyEntered = QtyEntered1;
				mTab.setValue("QtyEntered", QtyEntered);
			}
			QtyBilling = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, QtyEntered);
			if (QtyBilling == null)
				QtyBilling = QtyEntered;
			boolean conversion = QtyEntered.compareTo(QtyBilling) != 0;
			PriceActual = (BigDecimal)mTab.getValue("PriceActual");
			PriceEntered = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, PriceActual);
			if (PriceEntered == null)
				PriceEntered = PriceActual; 
			log.fine("qty - UOM=" + C_UOM_To_ID 
				+ ", QtyEntered/PriceActual=" + QtyEntered + "/" + PriceActual
				+ " -> " + conversion 
				+ " QtyBilling/PriceEntered=" + QtyBilling + "/" + PriceEntered);
			ctx.setContext(WindowNo, "UOMConversion", conversion ? "Y" : "N");
			mTab.setValue("QtyBilling", QtyBilling);
			mTab.setValue("PriceEntered", PriceEntered);
		}
		//	QtyEntered changed - calculate QtyBilling
		else if (mField.getColumnName().equals("QtyEntered"))
		{
			int C_UOM_To_ID = ctx.getContextAsInt(WindowNo, "C_UOM_ID");
			QtyEntered = (BigDecimal)value;
			BigDecimal QtyEntered1 = QtyEntered.setScale(MUOM.getPrecision(ctx, C_UOM_To_ID), BigDecimal.ROUND_HALF_UP);
			if (QtyEntered.compareTo(QtyEntered1) != 0)
			{
				log.fine("Corrected QtyEntered Scale UOM=" + C_UOM_To_ID 
					+ "; QtyEntered=" + QtyEntered + "->" + QtyEntered1);  
				QtyEntered = QtyEntered1;
				mTab.setValue("QtyEntered", QtyEntered);
			}
			QtyBilling = MUOMConversion.convertProductFrom (ctx, M_Product_ID, 
				C_UOM_To_ID, QtyEntered);
			if (QtyBilling == null)
				QtyBilling = QtyEntered;
			boolean conversion = QtyEntered.compareTo(QtyBilling) != 0;
			log.fine("qty - UOM=" + C_UOM_To_ID 
				+ ", QtyEntered=" + QtyEntered
				+ " -> " + conversion 
				+ " QtyBilling=" + QtyBilling);
			ctx.setContext(WindowNo, "UOMConversion", conversion ? "Y" : "N");
			mTab.setValue("QtyBilling", QtyBilling);
		}
		//	QtyBilling changed - calculate QtyEntered (should not happen)
		else if (mField.getColumnName().equals("QtyBilling"))
		{
			int C_UOM_To_ID = ctx.getContextAsInt(WindowNo, "C_UOM_ID");
			QtyBilling = (BigDecimal)value;
			int precision = MProduct.get(ctx, M_Product_ID).getUOMPrecision(); 
			BigDecimal QtyBilling1 = QtyBilling.setScale(precision, BigDecimal.ROUND_HALF_UP);
			if (QtyBilling.compareTo(QtyBilling1) != 0)
			{
				log.fine("Corrected QtyBilling Scale "
					+ QtyBilling + "->" + QtyBilling1);  
				QtyBilling = QtyBilling1;
				mTab.setValue("QtyBilling", QtyBilling);
			}
			QtyEntered = MUOMConversion.convertProductTo (ctx, M_Product_ID, 
				C_UOM_To_ID, QtyBilling);
			if (QtyEntered == null)
				QtyEntered = QtyBilling;
			boolean conversion = QtyBilling.compareTo(QtyEntered) != 0;
			log.fine("qty - UOM=" + C_UOM_To_ID 
				+ ", QtyBilling=" + QtyBilling
				+ " -> " + conversion 
				+ " QtyEntered=" + QtyEntered);
			ctx.setContext(WindowNo, "UOMConversion", conversion ? "Y" : "N");
			mTab.setValue("QtyEntered", QtyEntered);
		}
		//
		setCalloutActive(false);
		return "";
	}	//	qty

	/**
	 *	@param ctx context
	 *	@param WindowNo window no
	 *	@param mTab tab
	 *	@param mField field
	 *	@param value value
	 *	@return null or error message
	 */
	public String invoiceline (Ctx ctx, int WindowNo, GridTab mTab, GridField mField, Object value)
	{
		Integer C_InvoiceLine_ID = (Integer)value;
		if (C_InvoiceLine_ID == null || C_InvoiceLine_ID.intValue() == 0)
			return "";

		if (isCalloutActive() || value == null)
			return "";
		setCalloutActive(true);
		
		MInvoiceLine invoiceline = new MInvoiceLine(ctx, C_InvoiceLine_ID, null);
		mTab.setValue("Description", invoiceline.getDescription());
		
		if( invoiceline.getC_Activity_ID() != 0 )
			mTab.setValue("C_Activity_ID", new Integer(invoiceline.getC_Activity_ID()));
		if( invoiceline.getC_Campaign_ID() != 0 )
			mTab.setValue("C_Campaign_ID", new Integer(invoiceline.getC_Campaign_ID()));
		if( invoiceline.getC_Project_ID() != 0 )
			mTab.setValue("C_Project_ID", new Integer(invoiceline.getC_Project_ID()));
		if( invoiceline.getC_ProjectPhase_ID() != 0 )
			mTab.setValue("C_ProjectPhase_ID", new Integer(invoiceline.getC_ProjectPhase_ID()));
		if( invoiceline.getC_ProjectTask_ID() != 0 )
			mTab.setValue("C_ProjectTask_ID", new Integer(invoiceline.getC_ProjectTask_ID()));
		if( invoiceline.getC_OrderLine_ID() != 0 )
			mTab.setValue("C_OrderLine_ID", new Integer(invoiceline.getC_OrderLine_ID()));
		if( invoiceline.getC_Charge_ID() != 0 )
			mTab.setValue("C_Charge_ID", new Integer(invoiceline.getC_Charge_ID()));
		if( invoiceline.getUser1_ID() != 0 )
			mTab.setValue("User1_ID", new Integer(invoiceline.getUser1_ID()));
		if( invoiceline.getUser2_ID() != 0 )
			mTab.setValue("User2_ID", new Integer(invoiceline.getUser2_ID()));
		if(invoiceline.getM_Product_ID() != 0) 
			mTab.setValue("M_Product_ID", new Integer(invoiceline.getM_Product_ID()));
		if(invoiceline.getC_Tax_ID() != 0) 
			mTab.setValue("C_Tax_ID", new Integer(invoiceline.getC_Tax_ID()));
		if(invoiceline.getC_Tax_ID() != 0) 
				mTab.setValue("C_UOM_ID", new Integer(invoiceline.getC_UOM_ID()));		
		
		mTab.setValue("QtyEntered", invoiceline.getQtyEntered());
		mTab.setValue("QtyBilling", invoiceline.getQtyInvoiced());
		mTab.setValue("PriceEntered", invoiceline.getPriceEntered());
		mTab.setValue("PriceActual", invoiceline.getPriceActual());
		mTab.setValue("PriceLimit", invoiceline.getPriceLimit());
		mTab.setValue("PriceList", invoiceline.getPriceList());
		mTab.setValue("LineNetAmt", invoiceline.getLineNetAmt());
		mTab.setValue("LineTotalAmt", invoiceline.getLineTotalAmt());
		mTab.setValue("PriceList", invoiceline.getPriceList());
		mTab.setValue("RRAmt", invoiceline.getRRAmt());
		mTab.setValue("RRStartDate", invoiceline.getRRStartDate());
		
		if( invoiceline.getC_Charge_ID() != 0 )
			mTab.setValue("C_Charge_ID", new Integer(invoiceline.getC_Charge_ID()));
		
		setCalloutActive(false);
		
		return "";
	}	//	invocieline
	
}	//	CalloutBilling
