/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.model;

import java.io.*;
import java.math.*;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.logging.*;

import org.compiere.api.ModelValidator;
import org.compiere.framework.ModelValidationEngine;
import org.compiere.print.*;
import org.compiere.process.*;
import org.compiere.util.*;

/**
 *	Billing Model.
 * 	Please do not set DocStatus and C_DocType_ID directly. 
 * 	They are set in the process() method. 
 * 	Use DocAction and C_DocTypeTarget_ID instead.
 *
 *  @author Jirimuto
 *  @version $Id: MBilling.java,v 1.1 2010/04/09 09:14:45 jrmt Exp $
 */
public class MBilling extends X_C_Billing implements DocAction
{
	/**
	 * 	Get Payments Of BPartner
	 *	@param ctx context
	 *	@param C_BPartner_ID id
	 *	@param trxName transaction
	 *	@return array
	 */
	public static MBilling[] getOfBPartner (Ctx ctx, int C_BPartner_ID, String trxName)
	{
		ArrayList<MBilling> list = new ArrayList<MBilling>();
		String sql = "SELECT * FROM C_Billing WHERE C_BPartner_ID=?";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, trxName);
			pstmt.setInt(1, C_BPartner_ID);
			ResultSet rs = pstmt.executeQuery();
			while (rs.next())
				list.add(new MBilling(ctx,rs, trxName));
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}

		//
		MBilling[] retValue = new MBilling[list.size()];
		list.toArray(retValue);
		return retValue;
	}	//	getOfBPartner
	
	/**
	 * 	Get PDF File Name
	 *	@param documentDir directory
	 * 	@param C_Billing_ID invoice
	 *	@return file name
	 */
	public static String getPDFFileName (String documentDir, int C_Billing_ID)
	{
		StringBuffer sb = new StringBuffer (documentDir);
		if (sb.length() == 0)
			sb.append(".");
		if (!sb.toString().endsWith(File.separator))
			sb.append(File.separator);
		sb.append("C_Billing_ID_")
			.append(C_Billing_ID)
			.append(".pdf");
		return sb.toString();
	}	//	getPDFFileName
	
	
	/**
	 * 	Get MBilling from Cache
	 *	@param ctx context
	 *	@param C_Billing_ID id
	 *	@return MBilling
	 */
	public static MBilling get (Ctx ctx, int C_Billing_ID)
	{
		Integer key = new Integer (C_Billing_ID);
		MBilling retValue = (MBilling) s_cache.get (key);
		if (retValue != null)
			return retValue;
		retValue = new MBilling (ctx, C_Billing_ID, null);
		if (retValue.get_ID () != 0)
			s_cache.put (key, retValue);
		return retValue;
	} //	get

	/**	Cache						*/
	private static CCache<Integer,MBilling>	s_cache	= new CCache<Integer,MBilling>("C_Billing", 20, 2);	//	2 minutes
	
	
	/**************************************************************************
	 * 	Billing Constructor
	 * 	@param ctx context
	 * 	@param C_Billing_ID billing or 0 for new
	 * 	@param trxName trx name
	 */
	public MBilling (Ctx ctx, int C_Billing_ID, String trxName)
	{
		super (ctx, C_Billing_ID, trxName);
		if (C_Billing_ID == 0)
		{
			setDocStatus (DOCSTATUS_Drafted);		//	Draft
			setDocAction (DOCACTION_Complete);
			//
			setPaymentRule(PAYMENTRULE_OnCredit);	//	Payment Terms

			setDateBilling (new Timestamp (System.currentTimeMillis ()));
			setDateInvoiced (new Timestamp (System.currentTimeMillis ()));
			setDateAcct (new Timestamp (System.currentTimeMillis ()));
			//
			setChargeAmt (Env.ZERO);
			setTotalLines (Env.ZERO);
			setGrandTotal (Env.ZERO);
			//
			setIsSOTrx (true);
			setIsTaxIncluded (false);
			setIsApproved (false);
			setIsDiscountPrinted (false);
			setIsPaid (false);
			setSendEMail (false);
			setIsPrinted (false);
			setIsTransferred (false);
			setIsSelfService(false);
			setIsPayScheduleValid(false);
			setIsInDispute(false);
			setPosted(false);
			super.setProcessed (false);
			setProcessing(false);
		}
	}	//	MBilling

	/**
	 *  Load Constructor
	 *  @param ctx context
	 *  @param rs result set record
	 *	@param trxName transaction
	 */
	public MBilling (Ctx ctx, ResultSet rs, String trxName)
	{
		super(ctx, rs, trxName);
	}	//	MBilling

	/**
	 * 	Create Billing from Invoice
	 *	@param invoice invoice
	 *	@param C_DocTypeTarget_ID target document type
	 *	@param billingDate date or null
	 */
	public MBilling (MBPartner partner, MInvoice invoice, int C_DocTypeTarget_ID, Timestamp billingDate)
	{
		this (invoice.getCtx(), 0, invoice.get_TrxName());
		setClientOrg(invoice);
		setInvoice(invoice);	//	set base settings
		//
		if (C_DocTypeTarget_ID == 0)
			C_DocTypeTarget_ID =  DB.getSQLValue(null,
				"SELECT MIN(C_DOCTYPE_ID) FROM C_DOCTYPE WHERE ISACTIVE='Y' AND DOCBASETYPE IN ('BRI', 'BPI') AND ISSOTRX='"
					+(invoice.isSOTrx()?'Y':'N')+"'" );
		setC_DocTypeTarget_ID(C_DocTypeTarget_ID);
		setIsSOTrx(invoice.isSOTrx());
		if (billingDate != null)
			setDateBilling(billingDate);
		setDateAcct(getDateInvoiced());
		//
		setSalesRep_ID(invoice.getSalesRep_ID());
		
		int C_BPartner_ID = invoice.getC_BPartner_ID();
		int C_Invoice_Partner_ID = C_BPartner_ID;
		
		MBPartner invoicepartner = MBPartner.get(getCtx(), C_Invoice_Partner_ID);
		invoicepartner.getPrimaryC_BPartner_Location_ID();
		//
		setC_BPartner_ID(C_Invoice_Partner_ID);
		setC_BPartner_Location_ID(invoicepartner.getPrimaryC_BPartner_Location_ID());
		setAD_User_ID(invoicepartner.getPrimaryAD_User_ID());
		
		if( invoice.isSOTrx() ){
			MBilling lastBilling = MBilling.getLastBilling(getCtx(), C_Invoice_Partner_ID, billingDate, invoice.get_TrxName());
			BigDecimal lastInvoicedAmt = BigDecimal.ZERO;
			Timestamp lastDateBilling = this.getDateBilling();
			if( lastBilling != null ){
				lastInvoicedAmt = lastBilling.getTotalInvoiceAmt();
				this.setLastInvoicedAmt(lastBilling.getTotalInvoiceAmt());
			} else {
				Calendar lastBillingCal = TimeUtil.getToday();
				if( lastBillingCal!=null ){
					lastBillingCal.setTimeInMillis(this.getDateBilling().getTime());
					lastBillingCal.set(Calendar.MILLISECOND, 0);
					lastBillingCal.set(Calendar.SECOND, 0);
					lastBillingCal.set(Calendar.MINUTE, 0);
					lastBillingCal.set(Calendar.HOUR_OF_DAY, 0);
					lastBillingCal.add(Calendar.DAY_OF_MONTH, -1);
					lastDateBilling = new Timestamp (lastBillingCal.getTimeInMillis());
				}
			}
			BigDecimal lastPaid = MPayment.getSumOfBPartner(getCtx(), C_BPartner_ID, lastDateBilling, billingDate, invoice.get_TrxName());
			this.setLastPaidAmt(lastPaid);
			this.setAdjustAmt(BigDecimal.ZERO);
			BigDecimal lastTansferedAmt = lastInvoicedAmt.subtract(lastPaid);
			this.setLastTransferedAmt(lastTansferedAmt);
		}
		
	}	//	MBilling

	/**
	 * 	Get Payments Of BPartner
	 *	@param ctx context
	 *	@param C_BPartner_ID id
	 *	@param trxName transaction
	 *	@return array
	 */
	public static MBilling getLastBilling (Ctx ctx, int C_BPartner_ID, Timestamp billingDate, String trxName)
	{
		MBilling retValue = null;
		
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
		String billingDateStr = format.format(billingDate);
		
		String sql = "SELECT * FROM C_Billing WHERE C_BPartner_ID=? AND TO_CHAR(DateBilling, 'YYYYMMDD')<'"+billingDateStr+"'"
			+" AND IsSOTrx='Y' AND DOCSTATUS IN ('CO', 'CL', 'DR') ORDER BY DateBilling DESC";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, trxName);
			pstmt.setInt(1, C_BPartner_ID);
			ResultSet rs = pstmt.executeQuery();
			if (rs.next())
				retValue = new MBilling(ctx,rs, trxName);
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}

		return retValue;
	}	//	getOfBPartner
	
	
	/**	Billing Lines			*/
	private MBillingLine[]	m_lines;
	/**	Billing Taxes			*/
	private MBillingTax[]	m_taxes;
	/**	Logger			*/
	private static CLogger s_log = CLogger.getCLogger(MBilling.class);

	/**
	 * 	Overwrite Client/Org if required
	 * 	@param AD_Client_ID client
	 * 	@param AD_Org_ID org
	 */
	public void setClientOrg (int AD_Client_ID, int AD_Org_ID)
	{
		super.setClientOrg(AD_Client_ID, AD_Org_ID);
	}	//	setClientOrg

	/**
	 * 	Set Business Partner Defaults & Details
	 * 	@param bp business partner
	 */
	public void setBPartner (MBPartner bp)
	{
		if (bp == null)
			return;

		setC_BPartner_ID(bp.getC_BPartner_ID());
		//	Set Defaults
		int ii = 0;
		if (isSOTrx())
			ii = bp.getC_PaymentTerm_ID();
		else
			ii = bp.getPO_PaymentTerm_ID();
		if (ii != 0)
			setC_PaymentTerm_ID(ii);
		//
		if (isSOTrx())
			ii = bp.getM_PriceList_ID();
		else
			ii = bp.getPO_PriceList_ID();
		if (ii != 0)
			setM_PriceList_ID(ii);
		//
		String ss = bp.getPaymentRule();
		if (ss != null)
			setPaymentRule(ss);

		//	Set Locations
		MBPartnerLocation[] locs = bp.getLocations(false);
		if (locs != null)
		{
			for (int i = 0; i < locs.length; i++)
			{
				if ((locs[i].isBillTo() && isSOTrx()) 
				|| (locs[i].isPayFrom() && !isSOTrx()))
					setC_BPartner_Location_ID(locs[i].getC_BPartner_Location_ID());
			}
			//	set to first
			if (getC_BPartner_Location_ID() == 0 && locs.length > 0)
				setC_BPartner_Location_ID(locs[0].getC_BPartner_Location_ID());
		}
		if (getC_BPartner_Location_ID() == 0)
			log.log(Level.SEVERE, "Has no To Address: " + bp);

		//	Set Contact
		MUser[] contacts = bp.getContacts(false);
		if (contacts != null && contacts.length > 0)	//	get first User
			setAD_User_ID(contacts[0].getAD_User_ID());
	}	//	setBPartner

	/**
	 * 	Set Order References
	 * 	@param invoice order
	 */
	public void setInvoice (MInvoice invoice)
	{
		if (invoice == null)
			return;

		setC_Invoice_ID(invoice.getC_Invoice_ID());		// Bao modified for fixing bug. 2009/05/29
		setC_Order_ID(invoice.getC_Order_ID());
		setIsSOTrx(invoice.isSOTrx());
		setIsDiscountPrinted(invoice.isDiscountPrinted());
		setIsSelfService(invoice.isSelfService());
		setSendEMail(invoice.isSendEMail());
		//
		setM_PriceList_ID(invoice.getM_PriceList_ID());
		setIsTaxIncluded(invoice.isTaxIncluded());
		setC_Currency_ID(invoice.getC_Currency_ID());
		setC_ConversionType_ID(invoice.getC_ConversionType_ID());
		//
		setPaymentRule(invoice.getPaymentRule());
		setC_PaymentTerm_ID(invoice.getC_PaymentTerm_ID());
		setPOReference(invoice.getPOReference());
		setDescription(invoice.getDescription());
		setDateOrdered(invoice.getDateOrdered());
		setDateInvoiced(invoice.getDateInvoiced());
		setDateAcct(invoice.getDateAcct());
		//
		setAD_OrgTrx_ID(invoice.getAD_OrgTrx_ID());
		setC_Project_ID(invoice.getC_Project_ID());
		setC_Campaign_ID(invoice.getC_Campaign_ID());
		setC_Activity_ID(invoice.getC_Activity_ID());
		setUser1_ID(invoice.getUser1_ID());
		setUser2_ID(invoice.getUser2_ID());
	}	//	setOrder

	/**
	 * 	Set Target Document Type
	 * 	@param DocBaseType doc type MDocType.DOCBASETYPE_
	 */
	public void setC_DocTypeTarget_ID (String DocBaseType)
	{
		String sql = "SELECT C_DocType_ID FROM C_DocType "
			+ "WHERE AD_Client_ID=? AND DocBaseType=?"
			+ " AND IsActive='Y' "
			+ "ORDER BY IsDefault DESC";
		int C_DocType_ID = DB.getSQLValue(null, sql, getAD_Client_ID(), DocBaseType);
		if (C_DocType_ID <= 0)
			log.log(Level.SEVERE, "Not found for AC_Client_ID=" 
				+ getAD_Client_ID() + " - " + DocBaseType);
		else
		{
			log.fine(DocBaseType);
			setC_DocTypeTarget_ID (C_DocType_ID);
			boolean isSOTrx = MDocBaseType.DOCBASETYPE_BillingInvoiceReceivable.equals(DocBaseType);
			setIsSOTrx (isSOTrx);
		}
		
	}	//	setC_DocTypeTarget_ID

	/**
	 * 	Set Target Document Type.
	 * 	Based on SO flag AP/AP Billing
	 */
	public void setC_DocTypeTarget_ID ()
	{
		if (getC_DocTypeTarget_ID() > 0)
			return;
		if (isSOTrx())
			setC_DocTypeTarget_ID(MDocBaseType.DOCBASETYPE_BillingInvoiceReceivable);
		else
			setC_DocTypeTarget_ID(MDocBaseType.DOCBASETYPE_BillingInvoicePayable);
	}	//	setC_DocTypeTarget_ID

		
	/**
	 * 	Get Billing Lines of Invoice
	 * 	@param whereClause starting with AND
	 * 	@return lines
	 */
	private MBillingLine[] getLines (String whereClause)
	{
		ArrayList<MBillingLine> list = new ArrayList<MBillingLine>();
		String sql = "SELECT * FROM C_BillingLine WHERE C_Billing_ID=? ";
		if (whereClause != null)
			sql += whereClause;
		sql += " ORDER BY Line";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, get_TrxName());
			pstmt.setInt(1, getC_Billing_ID());
			ResultSet rs = pstmt.executeQuery();
			while (rs.next())
			{
				MBillingLine il = new MBillingLine(getCtx(), rs, get_TrxName());
				il.setBilling(this);
				list.add(il);
			}
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "getLines", e);
		}
		finally
		{
			try
			{
				if (pstmt != null)
					pstmt.close ();
			}
			catch (Exception e)
			{}
			pstmt = null;
		}

		//
		MBillingLine[] lines = new MBillingLine[list.size()];
		list.toArray(lines);
		return lines;
	}	//	getLines

	/**
	 * 	Get Billing Lines
	 * 	@param requery
	 * 	@return lines
	 */
	public MBillingLine[] getLines (boolean requery)
	{
		if (m_lines == null || m_lines.length == 0 || requery)
			m_lines = getLines(null);
		return m_lines;
	}	//	getLines

	/**
	 * 	Get Lines of Invoice
	 * 	@return lines
	 */
	public MBillingLine[] getLines()
	{
		return getLines(false);
	}	//	getLines
	
	
	/**
	 * 	Renumber Lines
	 *	@param step start and step
	 */
	public void renumberLines (int step)
	{
		int number = step;
		MBillingLine[] lines = getLines(false);
		for (int i = 0; i < lines.length; i++)
		{
			MBillingLine line = lines[i];
			line.setLine(number);
			line.save();
			number += step;
		}
		m_lines = null;
	}	//	renumberLines
	
	/** Reversal Flag		*/
	private boolean m_reversal = false;
	
	/**
	 * 	Set Reversal
	 *	@param reversal reversal
	 */
	private void setReversal(boolean reversal)
	{
		m_reversal = reversal;
	}	//	setReversal
	/**
	 * 	Is Reversal
	 *	@return reversal
	 */
	private boolean isReversal()
	{
		return m_reversal;
	}	//	isReversal
	
	/**
	 * 	Get Taxes
	 *	@param requery requery
	 *	@return array of taxes
	 */
	public MBillingTax[] getTaxes (boolean requery)
	{
		if (m_taxes != null && !requery)
			return m_taxes;
		String sql = "SELECT * FROM C_BillingTax WHERE C_Billing_ID=?";
		ArrayList<MBillingTax> list = new ArrayList<MBillingTax>();
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, get_TrxName());
			pstmt.setInt (1, getC_Billing_ID());
			ResultSet rs = pstmt.executeQuery ();
			while (rs.next ())
				list.add(new MBillingTax(getCtx(), rs, get_TrxName()));
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "getTaxes", e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		
		m_taxes = new MBillingTax[list.size ()];
		list.toArray (m_taxes);
		return m_taxes;
	}	//	getTaxes
	
	/**
	 * 	Add to Description
	 *	@param description text
	 */
	public void addDescription (String description)
	{
		String desc = getDescription();
		if (desc == null)
			setDescription(description);
		else
			setDescription(desc + " | " + description);
	}	//	addDescription

	/**
	 * 	Set Processed.
	 * 	Propergate to Lines/Taxes
	 *	@param processed processed
	 */
	public void setProcessed (boolean processed)
	{
		super.setProcessed (processed);
		if (get_ID() == 0)
			return;
		String set = "SET Processed='"
			+ (processed ? "Y" : "N")
			+ "' WHERE C_Billing_ID=" + getC_Billing_ID();
		int noLine = DB.executeUpdate("UPDATE C_BillingLine " + set, get_TrxName());
		int noTax = DB.executeUpdate("UPDATE C_BillingTax " + set, get_TrxName());
		m_lines = null;
		m_taxes = null;
		log.fine(processed + " - Lines=" + noLine + ", Tax=" + noTax);
	}	//	setProcessed

	
	/**************************************************************************
	 * 	Before Save
	 *	@param newRecord new
	 *	@return true
	 */
	protected boolean beforeSave (boolean newRecord)
	{
		log.fine("");
		//	No Partner Info - set Template
		if (getC_BPartner_ID() == 0)
			setBPartner(MBPartner.getTemplate(getCtx(), getAD_Client_ID()));
		if (getC_BPartner_Location_ID() == 0)
			setBPartner(new MBPartner(getCtx(), getC_BPartner_ID(), null));

		//	Price List
		if (getM_PriceList_ID() == 0)
		{
			int ii = getCtx().getContextAsInt("#M_PriceList_ID");
			if (ii != 0)
				setM_PriceList_ID(ii);
			else
			{
				String sql = "SELECT M_PriceList_ID FROM M_PriceList WHERE AD_Client_ID=? AND IsDefault='Y'";
				ii = DB.getSQLValue (null, sql, getAD_Client_ID());
				if (ii != 0)
					setM_PriceList_ID (ii);
			}
		}

		//	Currency
		if (getC_Currency_ID() == 0)
		{
			String sql = "SELECT C_Currency_ID FROM M_PriceList WHERE M_PriceList_ID=?";
			int ii = DB.getSQLValue (null, sql, getM_PriceList_ID());
			if (ii != 0)
				setC_Currency_ID (ii);
			else
				setC_Currency_ID(getCtx().getContextAsInt("#C_Currency_ID"));
		}

		//	Sales Rep
		if (getSalesRep_ID() == 0)
		{
			int ii = getCtx().getContextAsInt("#SalesRep_ID");
			if (ii != 0)
				setSalesRep_ID (ii);
		}

		//	Document Type
		if (getC_DocType_ID() == 0)
			setC_DocType_ID (0);	//	make sure it's set to 0
		if (getC_DocTypeTarget_ID() == 0)
			setC_DocTypeTarget_ID(isSOTrx() ? MDocBaseType.DOCBASETYPE_BillingInvoiceReceivable : MDocBaseType.DOCBASETYPE_BillingInvoicePayable);

		//	Payment Term
		if (getC_PaymentTerm_ID() == 0)
		{
			int ii = getCtx().getContextAsInt("#C_PaymentTerm_ID");
			if (ii != 0)
				setC_PaymentTerm_ID (ii);
			else
			{
				String sql = "SELECT C_PaymentTerm_ID FROM C_PaymentTerm WHERE AD_Client_ID=? AND IsDefault='Y'";
				ii = DB.getSQLValue(null, sql, getAD_Client_ID());
				if (ii != 0)
					setC_PaymentTerm_ID (ii);
			}
		}
		BigDecimal invoiceAmt = getLastTransferedAmt();
		invoiceAmt = invoiceAmt.add(getGrandTotal());
		setTotalInvoiceAmt(invoiceAmt);
		
		return true;
	}	//	beforeSave

	/**
	 * 	Before Delete
	 *	@return true if it can be deleted
	 */
	protected boolean beforeDelete ()
	{
		return true;
	}	//	beforeDelete

	/**
	 * 	String Representation
	 *	@return info
	 */
	public String toString ()
	{
		StringBuffer sb = new StringBuffer ("MBilling[")
			.append(get_ID()).append("-").append(getDocumentNo())
			.append(",GrandTotal=").append(getGrandTotal());
		if (m_lines != null)
			sb.append(" (#").append(m_lines.length).append(")");
		sb.append ("]");
		return sb.toString ();
	}	//	toString

	/**
	 * 	Get Document Info
	 *	@return document info (untranslated)
	 */
	public String getDocumentInfo()
	{
		MDocType dt = MDocType.get(getCtx(), getC_DocType_ID());
		return dt.getName() + " " + getDocumentNo();
	}	//	getDocumentInfo
	
	
	/**
	 * 	After Save
	 *	@param newRecord new
	 *	@param success success
	 *	@return success
	 */
	protected boolean afterSave (boolean newRecord, boolean success)
	{
		if (!success || newRecord)
			return success;
		
		if (is_ValueChanged("AD_Org_ID"))
		{
			String sql = "UPDATE C_BillingLine ol"
				+ " SET AD_Org_ID ="
					+ "(SELECT AD_Org_ID"
					+ " FROM C_Billing o WHERE ol.C_Billing_ID=o.C_Billing_ID) "
				+ "WHERE C_Billing_ID=" + getC_Billing_ID();
			int no = DB.executeUpdate(sql, get_TrxName());
			log.fine("Lines -> #" + no);
		}		
		return true;
	}	//	afterSave
	
	
	/**
	 * 	Set Price List (and Currency) when valid
	 * 	@param M_PriceList_ID price list
	 */
	public void setM_PriceList_ID (int M_PriceList_ID)
	{
		String sql = "SELECT M_PriceList_ID, C_Currency_ID "
			+ "FROM M_PriceList WHERE M_PriceList_ID=?";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, null);
			pstmt.setInt(1, M_PriceList_ID);
			ResultSet rs = pstmt.executeQuery();
			if (rs.next())
			{
				super.setM_PriceList_ID (rs.getInt(1));
				setC_Currency_ID (rs.getInt(2));
			}
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "setM_PriceList_ID", e);
		}
		finally
		{
			try
			{
				if (pstmt != null)
					pstmt.close ();
			}
			catch (Exception e)
			{}
			pstmt = null;
		}
	}	//	setM_PriceList_ID

	/**
	 * 	Set Paid Flag for invoices
	 * 	@param ctx context
	 *	@param C_BPartner_ID if 0 all
	 *	@param trxName transaction
	 */
	public static void setIsPaid (Ctx ctx, int C_BPartner_ID, String trxName)
	{
		int counter = 0;
		String sql = "SELECT * FROM C_Billing "
			+ "WHERE IsPaid='N' AND DocStatus IN ('CO','CL')";
		if (C_BPartner_ID > 1)
			sql += " AND C_BPartner_ID=?";
		else
			sql += " AND AD_Client_ID=" + ctx.getAD_Client_ID();
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, trxName);
			if (C_BPartner_ID > 1)
				pstmt.setInt (1, C_BPartner_ID);
			ResultSet rs = pstmt.executeQuery ();
			while (rs.next ())
			{
				MBilling billing = new MBilling(ctx, rs, trxName);
				// TODO
				// if (billing.testAllocation())
					if (billing.save())
						counter++;
			}
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		s_log.config("#" + counter);
		/**/
	}	//	setIsPaid

	/**
	 * 	Get Document Status
	 *	@return Document Status Clear Text
	 */
	public String getDocStatusName()
	{
		return MRefList.getListName(getCtx(), 131, getDocStatus());
	}	//	getDocStatusName

	
	/**************************************************************************
	 * 	Create PDF
	 *	@return File or null
	 */
	public File createPDF ()
	{
		try
		{
			File temp = File.createTempFile(get_TableName()+get_ID()+"_", ".pdf");
			return createPDF (temp);
		}
		catch (Exception e)
		{
			log.severe("Could not create PDF - " + e.getMessage());
		}
		return null;
	}	//	getPDF

	/**
	 * 	Create PDF file
	 *	@param file output file
	 *	@return file if success
	 */
	public File createPDF (File file)
	{
		ReportEngine re = ReportEngine.get (getCtx(), ReportEngine.INVOICE, getC_Invoice_ID());
		if (re == null)
			return null;
		return re.getPDF(file);
	}	//	createPDF

	/**
	 * 	Get PDF File Name
	 *	@param documentDir directory
	 *	@return file name
	 */
	public String getPDFFileName (String documentDir)
	{
		return getPDFFileName (documentDir, getC_Invoice_ID());
	}	//	getPDFFileName

	/**
	 *	Get ISO Code of Currency
	 *	@return Currency ISO
	 */
	public String getCurrencyISO()
	{
		return MCurrency.getISO_Code (getCtx(), getC_Currency_ID());
	}	//	getCurrencyISO

	/**
	 * 	Get Currency Precision
	 *	@return precision
	 */
	public int getPrecision()
	{
		return MCurrency.getStdPrecision(getCtx(), getC_Currency_ID());
	}	//	getPrecision
	
	
	/**************************************************************************
	 * 	Process document
	 *	@param processAction document action
	 *	@return true if performed
	 */
	public boolean processIt (String processAction)
	{
		m_processMsg = null;
		DocumentEngine engine = new DocumentEngine (this, getDocStatus());
		return engine.processIt (processAction, getDocAction());
	}	//	process
	
	/**	Process Message 			*/
	private String		m_processMsg = null;
	/**	Just Prepared Flag			*/
	private boolean		m_justPrepared = false;

	/**
	 * 	Unlock Document.
	 * 	@return true if success 
	 */
	public boolean unlockIt()
	{
		log.info("unlockIt - " + toString());
		setProcessing(false);
		return true;
	}	//	unlockIt
	
	/**
	 * 	Invalidate Document
	 * 	@return true if success 
	 */
	public boolean invalidateIt()
	{
		log.info("invalidateIt - " + toString());
		setDocAction(DOCACTION_Prepare);
		return true;
	}	//	invalidateIt
	
	/**
	 *	Prepare Document
	 * 	@return new status (In Progress or Invalid) 
	 */
	public String prepareIt()
	{
		log.info(toString());
		m_processMsg = ModelValidationEngine.get().fireDocValidate(this, ModelValidator.DOCTIMING_BEFORE_PREPARE);
		if (m_processMsg != null)
			return DocAction.STATUS_Invalid;
		MDocType dt = MDocType.get(getCtx(), getC_DocTypeTarget_ID());

		//	Std Period open?
		if (!MPeriod.isOpen(getCtx(), getDateAcct(), dt.getDocBaseType()))
		{
			m_processMsg = "@PeriodClosed@";
			return DocAction.STATUS_Invalid;
		}
		//	Lines
		MBillingLine[] lines = getLines(true);
		if (lines.length == 0)
		{
			m_processMsg = "@NoLines@";
			return DocAction.STATUS_Invalid;
		}
		//	No Cash Book
		if (PAYMENTRULE_Cash.equals(getPaymentRule())
			&& MCashBook.get(getCtx(), getAD_Org_ID(), getC_Currency_ID()) == null)
		{
			m_processMsg = "@NoCashBook@";
			return DocAction.STATUS_Invalid;
		}

		//	Convert/Check DocType
		if (getC_DocType_ID() != getC_DocTypeTarget_ID() )
			setC_DocType_ID(getC_DocTypeTarget_ID());
		if (getC_DocType_ID() == 0)
		{
			m_processMsg = "No Document Type";
			return DocAction.STATUS_Invalid;
		}

		explodeBOM();
		if (!calculateTaxTotal())	//	setTotals
		{
			m_processMsg = "Error calculating Tax";
			return DocAction.STATUS_Invalid;
		}
			
		//	Credit Status
		if (isSOTrx() && !isReversal())
		{
			MBPartner bp = new MBPartner (getCtx(), getC_BPartner_ID(), null);
			if (MBPartner.SOCREDITSTATUS_CreditStop.equals(bp.getSOCreditStatus()))
			{
				m_processMsg = "@BPartnerCreditStop@ - @TotalOpenBalance@=" 
					+ bp.getTotalOpenBalance()
					+ ", @SO_CreditLimit@=" + bp.getSO_CreditLimit();
				return DocAction.STATUS_Invalid;
			}
		}
		
		//	Add up Amounts
		m_justPrepared = true;
		if (!DOCACTION_Complete.equals(getDocAction()))
			setDocAction(DOCACTION_Complete);
		return DocAction.STATUS_InProgress;
		
	}	//	prepareIt
	
	/**
	 * 	Explode non stocked BOM.
	 */
	private void explodeBOM ()
	{
		String where = "AND IsActive='Y' AND EXISTS "
			+ "(SELECT * FROM M_Product p WHERE C_BillingLine.M_Product_ID=p.M_Product_ID"
			+ " AND	p.IsBOM='Y' AND p.IsVerified='Y' AND p.IsStocked='N')";
		//
		String sql = "SELECT COUNT(*) FROM C_BillingLine "
			+ "WHERE C_Billing_ID=? " + where; 
		int count = DB.getSQLValue(get_TrxName(), sql, getC_Billing_ID());
		while (count != 0)
		{
			renumberLines (100);

			//	Order Lines with non-stocked BOMs
			MBillingLine[] lines = getLines (where);
			for (int i = 0; i < lines.length; i++)
			{
				MBillingLine line = lines[i];
				MProduct product = MProduct.get (getCtx(), line.getM_Product_ID());
				log.fine(product.getName());
				//	New Lines
				int lineNo = line.getLine ();
				MProductBOM[] boms = MProductBOM.getBOMLines (product);
				for (int j = 0; j < boms.length; j++)
				{
					MProductBOM bom = boms[j];
					MBillingLine newLine = new MBillingLine (this);
					newLine.setLine (++lineNo);
					newLine.setM_Product_ID (bom.getProduct().getM_Product_ID(),
						bom.getProduct().getC_UOM_ID());
					newLine.setQty (line.getQtyBilling().multiply(
						bom.getBOMQty ()));		//	Invoiced/Entered
					if (bom.getDescription () != null)
						newLine.setDescription (bom.getDescription ());
					//
					newLine.setPrice ();
					newLine.save (get_TrxName());
				}
				//	Convert into Comment Line
				line.setM_Product_ID (0);
				line.setM_AttributeSetInstance_ID (0);
				line.setPriceEntered (Env.ZERO);
				line.setPriceActual (Env.ZERO);
				line.setPriceLimit (Env.ZERO);
				line.setPriceList (Env.ZERO);
				line.setLineNetAmt (Env.ZERO);
				//
				String description = product.getName ();
				if (product.getDescription () != null)
					description += " " + product.getDescription ();
				if (line.getDescription () != null)
					description += " " + line.getDescription ();
				line.setDescription (description);
				line.save (get_TrxName());
			} //	for all lines with BOM

			m_lines = null;
			count = DB.getSQLValue (get_TrxName(), sql, getC_Billing_ID ());
			renumberLines (10);
		}	//	while count != 0
	}	//	explodeBOM
	
	/**
	 * 	Calculate Tax and Total
	 * 	@return true if calculated
	 */
	private boolean calculateTaxTotal()
	{
		log.fine("");
		//	Delete Taxes
		DB.executeUpdate("DELETE FROM C_BillingTax WHERE C_Billing_ID=" + getC_Billing_ID(), get_TrxName());
		m_taxes = null;
		
		//	Lines
		BigDecimal totalLines = Env.ZERO;
		ArrayList<Integer> taxList = new ArrayList<Integer>();
		MBillingLine[] lines = getLines(false);
		for (int i = 0; i < lines.length; i++)
		{
			MBillingLine line = lines[i];
			/**	Sync ownership for SO
			if (isSOTrx() && line.getAD_Org_ID() != getAD_Org_ID())
			{
				line.setAD_Org_ID(getAD_Org_ID());
				line.save();
			}	**/
			Integer taxID = new Integer(line.getC_Tax_ID());
			if (!taxList.contains(taxID))
			{
				MBillingTax iTax = MBillingTax.get (line, getPrecision(), 
					false, get_TrxName());	//	current Tax
				if (iTax != null)
				{
					iTax.setIsTaxIncluded(isTaxIncluded());
					if (!iTax.calculateTaxFromLines())
						return false;
					if (!iTax.save())
						return false;
					taxList.add(taxID);
				}
			}
			totalLines = totalLines.add(line.getLineNetAmt());
		}
		
		//	Taxes
		BigDecimal grandTotal = totalLines;
		MBillingTax[] taxes = getTaxes(true);
		for (int i = 0; i < taxes.length; i++)
		{
			MBillingTax iTax = taxes[i];
			MTax tax = iTax.getTax();
			if (tax.isSummary())
			{
				MTax[] cTaxes = tax.getChildTaxes(false);	//	Multiple taxes
				for (int j = 0; j < cTaxes.length; j++)
				{
					MTax cTax = cTaxes[j];
					
					MBPartner partner = MBPartner.get(getCtx(), getC_BPartner_ID());
					BigDecimal taxAmt = cTax.calculateTax(iTax.getTaxBaseAmt(), isTaxIncluded(), getPrecision(), partner.getTaxRoundModeAsInt());
					//
					MBillingTax newITax = new MBillingTax(getCtx(), 0, get_TrxName());
					newITax.setClientOrg(this);
					newITax.setC_Billing_ID(getC_Billing_ID());
					newITax.setC_Tax_ID(cTax.getC_Tax_ID());
					newITax.setPrecision(getPrecision());
					newITax.setIsTaxIncluded(isTaxIncluded());
					newITax.setTaxBaseAmt(iTax.getTaxBaseAmt());
					newITax.setTaxAmt(taxAmt);
					if (!newITax.save(get_TrxName()))
						return false;
					//
					if (!isTaxIncluded())
						grandTotal = grandTotal.add(taxAmt);
				}
				if (!iTax.delete(true, get_TrxName()))
					return false;
			}
			else
			{
				if (!isTaxIncluded())
					grandTotal = grandTotal.add(iTax.getTaxAmt());
			}
		}		
		//
		setTotalLines(totalLines);
		setGrandTotal(grandTotal);
		return true;
	}	//	calculateTaxTotal

	
	/**
	 * 	Approve Document
	 * 	@return true if success 
	 */
	public boolean  approveIt()
	{
		log.info(toString());
		setIsApproved(true);
		return true;
	}	//	approveIt
	
	/**
	 * 	Reject Approval
	 * 	@return true if success 
	 */
	public boolean rejectIt()
	{
		log.info(toString());
		setIsApproved(false);
		return true;
	}	//	rejectIt
	
	/**
	 * 	Complete Document
	 * 	@return new status (Complete, In Progress, Invalid, Waiting ..)
	 */
	public String completeIt()
	{
		//	Re-Check
		if (!m_justPrepared)
		{
			String status = prepareIt();
			if (!DocAction.STATUS_InProgress.equals(status))
				return status;
		} else {
			if (!calculateTaxTotal())	//	setTotals
			{
				m_processMsg = "Error calculating Tax";
				return DocAction.STATUS_Invalid;
			}
		}
		//	Implicit Approval
		if (!isApproved())
			approveIt();
		log.info(toString());
		StringBuffer info = new StringBuffer();
		
		BigDecimal taxInvoice = BigDecimal.ZERO;
		HashMap<Integer, Integer> taxIDs = new HashMap<Integer, Integer>();
		
		//	Set lines to 0
		MBillingLine[] lines = getLines(false);
		for (int i = 0; i < lines.length; i++)
		{
			MBillingLine line = lines[i];
			MInvoiceLine invoiceline = new MInvoiceLine(getCtx(), line.getC_InvoiceLine_ID(), get_TrxName());
			invoiceline.setIsBilled(true);
			invoiceline.save();
			log.fine(" Set Billed for Line=" + invoiceline.getC_InvoiceLine_ID());
			Integer C_Invoice_ID = taxIDs.get(invoiceline.getC_Invoice_ID());
			if( C_Invoice_ID == null ){
				MInvoice invoice = new MInvoice(getCtx(), invoiceline.getC_Invoice_ID(), get_TrxName());
				taxIDs.put(invoiceline.getC_Invoice_ID(), invoiceline.getC_Invoice_ID());
				log.fine(" get tax for invoice=" + invoice.getDocumentNo());
				MInvoiceTax[] taxs = invoice.getTaxes(true);
				for(int t=0; t<taxs.length; t++){
					MInvoiceTax tax = taxs[t];
					taxInvoice = taxInvoice.add(tax.getTaxAmt());
					log.fine(" Added Tax=" + tax.getTaxAmt());
				}
			}
		}
		taxInvoice = taxInvoice.subtract(getTaxAmt());
		log.fine(" Tax Diff=" + taxInvoice);
		
		// Jirimuto added for tax adjust logic - 2009/07/28
		if( !taxInvoice.equals(BigDecimal.ZERO) ){
			MClientInfo clientinfo = new MClientInfo(getCtx(), getAD_Client_ID(), get_TrxName());
			int taxAdjustCharge_ID = clientinfo.getC_TaxAdjustCharge_ID();
			if( taxAdjustCharge_ID != 0 ){
				MInvoice invoice = new MInvoice(getCtx(), 0, get_TrxName());
				invoice.setAD_Org_ID(this.getAD_Org_ID());
				invoice.setAD_OrgTrx_ID(this.getAD_OrgTrx_ID());
				invoice.setAD_User_ID(this.getAD_User_ID());
				invoice.setC_Activity_ID(this.getC_Activity_ID());
				invoice.setC_BPartner_ID(this.getC_BPartner_ID());
				invoice.setC_BPartner_Location_ID(this.getC_BPartner_Location_ID());
				
				invoice.setC_ConversionType_ID(this.getC_ConversionType_ID());
				invoice.setC_Currency_ID(this.getC_Currency_ID());
				
				MInvoice firstInvoice = new MInvoice(getCtx(), this.getC_Invoice_ID(), get_TrxName());
				invoice.setC_DocTypeTarget_ID(firstInvoice.getC_DocTypeTarget_ID());
				invoice.setC_DocType_ID(firstInvoice.getC_DocType_ID());
				
				invoice.setC_Order_ID(this.getC_Order_ID());
				invoice.setC_PaymentTerm_ID(this.getC_PaymentTerm_ID());
				invoice.setC_Currency_ID(this.getC_Currency_ID());
				invoice.setDateAcct(this.getDateAcct());
				invoice.setDateInvoiced(this.getDateInvoiced());
				invoice.setDateOrdered(this.getDateOrdered());
				invoice.setDescription(Msg.translate(getCtx(), "C_Billing_ID")+":"+this.getDocumentNo()+" ==> "
						+Msg.translate(getCtx(), "C_TaxAdjustInvoice_ID"));
				invoice.setIsSOTrx(this.isSOTrx());
				invoice.setM_PriceList_ID(this.getM_PriceList_ID());
				invoice.setPaymentRule(this.getPaymentRule());
				invoice.setSalesRep_ID(this.getSalesRep_ID());
				if( invoice.save() ){
					MInvoiceLine invoiceline = new MInvoiceLine(invoice);
					
					invoiceline.setC_Charge_ID(taxAdjustCharge_ID);
					invoiceline.setQty(new BigDecimal(1));
					invoiceline.setPriceList(taxInvoice);
					invoiceline.setPrice(taxInvoice);
					log.fine(" Set Tax Revenue line=" + taxInvoice);
					MTax[] taxs = MTax.getAll(getCtx());
					for(int t=0; t<taxs.length; t++){
						MTax tax = taxs[t];
						if( BigDecimal.ZERO.equals(tax.getRate()) || tax.isTaxExempt() ){
							invoiceline.setC_Tax_ID(tax.getC_Tax_ID());
							break;
						}
					}
					invoiceline.setIsBilled(true);
					invoiceline.setIsPrinted(true);
					if( invoiceline.save() ){
						log.fine(" Saved Tax Revenue Line Amt=" + invoiceline.getPriceActual());
					}
					invoice.processIt(MInvoice.DOCACTION_Complete);
					if( invoice.save() ){
						log.fine(" Saved Tax Amt=" + invoice.getGrandTotal());
						this.setC_TaxAdjustInvoice_ID(invoice.getC_Invoice_ID());
					}
				}
				
			}
			
		}
		
		m_processMsg = info.toString().trim();
		setProcessed(true);
		setDocAction(DOCACTION_Close);
		return DocAction.STATUS_Completed;
	}	//	completeIt
		/**
	 * 	Void Document.
	 * 	@return true if success 
	 */
	public boolean voidIt()
	{
		log.info(toString());
		if (DOCSTATUS_Closed.equals(getDocStatus())
			|| DOCSTATUS_Reversed.equals(getDocStatus())
			|| DOCSTATUS_Voided.equals(getDocStatus()))
		{
			m_processMsg = "Document Closed: " + getDocStatus();
			setDocAction(DOCACTION_None);
			return false;
		}

		//	Not Processed
		if (DOCSTATUS_Drafted.equals(getDocStatus())
			|| DOCSTATUS_Invalid.equals(getDocStatus())
			|| DOCSTATUS_InProgress.equals(getDocStatus())
			|| DOCSTATUS_Approved.equals(getDocStatus())
			|| DOCSTATUS_NotApproved.equals(getDocStatus()) )
		{
			//	Set lines to 0
			MBillingLine[] lines = getLines(false);
			for (int i = 0; i < lines.length; i++)
			{
				MBillingLine line = lines[i];
				BigDecimal old = line.getQtyBilling();
				if (old.compareTo(Env.ZERO) != 0)
				{
					line.setQty(Env.ZERO);
					line.setTaxAmt(Env.ZERO);
					line.setLineNetAmt(Env.ZERO);
					line.setLineTotalAmt(Env.ZERO);
					line.addDescription(Msg.getMsg(getCtx(), "Voided") + " (" + old + ")");
					//	Unlink Shipment
					if (line.getC_InvoiceLine_ID() != 0)
					{
						MInvoiceLine invoiceline = new MInvoiceLine(getCtx(), line.getC_InvoiceLine_ID(), get_TrxName());
						invoiceline.setIsBilled(false);
						invoiceline.save();						
					}
					line.save(get_TrxName());
				}
			}
			addDescription(Msg.getMsg(getCtx(), "Voided"));
			setIsPaid(true);
			setC_Payment_ID(0);
		}
		else
		{
			return reverseCorrectIt();
		}
			
		setProcessed(true);
		setDocAction(DOCACTION_None);
		return true;
	}	//	voidIt
	
	/**
	 * 	Close Document.
	 * 	@return true if success 
	 */
	public boolean closeIt()
	{
		log.info(toString());
		setProcessed(true);
		setDocAction(DOCACTION_None);
		return true;
	}	//	closeIt
	
	/**
	 * 	Reverse Correction - same date
	 * 	@return true if success 
	 */
	public boolean reverseCorrectIt()
	{
		log.info(toString());
		MDocType dt = MDocType.get(getCtx(), getC_DocType_ID());
		if (!MPeriod.isOpen(getCtx(), getDateAcct(), dt.getDocBaseType()))
		{
			m_processMsg = "@PeriodClosed@";
			return false;
		}
		
		//	Set lines to 0
		MBillingLine[] lines = getLines(false);
		for (int i = 0; i < lines.length; i++)
		{
			MBillingLine line = lines[i];
			BigDecimal old = line.getQtyBilling();
			if (old.compareTo(Env.ZERO) != 0)
			{
				line.setQty(Env.ZERO);
				line.setTaxAmt(Env.ZERO);
				line.setLineNetAmt(Env.ZERO);
				line.setLineTotalAmt(Env.ZERO);
				line.addDescription(Msg.getMsg(getCtx(), "Voided") + " (" + old + ")");
				//	Unlink Shipment
				if (line.getC_InvoiceLine_ID() != 0)
				{
					MInvoiceLine invoiceline = new MInvoiceLine(getCtx(), line.getC_InvoiceLine_ID(), get_TrxName());
					invoiceline.setIsBilled(false);
					invoiceline.save();
				}
				line.save(get_TrxName());
			}
		}
		addDescription(Msg.getMsg(getCtx(), "Voided"));
		setIsPaid(true);
		setC_Payment_ID(0);
		
		return true;
	}	//	reverseCorrectIt
	
	/**
	 * 	Reverse Accrual - none
	 * 	@return false 
	 */
	public boolean reverseAccrualIt()
	{
		log.info(toString());
		return false;
	}	//	reverseAccrualIt
	
	/** 
	 * 	Re-activate
	 * 	@return false 
	 */
	public boolean reActivateIt()
	{
		log.info(toString());
		return false;
	}	//	reActivateIt
	
	
	/*************************************************************************
	 * 	Get Summary
	 *	@return Summary of Document
	 */
	public String getSummary()
	{
		StringBuffer sb = new StringBuffer();
		sb.append(getDocumentNo());
		//	: Grand Total = 123.00 (#1)
		sb.append(": ").
			append(Msg.translate(getCtx(),"GrandTotal")).append("=").append(getGrandTotal())
			.append(" (#").append(getLines(false).length).append(")");
		//	 - Description
		if (getDescription() != null && getDescription().length() > 0)
			sb.append(" - ").append(getDescription());
		return sb.toString();
	}	//	getSummary
	
	/**
	 * 	Get Process Message
	 *	@return clear text error message
	 */
	public String getProcessMsg()
	{
		return m_processMsg;
	}	//	getProcessMsg
	
	/**
	 * 	Get Document Owner (Responsible)
	 *	@return AD_User_ID
	 */
	public int getDoc_User_ID()
	{
		return getSalesRep_ID();
	}	//	getDoc_User_ID

	/**
	 * 	Get Document Approval Amount
	 *	@return amount
	 */
	public BigDecimal getApprovalAmt()
	{
		return getGrandTotal();
	}	//	getApprovalAmt
	
	
}	//	MInvoice
