/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.model;

import java.math.*;
import java.sql.*;
import java.util.*;
import java.util.logging.*;
import org.compiere.util.*;

/**
 *	Billing Line Model
 *
 *  @author Jirimuto
 *  @version $Id: MBillingLine.java,v 1.2 2010/06/14 04:04:25 jrmt Exp $
 */
public class MBillingLine extends X_C_BillingLine
{
	
	/**
	 * 	Get Billing Line referencing InOut Line
	 *	@param iLine invoice line
	 *	@return (first) Billing line
	 */
	public static MBillingLine getOfInvoiceLine (MInvoiceLine iLine)
	{
		if (iLine == null)
			return null;
		MBillingLine retValue = null;
		String sql = "SELECT * FROM C_BillingLine WHERE C_InvoiceLine_ID=?";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, iLine.get_TrxName());
			pstmt.setInt (1, iLine.getC_InvoiceLine_ID());
			ResultSet rs = pstmt.executeQuery ();
			if (rs.next ())
			{
				retValue = new MBillingLine (iLine.getCtx(), rs, iLine.get_TrxName());
				if (rs.next())
					s_log.warning("More than one C_BillingLine of " + iLine);
			}
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e); 
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		return retValue;
	}	//	getOfInvoiceLine
	
	/**
	 * 	Get Billing Line referencing InOut Line
	 *	@param sLine shipment line
	 *	@return (first) Billing line
	 */
	public static MBillingLine getOfInOutLine (MInOutLine sLine)
	{
		if (sLine == null)
			return null;
		MBillingLine retValue = null;
		String sql = "SELECT * FROM C_BillingLine WHERE M_InOutLine_ID=?";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, sLine.get_TrxName());
			pstmt.setInt (1, sLine.getM_InOutLine_ID());
			ResultSet rs = pstmt.executeQuery ();
			if (rs.next ())
			{
				retValue = new MBillingLine (sLine.getCtx(), rs, sLine.get_TrxName());
				if (rs.next())
					s_log.warning("More than one C_BillingLine of " + sLine);
			}
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e); 
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		return retValue;
	}	//	getOfInOutLine
	
	/**	Static Logger	*/
	private static CLogger	s_log	= CLogger.getCLogger (MBillingLine.class);

	
	/**************************************************************************
	 * 	Billing Line Constructor
	 * 	@param ctx context
	 * 	@param C_BillingLine_ID billing line or 0
	 * 	@param trxName transaction name
	 */
	public MBillingLine (Ctx ctx, int C_BillingLine_ID, String trxName)
	{
		super (ctx, C_BillingLine_ID, trxName);
		if (C_BillingLine_ID == 0)
		{
			setIsDescription(false);
			setIsPrinted (true);
			setLineNetAmt (Env.ZERO);
			setPriceEntered (Env.ZERO);
			setPriceActual (Env.ZERO);
			setPriceLimit (Env.ZERO);
			setPriceList (Env.ZERO);
			setM_AttributeSetInstance_ID(0);
			setTaxAmt(Env.ZERO);
			//
			setQtyEntered(Env.ZERO);
			setQtyBilling(Env.ZERO);
		}
	}	//	MBillingLine
	
	/**
	 * 	Parent Constructor
	 * 	@param billing parent
	 */
	public MBillingLine (MBilling billing)
	{
		this (billing.getCtx(), 0, billing.get_TrxName());
		if (billing.get_ID() == 0)
			throw new IllegalArgumentException("Header not saved");
		setClientOrg(billing.getAD_Client_ID(), billing.getAD_Org_ID());
		setC_Billing_ID (billing.getC_Billing_ID());
		setBilling(billing);
	}	//	MBillingLine


	/**
	 *  Load Constructor
	 *  @param ctx context
	 *  @param rs result set record
	 *  @param trxName transaction
	 */
	public MBillingLine (Ctx ctx, ResultSet rs, String trxName)
	{
		super(ctx, rs, trxName);
	}	//	MBillingLine

	private int			m_M_PriceList_ID = 0;
	private Timestamp	m_DateInvoiced = null;
	private Timestamp	m_DateBilling = null;
	private int			m_C_BPartner_ID = 0;
	private int			m_C_BPartner_Location_ID = 0;
	private boolean		m_IsSOTrx = true;
	private boolean		m_priceSet = false;
	private MProduct	m_product = null;
	
	/**	Cached Name of the line		*/
	private String		m_name = null;
	/** Cached Precision			*/
	private Integer		m_precision = null;
	/** Product Pricing				*/
	private MProductPricing	m_productPricing = null;
	/** Parent						*/
	private MBilling	m_parent = null;

	/**
	 * 	Set Defaults from billing.
	 * 	Called also from copy lines from billing
	 * 	Does not set Parent !!
	 * 	@param billing billing
	 */
	public void setBilling (MBilling billing)
	{
		m_parent = billing;
		m_M_PriceList_ID = billing.getM_PriceList_ID();
		m_DateInvoiced = billing.getDateInvoiced();
		m_DateBilling = billing.getDateBilling();
		m_C_BPartner_ID = billing.getC_BPartner_ID();
		m_C_BPartner_Location_ID = billing.getC_BPartner_Location_ID();
		m_IsSOTrx = billing.isSOTrx();
		m_precision = new Integer(billing.getPrecision());
	}	//	setOrder

	/**
	 * 	Get Parent
	 *	@return parent
	 */
	public MBilling getParent()
	{
		if (m_parent == null)
			m_parent = new MBilling(getCtx(), getC_Billing_ID(), get_TrxName());
		return m_parent;
	}	//	getParent
		
	/**
	 * 	Set values from invoice Line.
	 * 	Does not set quantity!
	 *	@param iLine invoice line
	 */
	public void setInvoiceLine (MInvoice invoice, MInvoiceLine iLine)
	{
		setC_InvoiceLine_ID(iLine.getC_InvoiceLine_ID());
		setM_InOutLine_ID(iLine.getM_InOutLine_ID());
		setC_OrderLine_ID(iLine.getC_OrderLine_ID());
		//
		setIsDescription(iLine.isDescription());
		setDescription(iLine.getDescription());
		//
		setM_Product_ID(iLine.getM_Product_ID());
		setC_UOM_ID(iLine.getC_UOM_ID());
		setM_AttributeSetInstance_ID(iLine.getM_AttributeSetInstance_ID());
		setS_ResourceAssignment_ID(iLine.getS_ResourceAssignment_ID());
		setC_Charge_ID(iLine.getC_Charge_ID());
		//
		setPriceActual(iLine.getPriceActual());
		setPriceEntered(iLine.getPriceEntered());
		setPriceLimit(iLine.getPriceLimit());
		setPriceList(iLine.getPriceList());
		setC_Tax_ID(iLine.getC_Tax_ID());
		
		if( !invoice.isReturnTrx() ){
			setQtyEntered(iLine.getQtyEntered());
			setQtyBilling(iLine.getQtyInvoiced());
			
			setLineNetAmt(iLine.getLineNetAmt());
			setLineTotalAmt(iLine.getLineTotalAmt());
		} else {
			setQtyEntered(iLine.getQtyEntered().negate());
			setQtyBilling(iLine.getQtyInvoiced().negate());
			
			setLineNetAmt(iLine.getLineNetAmt().negate());
			setLineTotalAmt(iLine.getLineTotalAmt().negate());
		}
		
		//
		setRRAmt(iLine.getRRAmt());
		setRRStartDate(iLine.getRRStartDate());
		//
		setA_Asset_ID(iLine.getA_Asset_ID());
		setC_Project_ID(iLine.getC_Project_ID());
		setC_ProjectPhase_ID(iLine.getC_ProjectPhase_ID());
		setC_ProjectTask_ID(iLine.getC_ProjectTask_ID());
		setC_Activity_ID(iLine.getC_Activity_ID());
		setC_Campaign_ID(iLine.getC_Campaign_ID());
		setAD_OrgTrx_ID(iLine.getAD_OrgTrx_ID());
		setUser1_ID(iLine.getUser1_ID());
		setUser2_ID(iLine.getUser2_ID());
	}	//	setInvoiceLine

	/**
	 * 	Add to Description
	 *	@param description text
	 */
	public void addDescription (String description)
	{
		String desc = getDescription();
		if (desc == null)
			setDescription(description);
		else
			setDescription(desc + " | " + description);
	}	//	addDescription

	/**
	 * 	Set M_AttributeSetInstance_ID
	 *	@param M_AttributeSetInstance_ID id
	 */
	public void setM_AttributeSetInstance_ID (int M_AttributeSetInstance_ID)
	{
		if (M_AttributeSetInstance_ID == 0)		//	 0 is valid ID
			set_Value("M_AttributeSetInstance_ID", new Integer(0));
		else
			super.setM_AttributeSetInstance_ID (M_AttributeSetInstance_ID);
	}	//	setM_AttributeSetInstance_ID

	
	/**************************************************************************
	 * 	Set Price for Product and PriceList.
	 * 	Uses standard SO price list of not set by billing constructor
	 */
	public void setPrice()
	{
		if (getM_Product_ID() == 0 || isDescription())
			return;
		if (m_M_PriceList_ID == 0 || m_C_BPartner_ID == 0)
			setBilling(getParent());
		if (m_M_PriceList_ID == 0 || m_C_BPartner_ID == 0)
			throw new IllegalStateException("setPrice - PriceList unknown!");
		setPrice (m_M_PriceList_ID, m_C_BPartner_ID);
	}	//	setPrice
	
	/**
	 * 	Set Price for Product and PriceList
	 * 	@param M_PriceList_ID price list
	 * 	@param C_BPartner_ID business partner
	 */
	public void setPrice (int M_PriceList_ID, int C_BPartner_ID)
	{
		if (getM_Product_ID() == 0 || isDescription())
			return;
		//
		log.fine("M_PriceList_ID=" + M_PriceList_ID);
		m_productPricing = new MProductPricing (getM_Product_ID(), 
			C_BPartner_ID, getQtyBilling(), m_IsSOTrx);
		m_productPricing.setM_PriceList_ID(M_PriceList_ID);
		m_productPricing.setPriceDate(m_DateInvoiced);
		//
		setPriceActual (m_productPricing.getPriceStd());
		setPriceList (m_productPricing.getPriceList());
		setPriceLimit (m_productPricing.getPriceLimit());
		//
		if (getQtyEntered().compareTo(getQtyBilling()) == 0)
			setPriceEntered(getPriceActual());
		else
			setPriceEntered(getPriceActual().multiply(getQtyBilling()
				.divide(getQtyEntered(), 6, BigDecimal.ROUND_HALF_UP)));	//	precision
		//
		if (getC_UOM_ID() == 0)
			setC_UOM_ID(m_productPricing.getC_UOM_ID());
		//
		m_priceSet = true;
	}	//	setPrice

	/**
	 * 	Set Price Entered/Actual.
	 * 	Use this Method if the Line UOM is the Product UOM 
	 *	@param PriceActual price
	 */
	public void setPrice (BigDecimal PriceActual)
	{
		setPriceEntered(PriceActual);
		setPriceActual (PriceActual);
	}	//	setPrice

	/**
	 * 	Set Price Actual.
	 * 	(actual price is not updateable)
	 *	@param PriceActual actual price
	 */
	public void setPriceActual (BigDecimal PriceActual)
	{
		if (PriceActual == null) 
			throw new IllegalArgumentException ("PriceActual is mandatory");
		set_ValueNoCheck("PriceActual", PriceActual);
	}	//	setPriceActual

	
	/**
	 *	Set Tax - requires Warehouse
	 *	@return true if found
	 */
	public boolean setTax()
	{
		if (isDescription())
			return true;
		//
		int M_Warehouse_ID = getCtx().getContextAsInt("#M_Warehouse_ID");
		//
		int C_Tax_ID = Tax.get(getCtx(), getM_Product_ID(), getC_Charge_ID() , m_DateInvoiced, m_DateInvoiced,
			getAD_Org_ID(), M_Warehouse_ID,
			m_C_BPartner_Location_ID,		//	should be bill to
			m_C_BPartner_Location_ID, m_IsSOTrx);
		if (C_Tax_ID == 0)
		{
			log.log(Level.SEVERE, "No Tax found");
			return false;
		}
		setC_Tax_ID (C_Tax_ID);
		if (m_IsSOTrx)
		{
			;
		}
		return true;
	}	//	setTax

	/**
	 * 	Calculare Tax Amt.
	 * 	Assumes Line Net is calculated
	 */
	public void setTaxAmt ()
	{
		BigDecimal TaxAmt = Env.ZERO; 
		if (getC_Tax_ID() == 0)
			return;
		// Jirimuto modified for Tax calculation Adjustment for Planex Spec. 2009/03/16
		MBilling billing = getParent();
		MBPartner partner = MBPartner.get(getCtx(), billing.getC_BPartner_ID());
		
		//	setLineNetAmt();
		// Jirimuto modified for Tax calculation Adjustment for Planex Spec. 2009/03/16
		MTax tax = MTax.get (getCtx(), getC_Tax_ID());
		if ( partner.isTaxDocumentLevel(tax.isDocumentLevel()) && m_IsSOTrx)		//	AR Inv Tax
			return;
		//
		TaxAmt = tax.calculateTax(getLineNetAmt(), isTaxIncluded(), getPrecision(), partner.getTaxRoundModeAsInt());
		if (isTaxIncluded())
			setLineTotalAmt(getLineNetAmt());
		else
			setLineTotalAmt(getLineNetAmt().add(TaxAmt));
		super.setTaxAmt (TaxAmt);
	}	//	setTaxAmt
	
	/**
	 * 	Calculate Extended Amt.
	 * 	May or may not include tax
	 */
	public void setLineNetAmt ()
	{
		//	Calculations & Rounding
		BigDecimal net = getPriceActual().multiply(getQtyBilling()); 
		if (net.scale() > getPrecision())
			net = net.setScale(getPrecision(), BigDecimal.ROUND_HALF_UP);
		super.setLineNetAmt (net);
	}	//	setLineNetAmt
	
	/**
	 * 	Set Qty Billing/Entered.
	 *	@param Qty Invoiced/Ordered
	 */
	public void setQty (int Qty)
	{
		setQty(new BigDecimal(Qty));
	}	//	setQtyBilling

	/**
	 * 	Set Qty Billing
	 *	@param Qty Billing/Entered
	 */
	public void setQty (BigDecimal Qty)
	{
		setQtyEntered(Qty);
		setQtyBilling(getQtyEntered());
	}	//	setQtyBilling

	/**
	 * 	Set Qty Entered - enforce entered UOM 
	 *	@param QtyEntered
	 */
	public void setQtyEntered (BigDecimal QtyEntered)
	{
		if (QtyEntered != null && getC_UOM_ID() != 0)
		{
			int precision = MUOM.getPrecision(getCtx(), getC_UOM_ID());
			QtyEntered = QtyEntered.setScale(precision, BigDecimal.ROUND_HALF_UP);
		}
		super.setQtyEntered (QtyEntered);
	}	//	setQtyEntered

	/**
	 * 	Set Qty Billing - enforce Product UOM 
	 *	@param QtyBilling
	 */
	public void setQtyBilling (BigDecimal QtyBilling)
	{
		MProduct product = getProduct();
		if (QtyBilling != null && product != null)
		{
			int precision = product.getUOMPrecision();
			QtyBilling = QtyBilling.setScale(precision, BigDecimal.ROUND_HALF_UP);
		}
		super.setQtyBilling(QtyBilling);
	}	//	setQtyBilling

	/**
	 * 	Set Product
	 *	@param product product
	 */
	public void setProduct (MProduct product)
	{
		m_product = product;
		if (m_product != null)
		{
			setM_Product_ID(m_product.getM_Product_ID());
			setC_UOM_ID (m_product.getC_UOM_ID());
		}
		else
		{
			setM_Product_ID(0);
			setC_UOM_ID (0);
		}
		setM_AttributeSetInstance_ID(0);
	}	//	setProduct

	
	/**
	 * 	Set M_Product_ID
	 *	@param M_Product_ID product
	 *	@param setUOM set UOM from product
	 */
	public void setM_Product_ID (int M_Product_ID, boolean setUOM)
	{
		if (setUOM)
			setProduct(MProduct.get(getCtx(), M_Product_ID));
		else
			super.setM_Product_ID (M_Product_ID);
		setM_AttributeSetInstance_ID(0);
	}	//	setM_Product_ID
	
	/**
	 * 	Set Product and UOM
	 *	@param M_Product_ID product
	 *	@param C_UOM_ID uom
	 */
	public void setM_Product_ID (int M_Product_ID, int C_UOM_ID)
	{
		super.setM_Product_ID (M_Product_ID);
		super.setC_UOM_ID(C_UOM_ID);
		setM_AttributeSetInstance_ID(0);
	}	//	setM_Product_ID
	
	/**
	 * 	Get Product
	 *	@return product or null
	 */
	public MProduct getProduct()
	{
		if (m_product == null && getM_Product_ID() != 0)
			m_product =  MProduct.get (getCtx(), getM_Product_ID());
		return m_product;
	}	//	getProduct

	/**
	 * 	Get C_Project_ID
	 *	@return project
	 */
	public int getC_Project_ID()
	{
		int ii = super.getC_Project_ID ();
		if (ii == 0)
			ii = getParent().getC_Project_ID();
		return ii;
	}	//	getC_Project_ID
	
	/**
	 * 	Get C_Activity_ID
	 *	@return Activity
	 */
	public int getC_Activity_ID()
	{
		int ii = super.getC_Activity_ID ();
		if (ii == 0)
			ii = getParent().getC_Activity_ID();
		return ii;
	}	//	getC_Activity_ID
	
	/**
	 * 	Get C_Campaign_ID
	 *	@return Campaign
	 */
	public int getC_Campaign_ID()
	{
		int ii = super.getC_Campaign_ID ();
		if (ii == 0)
			ii = getParent().getC_Campaign_ID();
		return ii;
	}	//	getC_Campaign_ID
	
	/**
	 * 	Get User2_ID
	 *	@return User2
	 */
	public int getUser1_ID ()
	{
		int ii = super.getUser1_ID ();
		if (ii == 0)
			ii = getParent().getUser1_ID();
		return ii;
	}	//	getUser1_ID

	/**
	 * 	Get User2_ID
	 *	@return User2
	 */
	public int getUser2_ID ()
	{
		int ii = super.getUser2_ID ();
		if (ii == 0)
			ii = getParent().getUser2_ID();
		return ii;
	}	//	getUser2_ID

	/**
	 * 	Get AD_OrgTrx_ID
	 *	@return trx org
	 */
	public int getAD_OrgTrx_ID()
	{
		int ii = super.getAD_OrgTrx_ID();
		if (ii == 0)
			ii = getParent().getAD_OrgTrx_ID();
		return ii;
	}	//	getAD_OrgTrx_ID

	/**
	 * 	String Representation
	 *	@return info
	 */
	public String toString ()
	{
		StringBuffer sb = new StringBuffer ("MBillingLine[")
			.append(get_ID()).append(",").append(getLine())
			.append(",QtyBilling=").append(getQtyBilling())
			.append(",LineNetAmt=").append(getLineNetAmt())
			.append ("]");
		return sb.toString ();
	}	//	toString

	/**
	 * 	Get (Product/Charge) Name
	 * 	@return name
	 */
	public String getName ()
	{
		if (m_name == null)
		{
			String sql = "SELECT COALESCE (p.Name, c.Name) "
				+ "FROM C_BillingLine il"
				+ " LEFT OUTER JOIN M_Product p ON (il.M_Product_ID=p.M_Product_ID)"
				+ " LEFT OUTER JOIN C_Charge C ON (il.C_Charge_ID=c.C_Charge_ID) "
				+ "WHERE C_BillingLine_ID=?";
			PreparedStatement pstmt = null;
			try
			{
				pstmt = DB.prepareStatement(sql, get_TrxName());
				pstmt.setInt(1, getC_BillingLine_ID());
				ResultSet rs = pstmt.executeQuery();
				if (rs.next())
					m_name = rs.getString(1);
				rs.close();
				pstmt.close();
				pstmt = null;
				if (m_name == null)
					m_name = "??";
			}
			catch (Exception e)
			{
				log.log(Level.SEVERE, "getName", e);
			}
			finally
			{
				try
				{
					if (pstmt != null)
						pstmt.close ();
				}
				catch (Exception e)
				{}
				pstmt = null;
			}
		}
		return m_name;
	}	//	getName

	/**
	 * 	Set Temporary (cached) Name
	 * 	@param tempName Cached Name
	 */
	public void setName (String tempName)
	{
		m_name = tempName;
	}	//	setName

	/**
	 * 	Get Description Text.
	 * 	For jsp access (vs. isDescription)
	 *	@return description
	 */
	public String getDescriptionText()
	{
		return super.getDescription();
	}	//	getDescriptionText

	/**
	 * 	Get Currency Precision
	 *	@return precision
	 */
	public int getPrecision()
	{
		if (m_precision != null)
			return m_precision.intValue();
		
		String sql = "SELECT c.StdPrecision "
			+ "FROM C_Currency c INNER JOIN C_Billing x ON (x.C_Currency_ID=c.C_Currency_ID) "
			+ "WHERE x.C_Billing_ID=?";
		int i = DB.getSQLValue(get_TrxName(), sql, getC_Billing_ID());
		if (i < 0)
		{
			log.warning("getPrecision = " + i + " - set to 2");
			i = 2;
		}
		m_precision = new Integer(i);
		return m_precision.intValue();
	}	//	getPrecision
	
	/**
	 *	Is Tax Included in Amount
	 *	@return true if tax is included
	 */
	public boolean isTaxIncluded()
	{
		if (m_M_PriceList_ID == 0)
		{
			m_M_PriceList_ID = DB.getSQLValue(get_TrxName(),
				"SELECT M_PriceList_ID FROM C_Billing WHERE C_Billing_ID=?",
				getC_Billing_ID());
		}
		MPriceList pl = MPriceList.get(getCtx(), m_M_PriceList_ID, get_TrxName());
		return pl.isTaxIncluded();
	}	//	isTaxIncluded
	
	
	/**************************************************************************
	 * 	Before Save
	 *	@param newRecord
	 *	@return true if save
	 */
	protected boolean beforeSave (boolean newRecord)
	{
		log.fine("New=" + newRecord);
		//	Charge
		if (getC_Charge_ID() != 0)
		{
			if (getM_Product_ID() != 0)
				setM_Product_ID(0);
		}
		else	//	Set Product Price
		{
			if (!m_priceSet 
				&&  Env.ZERO.compareTo(getPriceActual()) == 0
				&&  Env.ZERO.compareTo(getPriceList()) == 0)
				setPrice();
		}
		
		//	Set Tax
		if (getC_Tax_ID() == 0)
			setTax();

		//	Get Line No
		if (getLine() == 0)
		{
			String sql = "SELECT COALESCE(MAX(Line),0)+10 FROM C_BillingLine WHERE C_Billing_ID=?";
			int ii = DB.getSQLValue (get_TrxName(), sql, getC_Billing_ID());
			setLine (ii);
		}
		//	UOM
		if (getC_UOM_ID() == 0)
		{
			int C_UOM_ID = MUOM.getDefault_UOM_ID(getCtx());
			if (C_UOM_ID > 0)
				setC_UOM_ID (C_UOM_ID);
		}
		//	Qty Precision
		if (newRecord || is_ValueChanged("QtyEntered"))
			setQtyEntered(getQtyEntered());
		if (newRecord || is_ValueChanged("QtyBilling"))
			setQtyBilling(getQtyBilling());

		//	Calculations & Rounding
		setLineNetAmt();
		if (getTaxAmt().compareTo(Env.ZERO) == 0)
			setTaxAmt();
		//
		return true;
	}	//	beforeSave
	
	
	/**
	 * 	After Save
	 *	@param newRecord new
	 *	@param success success
	 *	@return saved
	 */
	protected boolean afterSave (boolean newRecord, boolean success)
	{
		if (!success)
			return success;
		if (!newRecord && is_ValueChanged("C_Tax_ID"))
		{
			//	Recalculate Tax for old Tax
			MBillingTax tax = MBillingTax.get (this, getPrecision(), 
				true, get_TrxName());	//	old Tax
			if (tax != null)
			{
				if (!tax.calculateTaxFromLines())
					return false;
				if (!tax.save(get_TrxName()))
					return true;
			}
		}
		return updateHeaderTax();
	}	//	afterSave

	/**
	 * 	After Delete
	 *	@param success success
	 *	@return deleted
	 */
	protected boolean afterDelete (boolean success)
	{
		if (!success)
			return success;
		return updateHeaderTax();
	}	//	afterDelete
	
	/**
	 *	Update Tax & Header
	 *	@return true if header updated with tax
	 */
	private boolean updateHeaderTax()
	{
		//	Recalculate Tax for this Tax
		MBillingTax tax = MBillingTax.get (this, getPrecision(), 
			false, get_TrxName());	//	current Tax
		if (tax != null)
		{
			if (!tax.calculateTaxFromLines())
				return false;
			if (!tax.save(get_TrxName()))
				return false;
		}
		
		//	Update Billing Header
		String sql = "UPDATE C_Billing i"
			+ " SET TotalLines="
				+ "(SELECT COALESCE(SUM(LineNetAmt),0) FROM C_BillingLine il WHERE i.C_Billing_ID=il.C_Billing_ID) "
			+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		int no = DB.executeUpdate(sql, get_TrxName());
		if (no != 1)
			log.warning("(1) #" + no);

		//	Update Billing Header
		sql = "UPDATE C_Billing i"
			+ " SET TaxAmt="
				+ "(SELECT COALESCE(SUM(TaxAmt),0) FROM C_BillingTax it WHERE i.C_Billing_ID=it.C_Billing_ID) "
			+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		no = DB.executeUpdate(sql, get_TrxName());
		if (no != 1)
			log.warning("(2) #" + no);

		if (isTaxIncluded())
			sql = "UPDATE C_Billing i "
				+ " SET GrandTotal=TotalLines "
				+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		else
			sql = "UPDATE C_Billing i "
				+ " SET GrandTotal=TotalLines+"
					+ "(SELECT COALESCE(SUM(TaxAmt),0) FROM C_BillingTax it WHERE i.C_Billing_ID=it.C_Billing_ID) "
					+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		no = DB.executeUpdate(sql, get_TrxName());
		if (no != 1)
			log.warning("(3) #" + no);
		
		if (!isTaxIncluded())
			sql = "UPDATE C_Billing i "
				+ " SET RevenueNetAmt=TotalLines "
				+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		else
			sql = "UPDATE C_Billing i "
				+ " SET RevenueNetAmt=TotalLines-"
					+ "(SELECT COALESCE(SUM(TaxAmt),0) FROM C_BillingTax it WHERE i.C_Billing_ID=it.C_Billing_ID) "
					+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		no = DB.executeUpdate(sql, get_TrxName());
		if (no != 1)
			log.warning("(4) #" + no);
		
		sql = "UPDATE C_Billing i "
			+ " SET TotalInvoiceAmt=GrandTotal+LastTransferedAmt "
			+ "WHERE C_Billing_ID=" + getC_Billing_ID();
		
		no = DB.executeUpdate(sql, get_TrxName());
		if (no != 1)
			log.warning("(5) #" + no);
		
		m_parent = null;
		
		return no == 1;
	}	//	updateHeaderTax
	
}	//	MBillingLine
