/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.model;

import java.math.*;
import java.sql.*;
import java.util.logging.*;
import org.compiere.util.*;

/**
 *	Billing Tax Model
 *	
 *  @author Jirimuto
 *  @version $Id: MBillingTax.java,v 1.1 2010/04/09 09:14:45 jrmt Exp $
 */
public class MBillingTax extends X_C_BillingTax
{
	/**
	 * 	Get Tax Line for Billing Line
	 *	@param line billing line
	 *	@param precision currency precision
	 *	@param oldTax if true old tax is returned
	 *	@param trxName transaction name
	 *	@return existing or new tax
	 */
	public static MBillingTax get (MBillingLine line, int precision, 
		boolean oldTax, String trxName)
	{
		MBillingTax retValue = null;
		if (line == null || line.getC_Billing_ID() == 0 || line.isDescription())
			return null;
		int C_Tax_ID = line.getC_Tax_ID();
		if (oldTax && line.is_ValueChanged("C_Tax_ID"))
		{
			Object old = line.get_ValueOld("C_Tax_ID");
			if (old == null)
				return null;
			C_Tax_ID = ((Integer)old).intValue();
		}
		if (C_Tax_ID == 0)
		{
			s_log.warning("C_Tax_ID=0");
			return null;
		}
		
		String sql = "SELECT * FROM C_BillingTax WHERE C_Billing_ID=? AND C_Tax_ID=?";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, trxName);
			pstmt.setInt (1, line.getC_Billing_ID());
			pstmt.setInt (2, C_Tax_ID);
			ResultSet rs = pstmt.executeQuery ();
			if (rs.next ())
			{
				retValue = new MBillingTax (line.getCtx(), rs, trxName);
			}
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			s_log.log(Level.SEVERE, sql, e); 
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		if (retValue != null)
		{
			retValue.set_TrxName(trxName);
			retValue.setPrecision(precision);
			s_log.fine("(old=" + oldTax + ") " + retValue);
			return retValue;
		}
		
		//	Create New
		retValue = new MBillingTax(line.getCtx(), 0, trxName);
		retValue.set_TrxName(trxName);
		retValue.setClientOrg(line);
		retValue.setC_Billing_ID(line.getC_Billing_ID());
		retValue.setC_Tax_ID(line.getC_Tax_ID());
		retValue.setPrecision(precision);
		retValue.setIsTaxIncluded(line.isTaxIncluded());
		s_log.fine("(new) " + retValue);
		return retValue;
	}	//	get
	
	/**	Static Logger	*/
	private static CLogger	s_log	= CLogger.getCLogger (MBillingTax.class);
	
	
	/**************************************************************************
	 * 	Persistency Constructor
	 *	@param ctx context
	 *	@param ignored ignored
	 *	@param trxName transaction
	 */
	public MBillingTax (Ctx ctx, int ignored, String trxName)
	{
		super(ctx, 0, trxName);
		if (ignored != 0)
			throw new IllegalArgumentException("Multi-Key");
		setTaxAmt (Env.ZERO);
		setTaxBaseAmt (Env.ZERO);
		setIsTaxIncluded(false);
	}	//	MBillingTax

	/**
	 * 	Load Constructor.
	 * 	Set Precision and TaxIncluded for tax calculations!
	 *	@param ctx context
	 *	@param rs result set
	 *	@param trxName transaction
	 */
	public MBillingTax (Ctx ctx, ResultSet rs, String trxName)
	{
		super(ctx, rs, trxName);
	}	//	MBillingTax
	
	/** Tax							*/
	private MTax 		m_tax = null;
	/** Cached Precision			*/
	private Integer		m_precision = null;
	
	
	/**
	 * 	Get Precision
	 * 	@return Returns the precision or 2
	 */
	private int getPrecision ()
	{
		if (m_precision == null)
			return 2;
		return m_precision.intValue();
	}	//	getPrecision

	/**
	 * 	Set Precision
	 *	@param precision The precision to set.
	 */
	protected void setPrecision (int precision)
	{
		m_precision = new Integer(precision);
	}	//	setPrecision

	/**
	 * 	Get Tax
	 *	@return tax
	 */
	protected MTax getTax()
	{
		if (m_tax == null)
			m_tax = MTax.get(getCtx(), getC_Tax_ID());
		return m_tax;
	}	//	getTax
	
	/**************************************************************************
	 * 	Calculate/Set Tax Base Amt from Billing Lines
	 * 	@return true if tax calculated
	 */
	public boolean calculateTaxFromLines ()
	{
		BigDecimal totalBaseAmt = Env.ZERO;
		BigDecimal taxBaseAmt = Env.ZERO;
		BigDecimal taxAmt = Env.ZERO;
		int m_InOut_ID = 0;
		MBilling billing = new MBilling(getCtx(), getC_Billing_ID(), get_TrxName());
		MBPartner partner = MBPartner.get(getCtx(), billing.getC_BPartner_ID());
		
		//
		boolean documentLevel = getTax().isDocumentLevel();
		documentLevel = partner.isTaxDocumentLevel(documentLevel);
		boolean shipmentLevel = false;
		if( documentLevel ) {
			shipmentLevel = partner.isTaxShipmentLevel();
		}
		
		MTax tax = getTax();
		//
		String sql = "SELECT il.LineNetAmt, COALESCE(il.TaxAmt,0), i.IsSOTrx, iol.M_InOut_ID "
			+ "FROM C_BillingLine il"
			+ " INNER JOIN C_Billing i ON (il.C_Billing_ID=i.C_Billing_ID) "
			+ " LEFT JOIN M_InOutLine iol ON (il.M_InOutLine_ID=iol.M_InOutLine_ID) "
			+ "WHERE il.C_Billing_ID=? AND il.C_Tax_ID=? ORDER BY iol.M_InOut_ID ";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, get_TrxName());
			pstmt.setInt (1, getC_Billing_ID());
			pstmt.setInt (2, getC_Tax_ID());
			ResultSet rs = pstmt.executeQuery ();
			while (rs.next ())
			{
				int temp_InOut_ID = rs.getInt(4);
				if( documentLevel && shipmentLevel && m_InOut_ID!=temp_InOut_ID ){
					BigDecimal shipTaxAmt = tax.calculateTax(taxBaseAmt, isTaxIncluded(), getPrecision(), partner.getTaxRoundModeAsInt());
					taxAmt = taxAmt.add(shipTaxAmt);
					taxBaseAmt = Env.ZERO;
					m_InOut_ID = temp_InOut_ID;
				}
				
				//	BaseAmt
				BigDecimal baseAmt = rs.getBigDecimal(1);
				taxBaseAmt = taxBaseAmt.add(baseAmt);
				totalBaseAmt = totalBaseAmt.add(baseAmt);
				
				//	TaxAmt
				BigDecimal amt = rs.getBigDecimal(2);
				if (amt == null)
					amt = Env.ZERO;
				boolean isSOTrx = "Y".equals(rs.getString(3));
				//
				if ( documentLevel || baseAmt.signum() == 0 )
					amt = Env.ZERO;
				else if (amt.signum() != 0 && !isSOTrx)	//	manually entered
					;
				else	// calculate line tax
					amt = tax.calculateTax(baseAmt, isTaxIncluded(), getPrecision(), partner.getTaxRoundModeAsInt());
				//
				taxAmt = taxAmt.add(amt);
			}
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "setTaxBaseAmt", e);
			taxBaseAmt = null;
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		if (taxBaseAmt == null)
			return false;
		
		//	Calculate Tax
		if (documentLevel || taxAmt.signum() == 0){
			BigDecimal amt = tax.calculateTax(taxBaseAmt, isTaxIncluded(), getPrecision(), partner.getTaxRoundModeAsInt());
			taxAmt = taxAmt.add(amt);
		}
		setTaxAmt(taxAmt);

		//	Set Base
		if (isTaxIncluded())
			setTaxBaseAmt (totalBaseAmt.subtract(taxAmt));
		else
			setTaxBaseAmt (totalBaseAmt);
		return true;
	}	//	calculateTaxFromLines

	/**
	 * 	String Representation
	 *	@return info
	 */
	public String toString ()
	{
		StringBuffer sb = new StringBuffer ("MBillingTax[");
		sb.append("C_Billing_ID=").append(getC_Billing_ID())
			.append(",C_Tax_ID=").append(getC_Tax_ID())
			.append(", Base=").append(getTaxBaseAmt()).append(",Tax=").append(getTaxAmt())
			.append ("]");
		return sb.toString ();
	}	//	toString

}	//	MBillingTax
