/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.process;

import java.math.BigDecimal;
import java.sql.*;
import java.util.Calendar;
import java.util.logging.*;
import org.compiere.model.*;
import org.compiere.util.*;

/**
 *	Generate Invoices
 *	
 *  @author Jirimuto
 *  @version $Id: BillingGenerate.java,v 1.3 2010/06/14 04:04:25 jrmt Exp $
 */
public class BillingGenerate extends SvrProcess
{
	/**	Manual Selection		*/
	private boolean 	p_Selection = false;
	/**	C_Billing_ID			*/
	private int			p_C_Billing_ID = 0;
	/**	Date Invoiced			*/
	private Timestamp	p_DateBilling = null;
	/**	Date Invoiced			*/
	private Timestamp	p_DateClosed = null;
	/**	Org						*/
	private int			p_AD_Org_ID = 0;
	/** BPartner				*/
	private int			p_C_BPartner_ID = 0;
	/**	IsSOTrx					*/
	private String		p_IsSOTrx = "Y";
	/** Invoice Document Action	*/
	private String		p_docAction = DocAction.ACTION_None;
	
	/**	The current Billing	*/
	private MBilling 	m_billing = null;
	/** Numner of Billing		*/
	private int			m_created = 0;
	/** Numner of Lines		*/
	private int			m_linecreated = 0;
	
	/**	Date Invoiced			*/
	private Timestamp	p_CutoffDate = null;
	/**
	 *  Prepare - e.g., get Parameters.
	 */
	protected void prepare()
	{
		ProcessInfoParameter[] para = getParameter();
		for (int i = 0; i < para.length; i++)
		{
			String name = para[i].getParameterName();
			if (para[i].getParameter() == null)
				;
			else if (name.equals("Selection"))
				p_Selection = "Y".equals(para[i].getParameter());
			else if (name.equals("DateBilling"))
				p_DateBilling = (Timestamp)para[i].getParameter();
			else if (name.equals("DateClosed"))
				p_DateClosed = (Timestamp)para[i].getParameter();
			else if (name.equals("AD_Org_ID"))
				p_AD_Org_ID = para[i].getParameterAsInt();
			else if (name.equals("C_BPartner_ID"))
				p_C_BPartner_ID = para[i].getParameterAsInt();
			else if (name.equals("IsSOTrx"))
				p_IsSOTrx = (String)para[i].getParameter();
			else if (name.equals("DocAction"))
				p_docAction = (String)para[i].getParameter();
			else
				log.log(Level.SEVERE, "Unknown Parameter: " + name);
		}

		//	Login Date
		if (p_DateBilling == null)
			p_DateBilling = getCtx().getContextAsDate("#Date");
		if (p_DateBilling == null)
			p_DateBilling = TimeUtil.getDay(System.currentTimeMillis());

		if (p_DateClosed == null)
			p_DateClosed = getCtx().getContextAsDate("#Date");
		if (p_DateClosed == null){
			p_DateClosed = TimeUtil.getDay(System.currentTimeMillis());
		}
		
		//	DocAction check
		if (!DocAction.ACTION_Complete.equals(p_docAction) 
				&& !DocAction.ACTION_Prepare.equals(p_docAction) )
			p_docAction = DocAction.ACTION_None;
	}	//	prepare

	/**
	 * 	Generate Billing
	 *	@return info
	 *	@throws Exception
	 */
	protected String doIt () throws Exception
	{
		p_C_Billing_ID = getRecord_ID();
		if( p_C_Billing_ID != 0 ){
			m_billing = new MBilling(getCtx(), p_C_Billing_ID, get_TrxName());
			p_DateBilling = m_billing.getDateBilling();
			p_AD_Org_ID = m_billing.getAD_Org_ID();
			p_C_BPartner_ID = m_billing.getC_BPartner_ID();
			p_IsSOTrx = m_billing.isSOTrx()?"Y":"N";
		}
		
		log.info("Selection=" + p_Selection + ", DateBilling=" + p_DateBilling
			+ ", AD_Org_ID=" + p_AD_Org_ID + ", C_BPartner_ID=" + p_C_BPartner_ID
		    + ", DocAction=" + p_docAction);
		//
		
		String sql = null;
		if (p_Selection)	//	VBillingGen
		{
			sql = "SELECT i.* FROM C_Invoice i, C_BPartner bp "
				+ "WHERE i.C_BPartner_ID=bp.C_BPartner_ID AND i.IsSelected='Y' AND i.DocStatus IN ('CO', 'CL') AND IsPaid='N' AND i.IsSOTrx='"+p_IsSOTrx+"' "
				+ "ORDER BY bp.C_BPartner_ID, i.C_Invoice_ID";
		}
		else
		{
			sql = "SELECT i.* FROM C_Invoice i, C_BPartner bp  "
				+ " WHERE i.C_BPartner_ID=bp.C_BPartner_ID AND i.DocStatus IN('CO','CL') AND i.IsPaid='N' AND i.IsSOTrx='"+p_IsSOTrx+"'";
			if (p_AD_Org_ID != 0)
				sql += " AND i.AD_Org_ID=?";
			if (p_C_BPartner_ID != 0)
				sql += " AND ( bp.C_BPartner_ID=? )";
			if (p_DateClosed != null)
				sql += " AND ( i.DateInvoiced<=? )";
			//
			sql += " AND EXISTS (SELECT * FROM C_InvoiceLine il "
					+ "WHERE i.C_Invoice_ID=il.C_Invoice_ID AND il.IsBilled='N'" 
					+ "AND NOT EXISTS (SELECT * FROM C_BillingLine bl, C_Billing b " 
					+ "WHERE bl.C_Billing_ID=b.C_Billing_ID " 
					+ "AND bl.C_InvoiceLine_ID=il.C_InvoiceLine_ID AND b.DocStatus NOT IN ('VO', 'RE'))  ) "
				+ "ORDER BY bp.C_BPartner_ID, i.C_Invoice_ID";
		}
		
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement (sql, get_TrxName());
			int index = 1;
			if (!p_Selection && p_AD_Org_ID != 0)
				pstmt.setInt(index++, p_AD_Org_ID);
			if (!p_Selection && p_C_BPartner_ID != 0){
				pstmt.setInt(index++, p_C_BPartner_ID);
			}
			if (!p_Selection && p_DateClosed != null){
				pstmt.setTimestamp(index++, p_DateClosed);
			}
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, sql, e);
		}
		
		return generate(pstmt);
	}	//	doIt
	
	/**
	 * 	Generate Invoice
	 * 	@param pstmt order query 
	 *	@return info
	 */
	private String generate (PreparedStatement pstmt)
	{
		MBPartner m_bp = null;
		try
		{
			ResultSet rs = pstmt.executeQuery ();
			while (rs.next ()) {
				MInvoice invoice = new MInvoice (getCtx(), rs, get_TrxName());
				m_bp = new MBPartner(getCtx(), invoice.getC_BPartner_ID(), get_TrxName());
				MOrder m_o = new MOrder(getCtx(), invoice.getC_Order_ID(), get_TrxName());
											
				//	New Invoice Location
				if ( p_C_Billing_ID==0 && m_billing != null 
					&& m_billing.getC_BPartner_Location_ID() != m_bp.getPrimaryC_BPartner_Location_ID() )
					completeBilling();
				
				p_CutoffDate = null;
				//	Schedule After Delivery
				boolean doInvoice = false;
				if (MBPartner.INVOICERULE_CustomerScheduleAfterDelivery.equals(m_bp.getInvoiceRule()))
				{
					if (m_bp.getC_InvoiceSchedule_ID() == 0)
					{
						doInvoice = true;
					}
					else
					{
						MInvoiceSchedule is = MInvoiceSchedule.get(getCtx(), m_bp.getC_InvoiceSchedule_ID(), get_TrxName());
						p_CutoffDate = is.getCutoffDate(p_DateBilling, p_DateClosed);
						Timestamp invoicabledate = invoice.getDateInvoiced();
						if (is.canInvoice(invoicabledate, p_DateBilling, p_DateClosed, invoice.getGrandTotal()))
							doInvoice = true;
						else
							continue;
					}
				}	//	Schedule
				
				//	After Delivery
				if (doInvoice || MBPartner.INVOICERULE_AfterDelivery.equals(m_bp.getInvoiceRule()))
				{
					MInvoiceLine[] invoiceLines = invoice.getLines();
					for (int i = 0; i < invoiceLines.length; i++)
					{
						MInvoiceLine invoiceLine = invoiceLines[i];
						if (invoiceLine.isBilled() || invoiceLine.isDescription() )
							continue;
						//
						createLine (m_bp, invoice, invoiceLine);
					}
				}
				//	After Order Delivered, Immediate
				else if ( MBPartner.INVOICERULE_AfterOrderDelivered.equals(m_bp.getInvoiceRule()))
				{
					MInvoiceLine[] invoiceLines = invoice.getLines();
					for (int i = 0; i < invoiceLines.length; i++)
					{
						MInvoiceLine invoiceLine = invoiceLines[i];
						if (invoiceLine.isBilled() || invoiceLine.isDescription() )
							continue;
						
						if( invoiceLine.getC_OrderLine_ID() > 0 ){
							MOrderLine orderline = new MOrderLine(getCtx(), invoiceLine.getC_OrderLine_ID(), get_TrxName());
							MOrder order = new MOrder(getCtx(), orderline.getC_Order_ID(), get_TrxName());
							MOrderLine[] orderlines = order.getLines();
							boolean completeOrder = true;
							for(int j=0; j<orderlines.length; j++){
								orderline = orderlines[j];
								boolean fullyDelivered = orderline.getQtyOrdered().compareTo(orderline.getQtyDelivered()) == 0;
						
								//	Complete Order
								if (!fullyDelivered)
								{
									log.fine("Failed CompleteOrder - " + invoiceLine);
									completeOrder = false;
									break;
								}
							}
							if( completeOrder ){
								createLine (m_bp, invoice, invoiceLine);
							}
						} else {
							createLine (m_bp, invoice, invoiceLine);
						}
					}
				}
				//	Immediate
				else if (MBPartner.INVOICERULE_Immediate.equals(m_bp.getInvoiceRule()))
				{
					MInvoiceLine[] invoiceLines = invoice.getLines();
					for (int i = 0; i < invoiceLines.length; i++)
					{
						MInvoiceLine invoiceLine = invoiceLines[i];
						if (invoiceLine.isBilled())
							continue;
						//
						createLine (m_bp, invoice, invoiceLine);
					}
				}
				else
					log.fine("Failed: " + m_bp.getInvoiceRule() 
						+ " - " + invoice);
			}	//	for all invoices
			rs.close ();
			pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "", e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close ();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		completeBilling();
		
		return "@Created@ = " + m_created + ", @Lines@ = " + m_linecreated;
	}	//	generate
	
	/**************************************************************************
	 * 	Create Invoice Line from Order Line
	 *	@param invoice invoice
	 *	@param orderLine line
	 *	@param qtyInvoiced qty
	 *	@param qtyEntered qty
	 */
	private void createLine (MBPartner partner, MInvoice invoice, MInvoiceLine invoiceLine)
	{
		if (m_billing == null)
		{
			m_billing = new MBilling (partner, invoice, 0, p_DateBilling);
			if( p_CutoffDate != null )
				m_billing.setDateAcct(p_CutoffDate);
			
			if (!m_billing.save())
				throw new IllegalStateException("Could not create Billing (o)");
		}
		//
		MBillingLine line = new MBillingLine (m_billing);
		line.setInvoiceLine(invoice, invoiceLine);
		if (!line.save())
			throw new IllegalStateException("Could not create Billing Line (o)");
		m_linecreated++;
		log.fine(line.toString());
	}	//	createLine

	/**
	 * 	Complete Billing
	 */
	private void completeBilling()
	{
		if (m_billing != null)
		{
			if( !m_billing.isAdjusted() ){
				MBilling lastBilling = MBilling.getLastBilling(getCtx(), m_billing.getC_BPartner_ID(), m_billing.getDateBilling(), m_billing.get_TrxName());
				BigDecimal lastInvoicedAmt = BigDecimal.ZERO;
				Timestamp lastDateBilling = m_billing.getDateBilling();
				if( lastBilling != null ){
					lastInvoicedAmt = lastBilling.getTotalInvoiceAmt();
					m_billing.setLastInvoicedAmt(lastBilling.getTotalInvoiceAmt());
					lastDateBilling = lastBilling.getDateBilling();
				} else {
					Calendar lastBillingCal = TimeUtil.getToday();
					if( lastBillingCal!=null ){
						lastBillingCal.setTimeInMillis(m_billing.getDateBilling().getTime());
						lastBillingCal.set(Calendar.MILLISECOND, 0);
						lastBillingCal.set(Calendar.SECOND, 0);
						lastBillingCal.set(Calendar.MINUTE, 0);
						lastBillingCal.set(Calendar.HOUR_OF_DAY, 0);
						lastBillingCal.add(Calendar.DAY_OF_MONTH, -1);
						lastDateBilling = new Timestamp (lastBillingCal.getTimeInMillis());
					}
				}
				
				BigDecimal lastPaid = MPayment.getSumOfBPartner(getCtx(), m_billing.getC_BPartner_ID(), lastDateBilling, m_billing.getDateBilling(), m_billing.get_TrxName());
				m_billing.setLastPaidAmt(lastPaid);
				m_billing.setAdjustAmt(BigDecimal.ZERO);
				BigDecimal lastTansferedAmt = lastInvoicedAmt.subtract(lastPaid);
				m_billing.setLastTransferedAmt(lastTansferedAmt);
			}
			if( !DocAction.ACTION_None.equals(p_docAction) ){
				if (!m_billing.processIt(p_docAction))
				log.warning("completeBilling - failed: " + m_billing);
			}
			m_billing.save();
			commit();
			
			//
			addLog(m_billing.getC_Billing_ID(), m_billing.getDateBilling(), new BigDecimal(m_linecreated), m_billing.getDocumentNo());
			m_created++;
		}
		m_billing = null;
	}	//	completeBilling
	
}	//	BillingGenerate
