/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2006 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software; you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program; if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 2620 Augustine Dr. #245, Santa Clara, CA 95054, USA        *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.process;

import java.math.*;
import java.rmi.RemoteException;
import java.sql.*;
import java.util.logging.*;

import org.compiere.db.CConnection;
import org.compiere.interfaces.Server;
import org.compiere.util.*;

/**
 *	Accounting Fact Reset
 *	
 *  @author Jorg Janke
 *  @version $Id: FactAcctRepost.java,v 1.1 2010/06/14 02:43:18 jrmt Exp $
 */
public class FactAcctRepost extends SvrProcess
{
	/**	Client Parameter		*/
	private int		p_AD_Client_ID = 0;
	/** Table Parameter			*/
	private int		p_AD_Table_ID = 0;
	/**	Date Acct			*/
	private Timestamp	p_DateAcct_From = null;
	private Timestamp	p_DateAcct_To = null;
	
	private int		m_countRepost = 0;
	private int		m_countError = 0;
	
	/**	Server Re-tries		*/
	private static int 		s_serverTries = 0;
	/**	Server Session		*/
	private static Server	s_server = null;
	/**	Logger			*/
	private static CLogger log = CLogger.getCLogger(FactAcctRepost.class);
	/**
	 *  Prepare - e.g., get Parameters.
	 */
	protected void prepare()
	{
		ProcessInfoParameter[] para = getParameter();
		for (int i = 0; i < para.length; i++)
		{
			String name = para[i].getParameterName();
			if (para[i].getParameter() == null)
				;
			else if (name.equals("AD_Client_ID"))
				p_AD_Client_ID = ((BigDecimal)para[i].getParameter()).intValue();
			else if (name.equals("AD_Table_ID"))
				p_AD_Table_ID = ((BigDecimal)para[i].getParameter()).intValue();
			else if (name.equals("DateAcct")){
				p_DateAcct_From = (Timestamp)para[i].getParameter();
				p_DateAcct_To = (Timestamp)para[i].getParameter_To();
				if (p_DateAcct_From == null)
				{
					p_DateAcct_From = p_DateAcct_To;
				}
				if (p_DateAcct_To == null)
				{
					p_DateAcct_To = p_DateAcct_From;
				}
			}
			else
				log.log(Level.SEVERE, "Unknown Parameter: " + name);
		}
	}	//	prepare

	/**
	 *  Perform process.
	 *  @return Message (clear text)
	 *  @throws Exception if not successful
	 */
	protected String doIt() throws Exception
	{
		log.info("AD_Client_ID=" + p_AD_Client_ID 
			+ ", AD_Table_ID=" + p_AD_Table_ID + ", DateAcct_From=" + p_DateAcct_From + ", DateAcct_To" + p_DateAcct_To);
		//	List of Tables with Accounting Consequences
		String sql = "SELECT AD_Table_ID, TableName "
			+ "FROM AD_Table t "
			+ "WHERE t.IsView='N'";
		if (p_AD_Table_ID > 0)
			sql += " AND t.AD_Table_ID=" + p_AD_Table_ID;
		sql += " AND EXISTS (SELECT * FROM AD_Column c "
				+ "WHERE t.AD_Table_ID=c.AD_Table_ID AND c.ColumnName='Posted' AND c.IsActive='Y')";
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, get_TrxName());
			ResultSet rs = pstmt.executeQuery();
			while (rs.next())
			{
				int AD_Table_ID = rs.getInt(1);
				String TableName = rs.getString(2);
				repost (TableName, AD_Table_ID);
			}
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, sql, e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
		//
		return "@Posted@ = " + m_countRepost + " @Error@ = " + m_countError;
	}	//	doIt

	/**
	 * 	Reset Accounting Table and update count
	 *	@param TableName table
	 */
	private void repost (String TableName, int AD_Table_ID)
	{
		//	List of Tables with Accounting Consequences
		String sql = "SELECT "+ TableName + "_ID, DATEACCT "
			+ " FROM " + TableName + " WHERE AD_Client_ID=" + p_AD_Client_ID
			+ " AND Processed='Y' AND Posted='Y' ";
		
		if( "C_BANKSTATEMENT".equalsIgnoreCase(TableName) )
			sql = sql + " AND StatementDate>=? AND StatementDate<=? ";
		else 
			sql = sql + " AND DateAcct>=? AND DateAcct<=? ";
		
		PreparedStatement pstmt = null;
		try
		{
			pstmt = DB.prepareStatement(sql, get_TrxName());
			pstmt.setTimestamp(1, p_DateAcct_From);
			pstmt.setTimestamp(2, p_DateAcct_To);
			ResultSet rs = pstmt.executeQuery();
			while (rs.next())
			{
				int Record_ID = rs.getInt(1);
				Timestamp DateAcct = rs.getTimestamp(2);
				
				String error = postImmediate(p_AD_Client_ID, AD_Table_ID, Record_ID, true);
				if( "".equals( error ) || error == null ){
					// addLog(Record_ID, DateAcct, null, TableName + ": Posted");
					log.info(TableName + ": Posted :" + Record_ID);
					m_countRepost++;
				}
				else {
					// addLog(Record_ID, DateAcct, null, TableName + ": Error, " + error );
					log.warning(TableName + ": Error :" + Record_ID);
					m_countError++;
				}
				commit();
			}
			rs.close();
			pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, sql, e);
		}
		try
		{
			if (pstmt != null)
				pstmt.close();
			pstmt = null;
		}
		catch (Exception e)
		{
			pstmt = null;
		}
	}	//	reset

	/**
	 *  Post Immediate
	 *  @param  WindowNo 		window
	 *  @param  AD_Table_ID     Table ID of Document
	 *  @param  AD_Client_ID    Client ID of Document
	 *  @param  Record_ID       Record ID of this document
	 *  @param  force           force posting
	 *  @return null if success, otherwise error
	 */
	public static String postImmediate (int AD_Client_ID, 
		int AD_Table_ID, int Record_ID, boolean force)
	{
		log.config("AD_Table_ID=" + AD_Table_ID + "/" + Record_ID
			+ ", Force=" + force);

		String error = null;
		//  try to get from Server when enabled
		if (isServerActive())
		{
			log.config("trying server");
			try
			{
				s_server = CConnection.get().getServer();
				if (s_server != null)
				{
					error = s_server.postImmediate(Env.getCtx(), AD_Client_ID, 
						AD_Table_ID, Record_ID, force, null);
					log.config("from Server: " + error== null ? "OK" : error);
				}
				else
				{
					return "NoAppsServer";
				}
			}
			catch (RemoteException e)
			{
				log.log(Level.WARNING, "(RE)", e);
				error = e.getMessage();
				s_server = null;
			}
			catch (Exception e)
			{
				log.log(Level.WARNING, "ex", e);
				error = e.getMessage();
				s_server = null;
			}
		}
		else
		{
			return "NoAppsServer";
		}
		return error;
	}   //  postImmediate
		
	/**
	 *  Is AppsServer Active ?
	 *  @return true if active
	 */
	public static boolean isServerActive()
	{
		boolean contactAgain = s_server == null && s_serverTries == 0;
		boolean ok = CConnection.get().isAppsServerOK(contactAgain);
		if (ok)
		{
			s_serverTries = 0;
			return true;
		}
		if (s_serverTries > 1)	//	try twice
			return false;

		//	Try to connect
		CLogMgt.enable(false);
		try
		{
			s_serverTries++;
			log.config("try #" + s_serverTries);
			ok = CConnection.get().isAppsServerOK(true);
			if (ok)
				s_serverTries = 0;
		}
		catch (Exception ex)
		{
			ok = false;
			s_server = null;
		}
		CLogMgt.enable(true);
		//
		return ok;
	}   //  isServerActive
	
}	//	FactAcctReset
