/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2007 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software, you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY, without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program, if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 3600 Bridge Parkway #102, Redwood City, CA 94065, USA      *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.grid;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.logging.Level;

import javax.swing.JFrame;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import org.compiere.apps.AArchive;
import org.compiere.apps.AChat;
import org.compiere.apps.ADialog;
import org.compiere.apps.AEnv;
import org.compiere.apps.APanel;
import org.compiere.apps.ARequest;
import org.compiere.apps.AUserDefDialog;
import org.compiere.apps.AWindow;
import org.compiere.apps.AZoomAcross;
import org.compiere.apps.AppsAction;
import org.compiere.apps.Attachment;
import org.compiere.apps.EMailDialog;
import org.compiere.apps.ProcessCtl;
import org.compiere.apps.search.Find;
import org.compiere.framework.Query;
import org.compiere.grid.ed.VButton;
import org.compiere.grid.ed.VDocAction;
import org.compiere.model.DataStatusEvent;
import org.compiere.model.DataStatusListener;
import org.compiere.model.GridField;
import org.compiere.model.GridTab;
import org.compiere.model.MRole;
import org.compiere.model.MUser;
import org.compiere.print.AReport;
import org.compiere.process.DocAction;
import org.compiere.process.ProcessInfo;
import org.compiere.process.ProcessInfoUtil;
import org.compiere.swing.CPanel;
import org.compiere.util.ASyncProcess;
import org.compiere.util.CLogMgt;
import org.compiere.util.CLogger;
import org.compiere.util.Ctx;
import org.compiere.util.Env;
import org.compiere.util.Language;
import org.compiere.util.Msg;
import org.compiere.util.Util;

/**
 *	Sub Application Panel.
 *
 * 	@author 	Jirimuto
 * 	@version 	$Id: ASubPanel.java,v 1.2.2.1 2010/09/15 10:32:29 jrmt Exp $
 */
public final class ASubPanel extends CPanel
	implements DataStatusListener, ActionListener, ASyncProcess
{
	/**
	 * Constructs a new instance.
	 * Need to call initPanel for dynamic initialization
	 */
	public ASubPanel()
	{
		super();
		m_ctx = Env.getCtx();
		//
		try
		{
			jbInit();
		}
		catch (Exception e)
		{
			log.log(Level.SEVERE, "", e);
		}
		createMenu();
	}	//	APanel

	/**	Logger			*/
	private static CLogger log = CLogger.getCLogger(ASubPanel.class);
	
	/**
	 *	Dispose
	 */
	public void dispose()
	{
	//	log.config("");
		//  ignore changes
		m_disposing = true;
		//
		mainPanel = null;
		
		//  ToolBar
		if (toolBar != null)
			toolBar.removeAll();
		toolBar = null;
		//  Prepare GC
		this.removeAll();
	}	//	dispose

	/**
	 * The Layout.
	 */
	private BorderLayout mainLayout = new BorderLayout();
	private CPanel mainPanel = new CPanel();
	
	private CPanel northPanel = new CPanel();
	private JToolBar toolBar = new JToolBar();
	private FlowLayout centerLayout = new FlowLayout();
	private FlowLayout northLayout = new FlowLayout();

	/**
	 * Initializes the state of this instance.
	 * @throws Exception
	 */
	private void jbInit() throws Exception
	{
		this.setLocale(Language.getLoginLanguage().getLocale());
		this.setLayout(mainLayout);

		//	tabPanel
		mainLayout.setHgap(2);
		mainLayout.setVgap(2);
		this.add(mainPanel, BorderLayout.CENTER);
		mainPanel.setLayout(centerLayout);
		centerLayout.setAlignment(FlowLayout.LEFT);
		//	northPanel
		this.add(northPanel, BorderLayout.NORTH);
		northPanel.setLayout(northLayout);
		northLayout.setAlignment(FlowLayout.LEFT);
		northPanel.add(toolBar, null);
	}	//	jbInit

	private AppsAction 		aPrevious, aNext, aParent, aDetail, aFirst, aLast,
							aNew, aCopy, aDelete, aIgnore, aPrint,
							aRefresh, aAttachment, aChat, aMulti, aFind,
							aWorkflow, aZoomAcross, aRequest, aUserDef, aArchive;

	/** Save Button			*/
	public AppsAction		aSave;
	/** Private Lock Button	*/
	public AppsAction		aLock;
	//	Local (added to toolbar)
	private AppsAction	    aReport;
	
	/**************************************************************************
	 *	Create Menu and Toolbar and registers keyboard actions.
	 *  - started from constructor
	 */
	private void createMenu()
	{
		aReport = 	addAction("Report",			KeyStroke.getKeyStroke(KeyEvent.VK_F11, 0),	false);
		aPrint = 	addAction("Print",			KeyStroke.getKeyStroke(KeyEvent.VK_F12, 0),	false);

		aNew = 		addAction("New", 			KeyStroke.getKeyStroke(KeyEvent.VK_F2, 0), false);
		aCopy =		addAction("Copy", 			KeyStroke.getKeyStroke(KeyEvent.VK_F2, Event.SHIFT_MASK),	false);
		aDelete = 	addAction("Delete",			KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0),	false);
		aSave = 	addAction("Save",			KeyStroke.getKeyStroke(KeyEvent.VK_F4, 0),	false);
		aIgnore = 	addAction("Ignore",			KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),	false);
		aRefresh = 	addAction("Refresh",		KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0),	false);
		
		aFind = 	addAction("Find",			KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0), true);	//	toggle
		aAttachment = addAction("Attachment",	KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0),	true);		//	toggle
		aChat = addAction("Chat",				KeyStroke.getKeyStroke(KeyEvent.VK_F8, 0),	true);		//	toggle
		aMulti =	addAction("Multi",			KeyStroke.getKeyStroke(KeyEvent.VK_F9, 0),	true);		//	toggle
		//			Go
		aFirst =	addAction("First", 			KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, Event.ALT_MASK),	false);
		aPrevious = addAction("Previous", 		KeyStroke.getKeyStroke(KeyEvent.VK_UP, Event.ALT_MASK),	false);
		aNext = 	addAction("Next", 			KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, Event.ALT_MASK),	false);
		aLast =		addAction("Last",	 		KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, Event.ALT_MASK),	false);
		
		aZoomAcross = addAction("ZoomAcross",	KeyStroke.getKeyStroke(KeyEvent.VK_F9, Event.SHIFT_MASK),	false);
		aRequest =  addAction("Request",		KeyStroke.getKeyStroke(KeyEvent.VK_F11, Event.SHIFT_MASK),	false);
		aArchive =  addAction("Archive",		KeyStroke.getKeyStroke(KeyEvent.VK_F10, 0),	false);

		aParent =  addAction("Parent",			KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, Event.CTRL_MASK | Event.SHIFT_MASK),
												KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_UP, Event.CTRL_MASK ), false);
		aDetail =  addAction("Detail",			KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, Event.CTRL_MASK),	
												KeyStroke.getKeyStroke(KeyEvent.VK_PAGE_DOWN, Event.CTRL_MASK), false);
		
		/**
		 *	ToolBar
		 */
		toolBar.add(aIgnore.getButton());		//	ESC
		toolBar.addSeparator();
		toolBar.add(aNew.getButton());
		toolBar.add(aDelete.getButton());
		toolBar.add(aSave.getButton());
		toolBar.addSeparator();
		toolBar.add(aRefresh.getButton());      //  F5
		toolBar.add(aFind.getButton());
		toolBar.add(aAttachment.getButton());
		toolBar.add(aChat.getButton());
		toolBar.addSeparator();
		toolBar.add(aMulti.getButton());		//	F9
		toolBar.add(aArchive.getButton());		//	F10 is Windows Menu Key		
		toolBar.add(aReport.getButton());		//	F11
		toolBar.add(aPrint.getButton());		//	F12

		toolBar.addSeparator();
		toolBar.add(aParent.getButton());		//	Ctrl-PageUp
		toolBar.add(aDetail.getButton());		//	Ctrl-PageDown
		
		toolBar.addSeparator();
		toolBar.add(aFirst.getButton());
		toolBar.add(aPrevious.getButton());
		toolBar.add(aNext.getButton());
		toolBar.add(aLast.getButton());
		toolBar.addSeparator();
		if (m_isPersonalLock)
		{
			toolBar.add(aLock.getButton());
			toolBar.addSeparator();
		}
		toolBar.add(aZoomAcross.getButton());	//	Shift-F9
		if (aWorkflow != null)
			toolBar.add(aWorkflow.getButton());	//	Shift-F10
		toolBar.add(aRequest.getButton());		//	Shift-F11
		//
		if (CLogMgt.isLevelAll())
			Util.printActionInputMap(this);
	}	//	createMenu

	/**
	 *	Add (Toggle) Action to Toolbar
	 *  @param actionName action name
	 *  @param accelerator accelerator
	 *  @param toggle toggle button
	 *  @return AppsAction
	 */
	private AppsAction addAction (String actionName, KeyStroke accelerator, KeyStroke accelerator2, boolean toggle)
	{
		AppsAction action = new AppsAction(actionName, accelerator, toggle);
		action.setDelegate(this);
	//	AbstractButton b = action.getButton();
	//	String s = null;
	//	if (b != null)
	//		s = b.getToolTipText();
		
		//	Key Strokes
		if (accelerator != null)
		{
			getInputMap(WHEN_IN_FOCUSED_WINDOW).put(accelerator, actionName);
			getActionMap().put(actionName, action);
		}
		if (accelerator2 != null)
		{
			getInputMap(WHEN_IN_FOCUSED_WINDOW).put(accelerator2, actionName);
			getActionMap().put(actionName, action);
		}
		//
		return action;
	}	//	addAction
	
	/**
	 *	Add (Toggle) Action to Toolbar
	 *  @param actionName action name
	 *  @param accelerator accelerator
	 *  @param toggle toggle button
	 *  @return AppsAction
	 */
	private AppsAction addAction (String actionName, KeyStroke accelerator, boolean toggle)
	{
		AppsAction action = new AppsAction(actionName, accelerator, toggle);
		action.setDelegate(this);
	//	AbstractButton b = action.getButton();
	//	String s = null;
	//	if (b != null)
	//		s = b.getToolTipText();
		
		//	Key Strokes
		if (accelerator != null)
		{
			getInputMap(WHEN_IN_FOCUSED_WINDOW).put(accelerator, actionName);
			getActionMap().put(actionName, action);
		}
		//
		return action;
	}	//	addAction
	
	/**
	 *	Get Title of Window
	 *  @return String with Title
	 */
	public String getTitle()
	{
		return Env.getHeader(m_ctx, m_curWindowNo);
	}	//	getTitle

	/**	The Context										*/
	private Ctx				m_ctx;

	/** Current MTab                                    */
	private GridTab			m_curTab;
	/** Current GridController                          */
	private GridController  m_parentGC;
	/** Current GridController                          */
	private GridController  m_curGC;
	/** Current GridController                          */
	private APanel  		m_APanel;
	
	/** Current Window No                               */
	private int				m_curWindowNo;
	
	/** Dispose active                                  */
	private boolean         m_disposing = false;
	/** Save Error Message indicator                    */
	private boolean         m_errorDisplayed = false;
	/** Process Info                                    */
	private boolean         m_isLocked = false;
	/** Show Personal Lock								*/
	private boolean 		m_isPersonalLock = MRole.getDefault().isPersonalLock();
	/**	Last Modifier of Action Event					*/
	private int 			m_lastModifiers;

	
	/**************************************************************************
	 *	Dynamic Panel Initialization - either single window or workbench.
	 *  @param WindowNo - Window No     
	 *  @param gc - GridController
	 *  @return true if Panel is initialized successfully
	 */
	public boolean initPanel (int WindowNo, GridController parent, GridController gc, APanel aPanel)
	{
		
		m_curWindowNo = WindowNo;
		m_parentGC = parent;
		m_curGC = gc;
		m_curTab = gc.getMTab();
		m_APanel = aPanel;
		mainPanel.add(m_curGC);
		
		m_curGC.addDataStatusListener(this);
		gc.registerESCAction(aIgnore);      //  register Escape Key
		//  stateChanged (<->) triggered
		m_curTab.getTableModel().setChanged(false);
		
		return true;
	}	//	initPanel

	/**
	 * 	Get Current Window No
	 *	@return win no
	 */
	public int getWindowNo()
	{
		return m_curWindowNo;
	}	//	getWindowNo

	public void setEnabled(boolean newValue)
	{
		
		if( !newValue ) {
			
			aPrevious.setEnabled(newValue);
			aNext.setEnabled(newValue);
			aParent.setEnabled(newValue);
			aDetail.setEnabled(newValue);
			aFirst.setEnabled(newValue);
			aLast.setEnabled(newValue);
			aNew.setEnabled(newValue);
			aCopy.setEnabled(newValue);
			aDelete.setEnabled(newValue);
			aIgnore.setEnabled(newValue);
			aPrint.setEnabled(newValue);
			
			aRefresh.setEnabled(newValue);
			aAttachment.setEnabled(newValue);
			aChat.setEnabled(newValue);
			aMulti.setEnabled(newValue);
			aFind.setEnabled(newValue);
			aZoomAcross.setEnabled(newValue);
			aRequest.setEnabled(newValue);
			aArchive.setEnabled(newValue);
	
			aSave.setEnabled(newValue);
			aReport.setEnabled(newValue);
	
			m_curGC.setEnabled(newValue);
			
		} else {
			
			aParent.setEnabled(newValue);
			aDetail.setEnabled(newValue);
			aNew.setEnabled(newValue);
			aPrint.setEnabled(newValue);
			
			aRefresh.setEnabled(newValue);
			aMulti.setEnabled(newValue);
			aFind.setEnabled(newValue);
			aZoomAcross.setEnabled(newValue);
			aRequest.setEnabled(newValue);
			aArchive.setEnabled(newValue);
	
			aReport.setEnabled(newValue);
	
			m_curGC.setEnabled(newValue);
			
		}
	}
	
	/**************************************************************************
	 *	Data Status Listener (row change)			^ | v
	 *  @param e event
	 */
	public void dataStatusChanged (DataStatusEvent e)
	{
		if (m_disposing)
			return;
		log.info(e.getMessage());
		String dbInfo = e.getMessage();
		boolean findPressed = m_curTab.isQueryActive() || m_curTab.getOnlyCurrentDays() > 0;
		if (findPressed)
			dbInfo = "[ " + dbInfo + " ]";
		// m_APanel.statusBar.setStatusDB(dbInfo, e);

		//	Set Message / Info
		if (e.getAD_Message() != null || e.getInfo() != null)
		{
			StringBuffer sb = new StringBuffer();
			String msg = e.getMessage();
			if (msg != null && msg.length() > 0)
				sb.append(Msg.getMsg(m_ctx, e.getAD_Message()));
			String info = e.getInfo();
			if (info != null && info.length() > 0)
			{
				if (sb.length() > 0 && !sb.toString().trim().endsWith(":"))
					sb.append(": ");
				sb.append(info);
			}
			if (sb.length() > 0)
			{
				int pos = sb.indexOf("\n");
				if (pos != -1)  // replace CR/NL
					sb.replace(pos, pos+1, " - ");
				setStatusLine (sb.toString (), e.isError ());
			}
		}

		//  Confirm Error
		if (e.isError() && !e.isConfirmed())
		{
			ADialog.error(m_curWindowNo, this, e.getAD_Message(), e.getInfo());
			e.setConfirmed(true);   //  show just once - if MTable.setCurrentRow is involved the status event is re-issued
			m_errorDisplayed = true;
		}
		//  Confirm Warning
		else if (e.isWarning() && !e.isConfirmed())
		{
			ADialog.warn(m_curWindowNo, this, e.getAD_Message(), e.getInfo());
			e.setConfirmed(true);   //  show just once - if MTable.setCurrentRow is involved the status event is re-issued
		}

		//	update Navigation
		boolean firstRow = e.isFirstRow();
		aFirst.setEnabled(!firstRow);
		aPrevious.setEnabled(!firstRow);
		boolean lastRow = e.isLastRow();
		aNext.setEnabled(!lastRow);
		aLast.setEnabled(!lastRow);

		//	update Change
		boolean changed = e.isChanged() || e.isInserting();
		boolean readOnly = m_curTab.isReadOnly();
		boolean insertRecord = !readOnly;
		if (insertRecord)
			insertRecord = m_curTab.isInsertRecord();
		aNew.setEnabled(!changed && insertRecord);
		aCopy.setEnabled(!changed && insertRecord);
		aRefresh.setEnabled(!changed);
		aDelete.setEnabled(!changed && !readOnly);
		//
		if (readOnly && m_curTab.isAlwaysUpdateField())
			readOnly = false;
		aIgnore.setEnabled(changed && !readOnly);
		aSave.setEnabled(changed && !readOnly);
		//
		//	No Rows
		if (e.getTotalRows() == 0 && insertRecord)
		{
			aNew.setEnabled(true);
			aDelete.setEnabled(false);
		}

		//	Single-Multi
		aMulti.setPressed(!m_curGC.isSingleRow());

		//	History	or Query Active
		aFind.setPressed(findPressed);

		//	Transaction info
		String trxInfo = m_curTab.getTrxInfo();
		if (trxInfo != null)
			m_APanel.setInfo(trxInfo);

		//	Check Attachment
		boolean canHaveAttachment = m_curTab.canHaveAttachment();		//	not single _ID column
		//
		if (canHaveAttachment && e.isLoading() && m_curTab.getCurrentRow() > e.getLoadedRows())
			canHaveAttachment = false;
		if (canHaveAttachment && m_curTab.getRecord_ID() == -1)    //	No Key
			canHaveAttachment = false;
		if (canHaveAttachment)
		{
			aAttachment.setEnabled(true);
			aAttachment.setPressed(m_curTab.hasAttachment());
			aChat.setEnabled(true);
			aChat.setPressed(m_curTab.hasChat());
		}
		else
		{
			aAttachment.setEnabled(false);
			aChat.setEnabled(false);
		}
		//	Lock Indicator
		if (m_isPersonalLock)
			aLock.setPressed(m_curTab.isLocked());

	//	log.info("- fini", e.getMessage());
	}	//	dataStatusChanged

	/**
	 *	Set Status Line to text
	 *  @param text clear text
	 *  @param error error flag
	 */
	public void setStatusLine (String text, boolean error)
	{
		log.fine(text);
		m_APanel.setStatusLine(text, error);
	}	//	setStatusLine

	/**
	 *	Indicate Busy
	 *  @param busy busy
	 *  @param focus request focus
	 */
	private void setBusy (boolean busy, boolean focus)
	{
		m_isLocked = busy;
		//
		JFrame frame = Env.getFrame(this);
		if (frame == null)  //  during init
			return;
		if (frame instanceof AWindow)
			((AWindow)frame).setBusy(busy);
	//	String processing = Msg.getMsg(m_ctx, "Processing");
		if (busy)
		{
	//		setStatusLine(processing);
			this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			frame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
		else
		{
			this.setCursor(Cursor.getDefaultCursor());
			frame.setCursor(Cursor.getDefaultCursor());
			if (focus)
				m_curGC.requestFocusInWindow();
	//		if (statusBar.getStatusLine().equals(processing))
	//			statusBar.setStatusLine("");
		}
	}	//	set Busy
	
	/**************************************************************************
	 *	Action Listener
	 *  @param e event
	 */
	public void actionPerformed (ActionEvent e)
	{
		log.info(e.getActionCommand() + " - " + e.getModifiers());
		//	+ " - " + new Timestamp(e.getWhen()) + " " + isUILocked());
		if (m_disposing || isUILocked())
			return;
			
		m_lastModifiers = e.getModifiers();
		String cmd = e.getActionCommand();
		//	Do ScreenShot w/o busy
		if (cmd.equals("ScreenShot"))
		{
			AEnv.actionPerformed (e.getActionCommand(), m_curWindowNo, this);
			return;
		}

		//  Problem: doubleClick detection - can't disable button as clicking button may change button status
		setBusy (true, true);
		//  Command Buttons
		if (e.getSource() instanceof VButton)
		{
			actionButton((VButton)e.getSource());
			setBusy(false, true);
			return;
		}

		try
		{			
			//	File
			if (cmd.equals(aReport.getName()))
				cmd_report();
			else if (cmd.equals(aPrint.getName()))
				cmd_print();
			//	Edit
			else if (cmd.equals(aNew.getName()))
				cmd_new(false);
			else if (cmd.equals(aSave.getName()))
				cmd_save(true);
			else if (cmd.equals(aCopy.getName()))
				cmd_new(true);
			else if (cmd.equals(aDelete.getName()))
				cmd_delete();
			else if (cmd.equals(aIgnore.getName()))
				cmd_ignore();
			else if (cmd.equals(aRefresh.getName()))
				cmd_refresh();
			else if (cmd.equals(aFind.getName()))
				cmd_find();
			else if (m_isPersonalLock && cmd.equals(aLock.getName()))
				cmd_lock();
			//	View
			else if (cmd.equals(aAttachment.getName()))
				cmd_attachment();
			else if (cmd.equals(aChat.getName()))
				cmd_chat();
			else if (cmd.equals(aMulti.getName()))
				m_curGC.switchRowPresentation();
			//	Go
			else if (cmd.equals(aFirst.getName()))
			{	/*cmd_save(false);*/
				m_curGC.getTable().removeEditor();
				m_curTab.navigate(0);
				m_curGC.requestFocusInWindow();
			}
			else if (cmd.equals(aPrevious.getName()))
			{	/*cmd_save(false);*/
				m_curGC.getTable().removeEditor();
				m_curTab.navigateRelative(-1);
				m_curGC.requestFocusInWindow();
			}
			else if (cmd.equals(aNext.getName()))
			{	/*cmd_save(false); */
				m_curGC.getTable().removeEditor();
				m_curTab.navigateRelative(+1);
				m_curGC.requestFocusInWindow();
			}
			else if (cmd.equals(aLast.getName()))
			{	/*cmd_save(false);*/
				m_curGC.getTable().removeEditor();
				m_curTab.navigate(m_curTab.getRowCount()-1);
				m_curGC.requestFocusInWindow();
			}
			else if (cmd.equals(aZoomAcross.getName()))
				cmd_zoomAcross();
			else if (cmd.equals(aRequest.getName()))
				cmd_request();
			else if (cmd.equals(aParent.getName())){
				if( m_parentGC != null ){
					m_curGC.getTable().removeEditor();
					m_parentGC.requestFocusInWindow();
					setBusy(false, true);
					return;
				}
			}
			else if (cmd.equals(aDetail.getName())){
				if( m_parentGC != null ){
					m_parentGC.getTable().removeEditor();
					m_curGC.getTable().removeEditor();
					m_curTab.navigate(0);
					m_curGC.requestFocusInWindow();
					VTable vTable = m_curGC.getTable();
					if( vTable != null ){
						int col = 0;
						while (!vTable.isCellEditable(0, col) || !vTable.isColumnDisplayed(col) ){
							col++;
							if( col >= vTable.getColumnCount() ){
								col = 0;
								break;
							}
						}
						vTable.changeSelection(0, col, false, false);
					}
					setBusy(false, true);
					return;
				}
			}
			else if (cmd.equals(aArchive.getName()))
				cmd_archive();
			//	Tools
			else if (aWorkflow != null && cmd.equals(aWorkflow.getName()))
			{
				if (m_curTab.getRecord_ID() <= 0)
					;
				else if (m_curTab.getTabNo() == 0 )
					AEnv.startWorkflowProcess(m_curTab.getAD_Table_ID(), m_curTab.getRecord_ID());
				else
					AEnv.startWorkflowProcess(m_curTab.getAD_Table_ID(), m_curTab.getRecord_ID());
			}
			else if (aUserDef != null && cmd.equals(aUserDef.getName()))
				cmd_userDef();
			//  General Commands (Environment)
			else if (!AEnv.actionPerformed (e.getActionCommand(), m_curWindowNo, this))
				log.log(Level.SEVERE, "No action for: " + cmd);
		}
		catch (Exception ex)
		{
			log.log(Level.SEVERE, cmd, ex);
			String msg = ex.getMessage();
			if (msg == null || msg.length() == 0)
				msg = ex.toString();
			msg = Msg.parseTranslation(m_ctx, msg);
			ADialog.error(m_curWindowNo, this, "Error", msg);
		}
		//
		this.requestFocusInWindow();
		setBusy(false, true);
	}	//	actionPerformed

	/**
	 *  Create New Record
	 *  @param copy true if current record is to be copied
	 */
	private void cmd_new (boolean copy)
	{
		log.config("copy=" + copy);
		if (!m_curTab.isInsertRecord())
		{
			log.warning("Insert Record disabled for Tab");
			return;
		}
		cmd_save(false);
		m_curTab.dataNew (copy);
		m_curGC.dynamicDisplay(0);
	//	m_curTab.getTableModel().setChanged(false);
	}   //  cmd_new

	/**
	 *  Confirm & delete record
	 */
	private void cmd_delete()
	{
		if (m_curTab.isReadOnly())
			return;
		int keyID = m_curTab.getRecord_ID();
		if (ADialog.ask(m_curWindowNo, this, "DeleteRecord?"))
			if (m_curTab.dataDelete())
				m_curGC.rowChanged(false, keyID);
		m_curGC.dynamicDisplay(0);
	}   //  cmd_delete

	/**
	 *  If required ask if you want to save and save it
	 *  @param manualCmd true if invoked manually (i.e. force)
	 *  @return true if saved
	 */
	private boolean cmd_save (boolean manualCmd)
	{
		log.config("Manual=" + manualCmd);
		m_errorDisplayed = false;
		m_curGC.stopEditor(true);

		if (m_curTab.getCommitWarning().length() > 0 && m_curTab.needSave(true, false))
			if (!ADialog.ask(m_curWindowNo, this, "SaveChanges?", m_curTab.getCommitWarning()))
				return false;

		//  manually initiated
		boolean retValue = m_curTab.dataSave(manualCmd);
		//   if there is no previous error
		if (manualCmd && !retValue && !m_errorDisplayed)
		{
			ADialog.error(m_curWindowNo, this, "SaveIgnored");
			setStatusLine(Msg.getMsg(m_ctx, "SaveIgnored"), true);
		}
		m_curGC.rowChanged(true, m_curTab.getRecord_ID());
		if (manualCmd)
			m_curGC.dynamicDisplay(0);
		return retValue;
	}   //  cmd_save

	/**
	 *  Ignore
	 */
	private void cmd_ignore()
	{
		m_curGC.stopEditor(false);
		m_curTab.dataIgnore();
		m_curGC.dynamicDisplay(0);
	}   //  cmd_ignore

	/**
	 *  Refresh
	 */
	private void cmd_refresh()
	{
		cmd_save(false);
		m_curTab.dataRefreshAll();
		m_curGC.dynamicDisplay(0);
	}   //  cmd_refresh

	/**
	 *	Print standard Report
	 */
	private void cmd_report ()
	{
		log.info("");
		if (!MRole.getDefault().isCanReport(m_curTab.getAD_Table_ID()))
		{
			ADialog.error(m_curWindowNo, this, "AccessCannotReport");
			return;
		}
		
		cmd_save(false);

		//	Query
		Query query = new Query(m_curTab.getTableName());
		//	Link for detail records
		String queryColumn = m_curTab.getLinkColumnName();
		//	Current row otherwise
		if (queryColumn.length() == 0)
			queryColumn = m_curTab.getKeyColumnName();
		//	Find display
		String infoName = null;
		String infoDisplay = null;
		for (int i = 0; i < m_curTab.getFieldCount(); i++)
		{
			GridField field = m_curTab.getField(i);
			if (field.isKey())
				infoName = field.getHeader();
			if ((field.getColumnName().equals("Name") || field.getColumnName().equals("DocumentNo") )
				&& field.getValue() != null)
				infoDisplay = field.getValue().toString();
			if (infoName != null && infoDisplay != null)
				break;
		}
		if (queryColumn.length() != 0)
		{
			if (queryColumn.endsWith("_ID"))
				query.addRestriction(queryColumn, Query.EQUAL,
					new Integer(m_ctx.getContextAsInt(m_curWindowNo, queryColumn)),
					infoName, infoDisplay);
			else
				query.addRestriction(queryColumn, Query.EQUAL,
					m_ctx.getContext(m_curWindowNo, queryColumn),
					infoName, infoDisplay);
		}

		new AReport (m_curTab.getAD_Table_ID(), aReport.getButton(), query);
	}	//	cmd_report

	
	/**
	 * 	Zoom Across Menu
	 */
	private void cmd_zoomAcross()
	{
		int record_ID = m_curTab.getRecord_ID();
		log.info("ID=" + record_ID);
		if (record_ID <= 0)
			return;

		//	Query
		Query query = new Query();
		//	Current row
		String link = m_curTab.getKeyColumnName();
		//	Link for detail records
		if (link.length() == 0)
			link = m_curTab.getLinkColumnName();
		if (link.length() != 0)
		{
			if (link.endsWith("_ID"))
				query.addRestriction(link, Query.EQUAL,
					new Integer(m_ctx.getContextAsInt(m_curWindowNo, link)));
			else
				query.addRestriction(link, Query.EQUAL,
					m_ctx.getContext( m_curWindowNo, link));
		}
		new AZoomAcross (aZoomAcross.getButton(), 
			m_curTab.getTableName(), query, m_curTab.getAD_Window_ID());
	}	//	cmd_zoomAcross
	
	/**
	 * 	Open/View Request
	 */
	private void cmd_request()
	{
		int record_ID = m_curTab.getRecord_ID();
		log.info("ID=" + record_ID);
		if (record_ID <= 0)
			return;

		int AD_Table_ID = m_curTab.getAD_Table_ID();
		int C_BPartner_ID = 0;
		Object BPartner_ID = m_curTab.getValue("C_BPartner_ID");
		if (BPartner_ID != null)
			C_BPartner_ID = ((Integer)BPartner_ID).intValue();
		new ARequest (aRequest.getButton(), AD_Table_ID, record_ID, C_BPartner_ID);
	}	//	cmd_request

	/**
	 * 	Open/View Archive
	 */
	private void cmd_archive()
	{
		int record_ID = m_curTab.getRecord_ID();
		log.info("ID=" + record_ID);
		if (record_ID <= 0)
			return;

		int AD_Table_ID = m_curTab.getAD_Table_ID();
		new AArchive (aArchive.getButton(), AD_Table_ID, record_ID);
	}	//	cmd_archive
	
	/**
	 *	Print specific Report - or start default Report
	 */
	private void cmd_print()
	{
		//	Get process defined for this tab
		int AD_Process_ID = m_curTab.getAD_Process_ID();
		log.info("ID=" + AD_Process_ID);

		//	No report defined
		if (AD_Process_ID == 0)
		{
			cmd_report();
			return;
		}

		cmd_save(false);
		//
		int table_ID = m_curTab.getAD_Table_ID();
		int record_ID = m_curTab.getRecord_ID();
		ProcessInfo pi = new ProcessInfo (getTitle(), AD_Process_ID, table_ID, record_ID);
		pi.setAD_User_ID (m_ctx.getAD_User_ID());
		pi.setAD_Client_ID (m_ctx.getAD_Client_ID());

		ProcessCtl.process(this, m_curWindowNo, pi, null); //  calls lockUI, unlockUI
	}   //  cmd_print

	/**
	 *	Find - Set Query
	 */
	private void cmd_find()
	{
		if (m_curTab == null)
			return;
		cmd_save(false);
		//	Gets Fields from AD_Field_v
		GridField[] findFields = GridField.createFields(m_ctx, m_curWindowNo, 0, m_curTab.getAD_Tab_ID(), 0);
		Find find = new Find (Env.getFrame(this), m_curWindowNo, 
			m_curTab.getName(), m_curTab.getAD_Tab_ID(),
			m_curTab.getAD_Table_ID(), m_curTab.getTableName(), 
			m_curTab.getWhereClause(), findFields, 0);
		//	Simple/Advanced Query
		Query query = find.getQuery();
		//	History
		int onlyCurrentDays = find.getCurrentDays();
		boolean created = find.getIsCreated();
		find.dispose();
		find = null;

		//	Confirmed query
		if (query != null && query.isActive())
		{
			log.config(query.toString()); 
			m_curTab.setQuery(query);
			m_curGC.query(0, 0, created);   //  autoSize
		}
		else
		{
			int maxRows = MRole.getDefault().getMaxQueryRecords();
			log.config("OnlyCurrentDays=" + onlyCurrentDays
				+ ", MaxRows=" + maxRows);
			m_curTab.setQuery(null);	//	reset previous queries
			m_curGC.query(onlyCurrentDays, maxRows, created);   //  autoSize
		}
		boolean findPressed = m_curTab.isQueryActive() || m_curTab.getOnlyCurrentDays() > 0;
		aFind.setPressed(findPressed);
	}	//	cmd_find

	/**
	 *	Attachment
	 */
	private void cmd_attachment()
	{
		int record_ID = m_curTab.getRecord_ID();
		log.info("Record_ID=" + record_ID);
		if (record_ID == -1)	//	No Key
		{
			aAttachment.setEnabled(false);
			return;
		}

	//	Attachment va = 
		new Attachment (Env.getFrame(this), m_curWindowNo,
			m_curTab.getAD_AttachmentID(), m_curTab.getAD_Table_ID(), record_ID, null);
		//
		m_curTab.loadAttachments();				//	reload
		aAttachment.setPressed(m_curTab.hasAttachment());
	}	//	attachment

	/**
	 *	Chat
	 */
	private void cmd_chat()
	{
		int record_ID = m_curTab.getRecord_ID();
		log.info("Record_ID=" + record_ID);
		if (record_ID == -1)	//	No Key
		{
			aChat.setEnabled(false);
			return;
		}
		//	Find display
		String infoName = null;
		String infoDisplay = null;
		for (int i = 0; i < m_curTab.getFieldCount(); i++)
		{
			GridField field = m_curTab.getField(i);
			if (field.isKey())
				infoName = field.getHeader();
			if ((field.getColumnName().equals("Name") || field.getColumnName().equals("DocumentNo") )
				&& field.getValue() != null)
				infoDisplay = field.getValue().toString();
			if (infoName != null && infoDisplay != null)
				break;
		}
		String description = infoName + ": " + infoDisplay;
		//
	//	AChat va = 
		new AChat (Env.getFrame(this), m_curWindowNo,
			m_curTab.getCM_ChatID(), m_curTab.getAD_Table_ID(), record_ID, 
			description, null);
		//
		m_curTab.loadChats();				//	reload
		aChat.setPressed(m_curTab.hasChat());
	}	//	chat

	/**
	 *	Lock
	 */
	private void cmd_lock()
	{
		log.info("Modifiers=" + m_lastModifiers);
		if (!m_isPersonalLock)
			return;
		int record_ID = m_curTab.getRecord_ID();
		if (record_ID == -1)	//	No Key
			return;
		//	Control Pressed
		if ((m_lastModifiers & InputEvent.CTRL_MASK) != 0)
		{
			new RecordAccessDialog(Env.getFrame(this), m_curTab.getAD_Table_ID(), record_ID);
		}
		else
		{
			m_curTab.lock (Env.getCtx(), record_ID, aLock.getButton().isSelected());
			m_curTab.loadAttachments();			//	reload
		}
		aLock.setPressed(m_curTab.isLocked());
	}	//	lock

	/**
	 * 	User Defined Window Customization
	 */
	private void cmd_userDef()
	{
		int AD_Window_ID = m_curTab.getAD_Window_ID();
		AUserDefDialog ud = new AUserDefDialog(Env.getFrame(this), AD_Window_ID, m_curWindowNo);
		int AD_Tab_ID = m_curTab.getAD_Tab_ID();
		Dimension size = getSize();
		ud.save(AD_Tab_ID, m_curGC.getTable(), size);
	}	//	cmdWinSize
	
	
	/**************************************************************************
	 *	Start Button Process
	 *  @param vButton button
	 */
	private void actionButton (VButton vButton)
	{
		log.info(vButton.toString());

		boolean startWOasking = false;
		boolean batch = false;
		String columnName = vButton.getColumnName();

		//  Zoom Button
		if (columnName.equals("Record_ID"))
		{
			int AD_Table_ID = m_ctx.getContextAsInt(m_curWindowNo, "AD_Table_ID");
			int Record_ID = m_ctx.getContextAsInt(m_curWindowNo, "Record_ID");
			AEnv.zoom(AD_Table_ID, Record_ID);
			return;
		}   //  Zoom

		//  save first	---------------
		if (m_curTab.needSave(true, false))
			if (!cmd_save(true))
				return;
		//
		int table_ID = m_curTab.getAD_Table_ID();
		//	Record_ID
		int record_ID = m_curTab.getRecord_ID();
		//	Record_ID - Language Handling
		if (record_ID == -1 && m_curTab.getKeyColumnName().equals("AD_Language"))
			record_ID = m_ctx.getContextAsInt(m_curWindowNo, "AD_Language_ID");
		//	Record_ID - Change Log ID
		if (record_ID == -1 
			&& (vButton.getProcess_ID() == 306 || vButton.getProcess_ID() == 307))
		{
			Integer id = (Integer)m_curTab.getValue("AD_ChangeLog_ID");
			record_ID = id.intValue();
		}
		//	Record_ID - EntityType
		if (record_ID == -1 && m_curTab.getKeyColumnName().equals("EntityType"))
		{
			Integer id = (Integer)m_curTab.getValue("AD_EntityType_ID");
			record_ID = id.intValue();
		}
		//	Ensure it's saved
		if (record_ID == -1 && m_curTab.getKeyColumnName().endsWith("_ID"))
		{
			ADialog.error(m_curWindowNo, this, "SaveErrorRowNotFound");
			return;
		}

		//	Pop up Payment Rules
		if (columnName.equals("PaymentRule"))
		{
			VPayment vp = new VPayment(m_curWindowNo, m_curTab, vButton);
			if (vp.isInitOK())		//	may not be allowed
				vp.setVisible(true);
			vp.dispose();
			if (vp.needSave())
			{
				cmd_save(false);
				cmd_refresh();
			}
		}	//	PaymentRule

		//	Pop up Document Action (Workflow)
		else if (columnName.equals("DocAction"))
		{
			VDocAction vda = new VDocAction(m_curWindowNo, m_curTab, vButton, record_ID);
			//	Something to select from?
			if (vda.getNumberOfOptions() == 0)
			{
				vda.dispose ();
				log.info("DocAction - No Options");
				return;
			}
			else
			{
				vda.setVisible(true);
				if (!vda.isStartProcess())
					return;
				batch = vda.isBatch();
				startWOasking = true;
				vda.dispose();
			}
		}	//	DocAction

		//  Pop up Create From
		else if (columnName.equals("CreateFrom"))
		{
			//  m_curWindowNo
			VCreateFrom vcf = VCreateFrom.create (m_curTab);
			if (vcf != null)
			{
				if (vcf.isInitOK())
				{
					vcf.setVisible(true);
					vcf.dispose();
					m_curTab.dataRefresh();
				}
				else
					vcf.dispose();
				return;
			}
			//	else may start process
		}	//	CreateFrom

		//  Posting -----
		else if (columnName.equals("Posted") && MRole.getDefault().isShowAcct())
		{
			//  Check Doc Status
			String processed = m_ctx.getContext( m_curWindowNo, "Processed");
			if (!processed.equals("Y"))
			{
				String docStatus = m_ctx.getContext( m_curWindowNo, "DocStatus");
				if (DocAction.STATUS_Completed.equals(docStatus)
					|| DocAction.STATUS_Closed.equals(docStatus)
					|| DocAction.STATUS_Reversed.equals(docStatus)
					|| DocAction.STATUS_Voided.equals(docStatus))
					;
				else
				{
					ADialog.error(m_curWindowNo, this, "PostDocNotComplete");
					return;
				}
			}

			//  Check Post Status
			Object ps = m_curTab.getValue("Posted");
			if (ps != null && ps.equals("Y"))
			{
				new org.compiere.acct.AcctViewer (m_ctx.getContextAsInt(m_curWindowNo, "AD_Client_ID"),
					m_curTab.getAD_Table_ID(), m_curTab.getRecord_ID());
			}
			else
			{
				if (ADialog.ask(m_curWindowNo, this, "PostImmediate?"))
				{
					boolean force = ps != null && !ps.equals ("N");		//	force when problems
					String error = AEnv.postImmediate ( Env.getCtx(), m_curWindowNo, m_ctx.getAD_Client_ID(),
						m_curTab.getAD_Table_ID(), m_curTab.getRecord_ID(), force);
					m_curTab.dataRefresh();
					if (error != null)
						ADialog.error(m_curWindowNo, this, "PostingError-N", error);
				}
			}
			return;
		}   //  Posted
		
		//	Send Email -----
		else if (columnName.equals("SendNewEMail"))
		{
			int AD_Process_ID = vButton.getProcess_ID();
			if (AD_Process_ID != 0)
			{
			}
			//	Mail Defaults
			String title = getTitle();
			String to = null;
			Object oo = m_curTab.getValue("AD_User_ID");
			if (oo instanceof Integer)
			{
				MUser user = new MUser(Env.getCtx (), ((Integer)oo).intValue (), null);
				to = user.getEMail();
			}
			if (to == null)
				to = (String)m_curTab.getValue("EMail");
			String subject = (String)m_curTab.getValue("Name");;
			String message = "";
			new EMailDialog (Env.getFrame(this), title, 
				MUser.get(Env.getCtx()), 
				to,	subject, message, 
				null);
			return;
		}

		/**
		 *  Start Process ----
		 */

		log.config("Process_ID=" + vButton.getProcess_ID() + ", Record_ID=" + record_ID);
		if (vButton.getProcess_ID() == 0)
			return;
		//	Save item changed
		if (m_curTab.needSave(true, false))
			if (!cmd_save(true))
				return;

		//	Ask user to start process, if Description and Help is not empty
		if (!startWOasking && !(vButton.getDescription().equals("") && vButton.getHelp().equals("")))
			if (!ADialog.ask(m_curWindowNo, this, "StartProcess?", 
				//	"<b><i>" + vButton.getText() + "</i></b><br>" +
				vButton.getDescription() + "\n" + vButton.getHelp()))
				return;
		//
		String title = vButton.getDescription();
		if (title == null || title.length() == 0)
			title = vButton.getName();
		ProcessInfo pi = new ProcessInfo (title, vButton.getProcess_ID(), table_ID, record_ID);
		pi.setAD_User_ID (m_ctx.getAD_User_ID());
		pi.setAD_Client_ID (m_ctx.getAD_Client_ID());
		pi.setIsBatch(batch);

	//	Trx trx = Trx.get(Trx.createTrxName("AppsPanel"), true);
		ProcessCtl.process(this, m_curWindowNo, pi, null); //  calls lockUI, unlockUI
	}	//	actionButton

	
	/**************************************************************************
	 *  Lock User Interface.
	 *  Called from the Worker before processing
	 *  @param pi process info
	 */
	public void lockUI (ProcessInfo pi)
	{
	//	log.fine("" + pi);
		setBusy(true, false);
	}   //  lockUI

	/**
	 *  Unlock User Interface.
	 *  Called from the Worker when processing is done
	 *  @param pi of execute ASync call
	 */
	public void unlockUI (ProcessInfo pi)
	{
	//	log.fine("" + pi);
		boolean notPrint = pi != null 
			&& pi.getAD_Process_ID() != m_curTab.getAD_Process_ID();
		//
		setBusy(false, notPrint);
		//  Process Result
		if (notPrint)		//	refresh if not print 
		{
			//	Refresh data
			m_curTab.dataRefresh();
			//	Timeout
			if (pi.isTimeout())		//	set temporarily to R/O
				m_ctx.setContext(m_curWindowNo, "Processed", "Y");
			m_curGC.dynamicDisplay(0);
			//	Update Status Line
			setStatusLine(pi.getSummary(), pi.isError());
			//	Get Log Info
			ProcessInfoUtil.setLogFromDB(pi);
			String logInfo = pi.getLogInfo();
			if (logInfo.length() > 0)
				ADialog.info(m_curWindowNo, this, Env.getHeader(m_ctx, m_curWindowNo),
					pi.getTitle(), logInfo);	//	 clear text
		}
	}   //  unlockUI

	/**
	 *  Is the UI locked (Internal method)
	 *  @return true, if UI is locked
	 */
	public boolean isUILocked()
	{
		return m_isLocked;
	}   //  isLoacked

	/**
	 *  Method to be executed async.
	 *  Called from the ASyncProcess worker
	 *  @param pi process info
	 */
	public void executeASync (ProcessInfo pi)
	{
		log.config("-");
	}   //  executeASync

	/**
	 * 	Get Current Tab
	 *	@return current tab
	 */
	protected GridTab getCurrentTab()
	{
		return m_curTab;
	}	//	getCurrentTab
	
	/**
	 * 	Remove Table Editor
	 */
	public void removeEditor()
	{
		m_curGC.getTable().removeEditor();
	}	//	removeEditor
	
	/**
	 *  String representation
	 *  @return String representation
	 */
	public String toString()
	{
		String s = "APanel[curWindowNo=" + m_curWindowNo;
		s += "]";
		return s;
	}   //  toString

	public void executeRefresh() {
		cmd_refresh();
	}

}	//	APanel
