/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2007 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software, you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY, without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program, if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 3600 Bridge Parkway #102, Redwood City, CA 94065, USA      *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/
package org.compiere.grid;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;

import javax.swing.InputMap;
import javax.swing.KeyStroke;
import javax.swing.table.*;
import org.compiere.grid.ed.*;
import org.compiere.model.*;
import org.compiere.swing.*;
import org.compiere.util.*;

/**
 *  Table Grid based on CTable.
 * 	Used in GridController
 *
 * 	@author 	Jorg Janke
 * 	@version 	$Id: VTable.java,v 1.3 2010/06/14 02:42:37 jrmt Exp $
 */
public final class VTable extends CTable 
	implements PropertyChangeListener
{
	/**
	 *	Default Constructor
	 *	@param withRowHeader with numbers in Front
	 */
	public VTable(boolean withRowHeader)
	{
		super(withRowHeader);
		setAutoscrolls(true);
		
		InputMap im = getInputMap(CTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
		KeyStroke tab   = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0);
		KeyStroke enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
		KeyStroke altright   = KeyStroke.getKeyStroke(KeyEvent.VK_RIGHT, InputEvent.CTRL_MASK);
		im.put(enter, im.get(tab));
		im.put(altright, im.get(tab));
		
		KeyStroke shifttab   = KeyStroke.getKeyStroke(KeyEvent.VK_TAB, InputEvent.SHIFT_MASK);
		KeyStroke altleft = KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, InputEvent.CTRL_MASK);
		im.put(altleft, im.get(shifttab));
		
		addMouseListener(new MouseAdapter()
		{
			// ensures that right clicks start select, start editing, and, if the
			// right mouse button is pressed, show the popup menu
			public void mouseClicked(MouseEvent e) {
				Point p = e.getPoint();
				boolean canEdit = !isEditing() || getCellEditor().stopCellEditing();
		
				if (canEdit && editCellAt(rowAtPoint(p), columnAtPoint(p)))
				{
					getEditorComponent().dispatchEvent(e);
				}
			}
		});
	}	//	VTable

	/**	Logger			*/
	private static CLogger log = CLogger.getCLogger(VTable.class);
	
	/**
	 *  Property Change Listener for CurrentRow.
	 *  - Selects the current row if not already selected
	 *  - Required when navigating via Buttons
	 *  @param evt event
	 */
	public void propertyChange(PropertyChangeEvent evt)
	{
	//	log.config(evt);
		if (evt.getPropertyName().equals(GridTab.PROPERTY))
		{
			int row = ((Integer)evt.getNewValue()).intValue();
			int selRow = getSelectedRow();
			if (row == selRow)
				return;
			log.config(GridTab.PROPERTY + "=" + row + " from " + selRow);
			setRowSelectionInterval(row,row);
		    Rectangle cellRect = getCellRect(row, 0, false);
		    if (cellRect != null)
		    	scrollRectToVisible(cellRect);
			log.config(GridTab.PROPERTY + "=" + row + " from " + selRow);
		}
	}   //  propertyChange

	/**
	 *	Get ColorCode for Row.
	 *  <pre>
	 *	If numerical value in compare column is
	 *		negative = -1,
	 *      positive = 1,
	 *      otherwise = 0
	 *  </pre>
	 *  @param row row
	 *  @return color code
	 */
	public int getColorCode (int row)
	{
		return ((GridTable)getModel()).getColorCode(row);
	}   //  getColorCode

	/**
	 *  Sort Table
	 *  @param modelColumnIndex model column sort index
	 */
	protected void sort (int modelColumnIndex)
	{
		int rows = getRowCount();
		if (rows == 0)
			return;
		//
		TableModel model = getModel();
		if (!(model instanceof GridTable))
		{
			super.sort(modelColumnIndex);
			return;
		}

		//  other sort column
		if (modelColumnIndex != p_lastSortIndex)
			p_asc = true;
		else
			p_asc = !p_asc;

		p_lastSortIndex = modelColumnIndex;
		//
		log.config("#" + modelColumnIndex
			+ " - rows=" + rows + ", asc=" + p_asc);

		((GridTable)model).sort(modelColumnIndex, p_asc);
		//  table model fires "Sorted" DataStatus event which causes MTab to position to row 0
	}   //  sort

	/**
	 *  Transfer focus explicitly to editor due to editors with multiple components
	 *
	 *  @param row row
	 *  @param column column
	 *  @param e event
	 *  @return true if cell is editing
	 */
	public boolean editCellAt (int row, int column, java.util.EventObject e)
	{
		if (!super.editCellAt(row, column, e))
			return false;
	//	log.fine( "VTable.editCellAt", "r=" + row + ", c=" + column);

		Object ed = getCellEditor();
		if (ed instanceof VEditor)
			((Component)ed).requestFocus();
		else if (ed instanceof VCellEditor)
		{
			ed = ((VCellEditor)ed).getEditor();
			((Component)ed).requestFocus();
			if (ed instanceof VString)
			{
				((VString)ed).selectAll();
			}
			else if (ed instanceof VNumber)
			{
				((VNumber)ed).selectAll();
			}
		}
		return true;
	}   //  editCellAt

	/**
	 *  toString
	 *  @return String representation
	 */
	public String toString()
	{
		return new StringBuffer("VTable[")
			.append(getModel()).append("]").toString();
	}   //  toString

	// Jirimuto Added for editing column on focusgain -- 2010/04/25
	public void changeSelection(final int row, final int column, boolean toggle, boolean extend)  {
		
	    if (this.isRequestFocusEnabled()) {
            this.requestFocus();
	    }
	    
	    super.changeSelection(row, column, toggle, extend);
	    
	    if( isCellEditable(row, column) && isColumnDisplayed( column ) ){
		    if (editCellAt(row, column)) {
		        getEditorComponent().requestFocusInWindow();
		    }
	    } else {
	    	TableCellRenderer renderer = this.getCellRenderer(row, column);
		    	if( renderer instanceof	TableCellNone ){
		    	int r = row;
		    	int c = column + 1;
		    	if( c >= getColumnCount() ){
		    		c = 0;
		    		r++;
		    		if( r >= getRowCount())
		    			r = 0;
		    	}
		    	this.changeSelection(r, c, toggle, extend);
	    	}
	    }
	}
	
	public boolean isColumnDisplayed(int column){

		String columnname = this.getColumnName(column);
		TableColumn tc = this.getColumn(columnname);
		
		TableCellEditor cellEditor = tc.getCellEditor();
		if( cellEditor instanceof TableCellNone )
			return false;
		
		return true;
	}
	// Jirimuto Added for CellRendererCacheClear -- 2010/06/11
	public void CellRendererCacheClear(){
		
		for(int i=0; i<getColumnCount(); i++) {
			TableCellRenderer renderer = this.getCellRenderer(0, i);
			if( renderer instanceof VCellRenderer ){
				VCellRenderer vrenderer = (VCellRenderer)renderer;
				vrenderer.cacheClear();
			}
		}
	}
	
}	//	VTable
